﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
//#define NN_USB_ENABLE_TRACE 1

#include "usb_DsProtocol.h"
#include "usb_DsController.h"
#include "usb_DsDummyController.h"
#include <nn/nn_SystemThreadDefinition.h>
#include <nn/settings/factory/settings_SerialNumber.h>
#include <nn/settings/fwdbg/settings_SettingsGetterApi.h>
#include "usb_DsChargingDevice.h"

using namespace nn::dd;

namespace nn {
namespace usb {
namespace ds {

//  Anonyumous namespace
namespace {

UsbStringDescriptor m_UsbStringDescriptorLanguage       = {4,   UsbDescriptorType_String, {0x0409}};
UsbStringDescriptor m_UsbStringDescriptorManufacturer   = {18,  UsbDescriptorType_String, {'N', 'i', 'n', 't', 'e', 'n', 'd', 'o'}};
UsbStringDescriptor m_UsbStringDescriptorProduct        = {2,   UsbDescriptorType_String, {}}; // fetch product string from settings

UsbDeviceDescriptor m_UsbDeviceDescriptorFullSpeed =
{
    UsbDescriptorSize_Device,                   // bLength
    UsbDescriptorType_Device,                   // bDescriptorType
    0x0110,                                     // bcdUSB 1.1
    0x00,                                       // bDeviceClass
    0x00,                                       // bDeviceSubClass
    0x00,                                       // bDeviceProtocol
    0x40,                                       // bMaxPacketSize0
    0x057e,                                     // idVendor
    0x2000,                                     // idProduct
    0x0100,                                     // bcdDevice
    0x01,                                       // iManufacturer
    0x02,                                       // iProduct
    0x00,                                       // iSerialNumber
    0x01                                        // bNumConfigurations
};


UsbDeviceDescriptor m_UsbDeviceDescriptorHighSpeed =
{
    UsbDescriptorSize_Device,                   // bLength
    UsbDescriptorType_Device,                   // bDescriptorType
    0x0200,                                     // bcdUSB 2.0
    0x00,                                       // bDeviceClass
    0x00,                                       // bDeviceSubClass
    0x00,                                       // bDeviceProtocol
    0x40,                                       // bMaxPacketSize0
    0x057e,                                     // idVendor
    0x2000,                                     // idProduct
    0x0100,                                     // bcdDevice
    0x01,                                       // iManufacturer
    0x02,                                       // iProduct
    0x00,                                       // iSerialNumber
    0x01                                        // bNumConfigurations
};


UsbDeviceDescriptor m_UsbDeviceDescriptorSuperSpeed =
{
    UsbDescriptorSize_Device,                   // bLength
    UsbDescriptorType_Device,                   // bDescriptorType
    0x0300,                                     // bcdUSB 3.0
    0x00,                                       // bDeviceClass
    0x00,                                       // bDeviceSubClass
    0x00,                                       // bDeviceProtocol
    0x09,                                       // bMaxPacketSize0, SS 512 or 2^9
    0x057e,                                     // idVendor
    0x2000,                                     // idProduct
    0x0100,                                     // bcdDevice
    0x01,                                       // iManufacturer
    0x02,                                       // iProduct
    0x00,                                       // iSerialNumber
    0x01                                        // bNumConfigurations
};


uint8_t m_BinaryObjectStore[] =
{
    0x05,                                       // bLength
    UsbDescriptorType_Bos,                      // bDescriptorType
    0x16,0x00,                                  // Length of this descriptor and all sub descriptors
    0x02,                                       // Number of device capability descriptors

    // USB 2.0 extension
    0x07,                                       // bLength
    UsbDescriptorType_DeviceCapability,         // bDescriptorType
    0x02,                                       // USB 2.0 extension capability type
    0x0e,0x0f,0x00,0x00,                        // Supported device level features: LPM support

    // SuperSpeed device capability
    0x0A,                                       // bLength
    UsbDescriptorType_DeviceCapability,         // bDescriptorType
    0x03,                                       // SuperSpeed device capability type
    0x00,                                       // Supported device level features
    0x0e,0x00,                                  // Speeds supported by the device : SS, HS, FS
    0x01,                                       // Functionality support
    0x0a,                                       // U1 Device Exit latency
    0x20,0x00                                   // U2 Device Exit latency
};


UsbInterfaceDescriptor m_UsbInterfaceDescriptor =
{
    UsbDescriptorSize_Interface,                // bLength
    UsbDescriptorType_Interface,                // bDescriptorType
    0,                                          // bInterfaceNumber
    0,                                          // bAlternateSetting
    1,                                          // bNumEndpoints
    UsbClass_Hid,                               // bInterfaceClass
    0,                                          // bInterfaceSubClass
    0,                                          // bInterfaceProtocol
    2,                                          // iInterface
};


UsbEndpointDescriptor m_UsbEndpointDescriptor =
{
    UsbDescriptorSize_Endpoint,                 // bLength;
    UsbDescriptorType_Endpoint,                 // bDescriptorType;
    0x81,                                       // bEndpointAddress;
    UsbEndpointAttributeMask_XferTypeInt,       // bmAttributes;
    1,                                          // wMaxPacketSize;
    10,                                         // bInterval;
};


UsbEndpointCompanionDescriptor m_UsbEndpointCompanionDescriptor =
{
    0x06,                                       // bLength
    nn::usb::UsbDescriptorType_EndpointCompanion, // bDescriptorType
    0,                                          // bMaxBurst
    0,                                          // bmAttributes
    1                                           // wBytesPerInterval
};


uint8_t m_HidReportDescriptor[] =
{
    0x05, 0x06, 0x09, 0x20, 0xA1, 0x01, 0x09, 0x20,
    0xA1, 0x00, 0x05, 0x06, 0x09, 0x20, 0x15, 0x81,
    0x25, 0x7F, 0x75, 0x08, 0x95, 0x01, 0x81, 0x06,
    0xC0, 0xC0
};


UsbHidDescriptor m_UsbHidDescriptor =
{
    sizeof(UsbHidDescriptor),               // uint8_t bLength;
    UsbDescriptorType_Hid,                  // uint8_t bDescriptorType;
    0x0111,                                 // uint16_t bcdHid;
    0,                                      // uint8_t bCountryCode;
    1,                                      // uint8_t bNumDescriptors;

    {
        {
            UsbDescriptorType_Report,       // uint8_t bDescriptorType;
            sizeof(m_HidReportDescriptor),  // uint16_t wDescriptorLength;
        }
    }
};


} // Anonymous namespace

#undef  NN_USB_TRACE_CLASS_NAME
#define NN_USB_TRACE_CLASS_NAME "DsDefaultInterface"

//////////////////////////////////////////////////////////////////////////////
DsChargingDevice::DsChargingDevice() NN_NOEXCEPT
{

    // Product string
    {
        uint8_t productString[32];
        size_t bytes = nn::settings::fwdbg::GetSettingsItemValue(productString, 32, "productinfo", "product_name") - 1; // less zero terminator

        m_UsbStringDescriptorProduct.bLength = 2 + (bytes * 2);
        uint16_t *pDst  = &m_UsbStringDescriptorProduct.wData[0];
        uint8_t *pSrc   = productString;

        if (bytes)
        {
            while (bytes--)
            {
                *pDst++ = static_cast<uint16_t>(*pSrc++);
            }
        }
        else
        {
            NN_USB_LOG_WARN("product_name could not be loaded\n");
        }
    }
}


//////////////////////////////////////////////////////////////////////////////
void DsChargingDevice::TxLoop() NN_NOEXCEPT
{
    memset(m_TxBuffer, 0, sizeof(m_TxBuffer));

    StartPeriodicTimerEvent(&m_TimerEvent, nn::TimeSpan::FromSeconds(1), nn::TimeSpan::FromSeconds(1));

    while (true)
    {
        nn::os::MultiWaitHolderType *pHolder = nn::os::WaitAny(&m_MultiWait);

        if (pHolder == &m_FinalizeHolder)
        {
            nn::os::ClearEvent(&m_FinalizeEvent);
            break;
        }

        if (pHolder == &m_TimerHolder)
        {
            nn::os::ClearTimerEvent(&m_TimerEvent);

            if (m_pDsProtocol->GetState(nullptr) == UsbState_Configured)
            {
                uint32_t bytesTransferred;
                m_pDsProtocol->PostBuffer(0x81, m_TxBuffer, 1, &bytesTransferred);
            }
        }
    }

    StopTimerEvent(&m_TimerEvent);
}


//////////////////////////////////////////////////////////////////////////////
Result DsChargingDevice::Initialize(DsProtocol *pDsProtocol) NN_NOEXCEPT
{
    NN_USB_TRACE;

    uint8_t stringIndex;

    m_pDsProtocol = pDsProtocol;

    m_pDsProtocol->ClearDeviceData();

    m_pDsProtocol->AddUsbStringDescriptor(&m_UsbStringDescriptorLanguage, &stringIndex);
    m_pDsProtocol->AddUsbStringDescriptor(&m_UsbStringDescriptorManufacturer, &stringIndex);
    m_pDsProtocol->AddUsbStringDescriptor(&m_UsbStringDescriptorProduct, &stringIndex);

    m_pDsProtocol->SetUsbDeviceDescriptor(&m_UsbDeviceDescriptorFullSpeed, UsbDeviceSpeed_Full);
    m_pDsProtocol->SetUsbDeviceDescriptor(&m_UsbDeviceDescriptorHighSpeed, UsbDeviceSpeed_High);
    m_pDsProtocol->SetUsbDeviceDescriptor(&m_UsbDeviceDescriptorSuperSpeed, UsbDeviceSpeed_Super);
    m_pDsProtocol->SetBinaryObjectStore(&m_BinaryObjectStore[0], sizeof(m_BinaryObjectStore));

    m_pDsProtocol->AppendConfigurationData(m_UsbInterfaceDescriptor.bInterfaceNumber, UsbDeviceSpeed_Full, reinterpret_cast<uint8_t*>(&m_UsbInterfaceDescriptor), sizeof(UsbInterfaceDescriptor));
    m_pDsProtocol->AppendConfigurationData(m_UsbInterfaceDescriptor.bInterfaceNumber, UsbDeviceSpeed_Full, reinterpret_cast<uint8_t*>(&m_UsbHidDescriptor), sizeof(UsbHidDescriptor));
    m_pDsProtocol->AppendConfigurationData(m_UsbInterfaceDescriptor.bInterfaceNumber, UsbDeviceSpeed_Full, reinterpret_cast<uint8_t*>(&m_UsbEndpointDescriptor), sizeof(UsbEndpointDescriptor));

    m_pDsProtocol->AppendConfigurationData(m_UsbInterfaceDescriptor.bInterfaceNumber, UsbDeviceSpeed_High, reinterpret_cast<uint8_t*>(&m_UsbInterfaceDescriptor), sizeof(UsbInterfaceDescriptor));
    m_pDsProtocol->AppendConfigurationData(m_UsbInterfaceDescriptor.bInterfaceNumber, UsbDeviceSpeed_High, reinterpret_cast<uint8_t*>(&m_UsbHidDescriptor), sizeof(UsbHidDescriptor));
    m_pDsProtocol->AppendConfigurationData(m_UsbInterfaceDescriptor.bInterfaceNumber, UsbDeviceSpeed_High, reinterpret_cast<uint8_t*>(&m_UsbEndpointDescriptor), sizeof(UsbEndpointDescriptor));

    m_pDsProtocol->AppendConfigurationData(m_UsbInterfaceDescriptor.bInterfaceNumber, UsbDeviceSpeed_Super, reinterpret_cast<uint8_t*>(&m_UsbInterfaceDescriptor), sizeof(UsbInterfaceDescriptor));
    m_pDsProtocol->AppendConfigurationData(m_UsbInterfaceDescriptor.bInterfaceNumber, UsbDeviceSpeed_Super, reinterpret_cast<uint8_t*>(&m_UsbHidDescriptor), sizeof(UsbHidDescriptor));
    m_pDsProtocol->AppendConfigurationData(m_UsbInterfaceDescriptor.bInterfaceNumber, UsbDeviceSpeed_Super, reinterpret_cast<uint8_t*>(&m_UsbEndpointDescriptor), sizeof(UsbEndpointDescriptor));
    m_pDsProtocol->AppendConfigurationData(m_UsbInterfaceDescriptor.bInterfaceNumber, UsbDeviceSpeed_Super, reinterpret_cast<uint8_t*>(&m_UsbEndpointCompanionDescriptor), sizeof(UsbEndpointCompanionDescriptor));

    m_pDsProtocol->SetReportData(m_UsbInterfaceDescriptor.bInterfaceNumber, m_HidReportDescriptor, sizeof(m_HidReportDescriptor));

    m_pDsProtocol->RegisterInterface(m_UsbInterfaceDescriptor.bInterfaceNumber);
    m_pDsProtocol->RegisterEndpoint(m_UsbEndpointDescriptor.bEndpointAddress, m_UsbInterfaceDescriptor.bInterfaceNumber);
    m_pDsProtocol->EnableInterface(m_UsbInterfaceDescriptor.bInterfaceNumber);
    m_pDsProtocol->EnableDevice();

    nn::os::InitializeMultiWait(&m_MultiWait);
    nn::os::InitializeEvent(&m_FinalizeEvent, false, nn::os::EventClearMode_ManualClear);
    nn::os::InitializeTimerEvent(&m_TimerEvent, nn::os::EventClearMode_ManualClear);
    nn::os::InitializeMultiWaitHolder(&m_FinalizeHolder, &m_FinalizeEvent);
    nn::os::InitializeMultiWaitHolder(&m_TimerHolder, &m_TimerEvent);
    nn::os::LinkMultiWaitHolder(&m_MultiWait, &m_FinalizeHolder);
    nn::os::LinkMultiWaitHolder(&m_MultiWait, &m_TimerHolder);

    NN_USB_ABORT_UNLESS_SUCCESS(nn::os::CreateThread(
                                                        &m_Thread,
                                                        TxThread,
                                                        this,
                                                        &m_ThreadStack,
                                                        sizeof(m_ThreadStack),
                                                        NN_SYSTEM_THREAD_PRIORITY(usb,DsThread))
                                                        );
    nn::os::SetThreadNamePointer(&m_Thread, NN_SYSTEM_THREAD_NAME(usb, DsThread));
    nn::os::StartThread(&m_Thread);

    return ResultSuccess();
}


//////////////////////////////////////////////////////////////////////////////
Result DsChargingDevice::Finalize() NN_NOEXCEPT
{
    NN_USB_TRACE;

    // stop the TX loop thread
    nn::os::SignalEvent(&m_FinalizeEvent);
    m_pDsProtocol->Cancel(0x81);
    nn::os::WaitThread(&m_Thread);
    nn::os::DestroyThread(&m_Thread);

    m_pDsProtocol->DisableDevice();
    m_pDsProtocol->DisableInterface(m_UsbInterfaceDescriptor.bInterfaceNumber);
    m_pDsProtocol->UnRegisterEndpoint(m_UsbEndpointDescriptor.bEndpointAddress);
    m_pDsProtocol->UnRegisterInterface(m_UsbInterfaceDescriptor.bInterfaceNumber);
    m_pDsProtocol->ClearDeviceData();

    nn::os::UnlinkMultiWaitHolder(&m_TimerHolder);
    nn::os::UnlinkMultiWaitHolder(&m_FinalizeHolder);
    nn::os::FinalizeMultiWaitHolder(&m_TimerHolder);
    nn::os::FinalizeMultiWaitHolder(&m_FinalizeHolder);
    nn::os::FinalizeTimerEvent(&m_TimerEvent);
    nn::os::FinalizeEvent(&m_FinalizeEvent);
    nn::os::FinalizeMultiWait(&m_MultiWait);

    return ResultSuccess();
}


} // end of namespace ds
} // end of namespace usb
} // end of namespace nn

