﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file  usb_Utils.h
 * @brief Helper utilities for USB stack
 *
 * @details
 */

#pragma once

#include <list>
#include <memory>
#include <algorithm>
#include <type_traits>

#include <nn/nn_Common.h>
#include <nn/nn_SdkLog.h>

#include <nn/sf/sf_ExpHeapAllocator.h>
#include <nn/sf/sf_ObjectFactory.h>

#include <nn/util/util_IntrusiveList.h>

#include <nn/usb/usb_Result.h>
#include <nn/usb/usb_Types.h>
#include <nn/usb/usb_DiagTypes.h>

#include "usb_Memory.h"

// Log macros

#if defined (NN_USB_HS)

#define NN_USB_LOG_TAG    "hs"
#define NN_USB_LOG_MODULE nn::usb::LogModule_Generic

#elif defined (NN_USB_DS)

#define NN_USB_LOG_TAG    "ds"
#define NN_USB_LOG_MODULE nn::usb::LogModule_Generic

#elif defined (NN_USB_PM)

#define NN_USB_LOG_TAG    "pm"
#define NN_USB_LOG_MODULE nn::usb::LogModule_Generic

#elif defined (NN_USB_HS_SHIM)

#define NN_USB_LOG_TAG    "hs"
#define NN_USB_LOG_MODULE nn::usb::LogModule_Shim

#elif defined (NN_USB_DS_SHIM)

#define NN_USB_LOG_TAG    "ds"
#define NN_USB_LOG_MODULE nn::usb::LogModule_Shim

#elif defined (NN_USB_PM_SHIM)

#define NN_USB_LOG_TAG    "pm"
#define NN_USB_LOG_MODULE nn::usb::LogModule_Shim

#else

#define NN_USB_LOG_TAG    "**"
#define NN_USB_LOG_MODULE nn::usb::LogModule_Generic

#endif

#define NN_USB_LOG_TRACE(format, ...)                           \
    nn::usb::detail::Log(nn::usb::LogLevel_Trace,               \
                         NN_USB_LOG_MODULE,                     \
                         NN_USB_LOG_TAG, format, ##__VA_ARGS__)
#define NN_USB_LOG_INFO(format, ...)                            \
    nn::usb::detail::Log(nn::usb::LogLevel_Info,                \
                         NN_USB_LOG_MODULE,                     \
                         NN_USB_LOG_TAG, format, ##__VA_ARGS__)
#define NN_USB_LOG_WARN(format, ...)                            \
    nn::usb::detail::Log(nn::usb::LogLevel_Warn,                \
                         NN_USB_LOG_MODULE,                     \
                         NN_USB_LOG_TAG, format, ##__VA_ARGS__)
#define NN_USB_LOG_ERROR(format, ...)                           \
    nn::usb::detail::Log(nn::usb::LogLevel_Error,               \
                         NN_USB_LOG_MODULE,                     \
                         NN_USB_LOG_TAG, format, ##__VA_ARGS__)


// USB Trace helpers - Enable this define globally or first in each file to trace
#ifdef NN_USB_ENABLE_TRACE

// You can undef/redef this macro multiple times per file to change the string
#ifndef NN_USB_TRACE_CLASS_NAME
#define NN_USB_TRACE_CLASS_NAME ""
#endif

#define NN_USB_TRACE \
    NN_USB_LOG_INFO("%s::%s(@%d)\n", NN_USB_TRACE_CLASS_NAME, __FUNCTION__,__LINE__);

#define NN_USB_TRACE_END \
    NN_USB_LOG_INFO("    End of %s::%s(@%d)\n", NN_USB_TRACE_CLASS_NAME, __FUNCTION__,__LINE__);
#else

#define NN_USB_TRACE
#define NN_USB_TRACE_END

#endif

#define NN_USB_ABORT(format,...)                                                         \
    NN_ABORT("USB " NN_USB_LOG_TAG " abort: " format, ##__VA_ARGS__)
#define NN_USB_ABORT_UNLESS_SUCCESS(result)                                              \
    NN_ABORT_UNLESS((result).IsSuccess())
#define NN_USB_ABORT_IF_NULL(pointerVal)                                                 \
    NN_ABORT_UNLESS_NOT_NULL(pointerVal)

// Error handling macros
#define NN_USB_ASSERT_IF_ZERO(uintval)                                                   \
    if (!(uintval))                                                                      \
    {                                                                                    \
        NN_USB_ABORT("%s is zero in func-%s @ line-%d.\n",                               \
                        #uintval,__FUNCTION__,__LINE__);                                 \
    }

#define NN_USB_BREAK_UPON_ERROR(attemptedMethodCall)                                     \
    if ((result=(attemptedMethodCall)).IsFailure())                                      \
    {                                                                                    \
        NN_USB_LOG_WARN("%s unsuccessful %d:%d in func-%s @ line-%d.\n",                 \
                           #attemptedMethodCall,result.GetModule(),                      \
                           result.GetDescription(),__FUNCTION__,__LINE__);               \
        break;                                                                           \
    }

#define NN_USB_BREAK_UNLESS(condition, failureResult)                                    \
    if (!(condition))                                                                    \
    {                                                                                    \
        NN_USB_LOG_WARN("condition %s false in func-%s @ line-%d.\n",                    \
                         #condition,__FUNCTION__,__LINE__);                              \
        result=failureResult;                                                            \
        break;                                                                           \
    }

#define NN_USB_BREAK_UPON_MEM_ALLOC_FAIL(attemptedAlloc)                                 \
    if ((attemptedAlloc) == nullptr)                                                     \
    {                                                                                    \
        result = ResultMemAllocFailure();                                                \
        NN_USB_LOG_WARN("%s unsuccessful alloc in func-%s @ line-%d.\n",                 \
                        #attemptedAlloc,__FUNCTION__,__LINE__);                          \
        break;                                                                           \
    }

#define NN_USB_BREAK_UPON_NULL_WITH_ERROR(pointer, resultCode)                           \
    if ((pointer) == nullptr)                                                            \
    {                                                                                    \
        result = resultCode;                                                             \
        NN_USB_LOG_WARN("%s is NULL in func-%s @ line-%d, error %d:%d.\n",               \
                        #pointer,__FUNCTION__,__LINE__,result.GetModule(),               \
                        result.GetDescription());                                        \
        break;                                                                           \
    }

#define NN_USB_ABORT_UNLESS(condition)                                                   \
    if (!(condition))                                                                    \
    {                                                                                    \
        NN_USB_ABORT("condition %s false in func-%s @ line-%d.\n",                       \
                     #condition,__FUNCTION__,__LINE__);                                  \
    }


#define NN_USB_RETURN_UPON_ERROR(attemptedMethodCall)                                    \
    if ((result=(attemptedMethodCall)).IsFailure())                                      \
    {                                                                                    \
        NN_USB_LOG_WARN("%s unsuccessful %d:%d in func-%s @ line-%d.\n",                 \
                        #attemptedMethodCall,result.GetModule(),                         \
                        result.GetDescription(),__FUNCTION__,__LINE__);                  \
        return result;                                                                   \
    }

#define NN_USB_VOID_RETURN_UPON_ERROR(attemptedMethodCall)                               \
    do {                                                                                 \
        Result result = (attemptedMethodCall);                                           \
        if (result.IsFailure())                                                          \
        {                                                                                \
            NN_USB_LOG_WARN("%s unsuccessful %d:%d in func-%s @ line-%d.\n",             \
                            #attemptedMethodCall,result.GetModule(),                     \
                            result.GetDescription(),__FUNCTION__,__LINE__);              \
            return;                                                                      \
        }                                                                                \
    } while (0)

#define NN_USB_ABORT_UPON_ERROR(attemptedMethodCall)                                     \
    if ((result=(attemptedMethodCall)).IsFailure())                                      \
    {                                                                                    \
        NN_USB_ABORT("%s unsuccessful %d:%d in func-%s @ line-%d.\n",                    \
                     #attemptedMethodCall,result.GetModule(),                            \
                     result.GetDescription(),__FUNCTION__,__LINE__);                     \
    }

#define NN_USB_EXPECT(expectation, result)                                               \
    do {                                                                                 \
        if (!(expectation))                                                              \
        {                                                                                \
            NN_USB_LOG_WARN("Excpecting %s %d:%d in func-%s @ line-%d.\n",               \
                            #expectation, result.GetModule(),                            \
                            result.GetDescription(),__FUNCTION__,__LINE__);              \
            return (result);                                                             \
        }                                                                                \
    } while (0)

#define NN_USB_LOG_UPON_ERROR(attemptedMethodCall)                                       \
    if ((result=(attemptedMethodCall)).IsFailure())                                      \
    {                                                                                    \
        NN_USB_LOG_WARN("%s unsuccessful %d:%d in func-%s @ line-%d.\n",                 \
                        #attemptedMethodCall,result.GetModule(),                         \
                        result.GetDescription(),__FUNCTION__,__LINE__);                  \
    }

// Maximum and minimum
#define NN_USB_MIN(a, b)   (((a) < (b)) ? (a) : (b))
#define NN_USB_MAX(a, b)   (((a) > (b)) ? (a) : (b))

// Arithmetic
#define NN_USB_SATURATED_ADD_U32(a, b)                                                   \
    (((b) > (0xFFFFFFFF - (a))) ? 0xFFFFFFFF : ((a) += (b)))
#define NN_USB_ROUNDUP_SIZE(size, significance)                                          \
    ((((size) + (significance) - 1) / (significance)) * (significance))

#define NN_USB_ROUNDUP_DMA_SIZE(size)                                                    \
    NN_USB_ROUNDUP_SIZE(size,nn::dd::DeviceAddressSpaceMemoryRegionAlignment)

#define NN_USB_ARRAY_SIZE(array) (sizeof(array) / sizeof((array)[0]))

#define NN_USB_ARRAY_COUNT32(array) (int32_t)(sizeof(array) / sizeof((array)[0]))

namespace nn {
namespace usb {
namespace detail {

template<class T>
inline bool IsInRange(const T& var, const T& from, const T& to)
{
    return (var >= from ) && (var <= to);
}

void Log(LogLevel level, LogModule module, const char *tag, const char *format, ...);
bool IsLogVerbose();

void     UsbDiagInitialize();
void     UsbDiagFinalize();
void     SetDiagData(DiagData data, uint32_t value);
uint32_t GetDiagData(DiagData data);

int  GetEndpointIndex(uint8_t address);

void DumpCtrlRequest(UsbCtrlRequest *pCtrl);
void DumpDeviceDescriptor(UsbDeviceDescriptor *pDesc);
void DumpInterfaceDescriptor(UsbInterfaceDescriptor *pDesc);
void DumpEndpointDescriptor(UsbEndpointDescriptor *pDesc);
void DumpBuffer(void *buffer, int size);

template <class T>
struct Allocator : public std::allocator<T>
{
    typedef std::allocator<T> Base;
    typedef typename Base::pointer pointer;
    typedef typename Base::reference reference;
    typedef typename Base::const_pointer const_pointer;
    typedef typename Base::const_reference const_reference;
    typedef typename Base::size_type size_type;

    Allocator() NN_NOEXCEPT {}

    Allocator(const Allocator&) NN_NOEXCEPT {}

    template<class U>
    Allocator(const Allocator<U>&) NN_NOEXCEPT {}

    template<class U>
    struct rebind
    {
        typedef Allocator<U> other;
    };

    pointer address(reference x) const
    {
        return &x;
    }

    const_pointer address(const_reference x) const
    {
        return &x;
    }

    pointer allocate(size_type count, const_pointer hint = 0) NN_NOEXCEPT
    {
        NN_UNUSED(hint);

        if(count == 0)
        {
            return nullptr;
        }

        pointer p = static_cast<pointer>(
            // roundup to avoid heap fragmentation
            UsbMemoryAlloc(NN_USB_ROUNDUP_SIZE(sizeof(T) * count,128), "usb_Util::Allocator")
        );

        if(p == nullptr)
        {
            NN_USB_ABORT("Allocator::allocate() failed.\n");
        }

        return p;
    }

    void deallocate(pointer p, size_type count) NN_NOEXCEPT
    {
        NN_UNUSED(count);
        UsbMemoryFree(p, "usb_Util::Allocator");
    }
};


/*
 * std::list with lmem allocator
 * UsbMemoryInit() must be called before StdList can be used
 */
template<class T>
class StdList : public std::list<T, Allocator<T>>
{
    // nothing special
};

template<class T>
T Clamp(const T& value, const T& low, const T& high)
{
    return value < low ? low : (value > high ? high : value);
}

/*
 * Type alias for IntrusiveList
 */
template<class T>
using NnList = nn::util::IntrusiveList<T, nn::util::IntrusiveListMemberNodeTraits<T, &T::m_ListNode>>;

typedef nn::sf::ObjectFactory<nn::sf::ExpHeapAllocator::Policy> Factory;

/*
 * ScopedRefCount
 */
template<class MyRefCountClass>
class ScopedRefCount
{
public:
    NN_IMPLICIT ScopedRefCount(MyRefCountClass & refCount) : m_RefCount(refCount)
    {
        m_RefCount++;
    }
    ~ScopedRefCount()
    {
        m_RefCount--;
    }
private:
    MyRefCountClass & m_RefCount;
};

} // end of namespace detail
} // end of namespace usb
} // end of namespace nn


