﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/os.h>
#include <nn/nn_SdkLog.h>
#include <nn/util/util_FormatString.h>
#include <nn/usb/usb_Types.h>
#include <nn/usb/detail/usb_Log.h>

#include "usb_Util.h"

namespace nn {
namespace usb {
namespace detail {

static uint32_t g_DiagData[DiagData_Max];

void Log(LogLevel level, LogModule module, const char *tag, const char *format, ...)
{
#ifndef NN_SDK_BUILD_RELEASE

    va_list ap;
    uint32_t offset;
    const uint32_t bufSize = 512;
    char buffer[bufSize];

    uint32_t currentLevel  = GetDiagData(DiagData_LogLevel);
    uint32_t enabledModule = GetDiagData(DiagData_LogModule);

    if (level < currentLevel || (enabledModule & (1 << module)) == 0)
    {
        return;
    }

    va_start(ap, format);

    nn::util::SNPrintf(buffer, bufSize, "%s | ", tag);
    offset = strnlen(buffer, bufSize);
    nn::util::VSNPrintf(buffer + offset, bufSize - offset, format, ap);

    va_end(ap);

    switch (level)
    {
    case LogLevel_Trace:
        NN_DETAIL_USB_TRACE(buffer);
        break;

    case LogLevel_Info:
        NN_DETAIL_USB_INFO(buffer);
        break;

    case LogLevel_Warn:
        NN_DETAIL_USB_WARN(buffer);
        break;

    case LogLevel_Error:
        NN_DETAIL_USB_ERROR(buffer);
        break;

    default:
        NN_UNEXPECTED_DEFAULT;
    }

#endif
}

bool IsLogVerbose()
{
    return g_DiagData[DiagData_LogLevel] <= LogLevel_Trace;
}

void UsbDiagInitialize()
{
    g_DiagData[DiagData_Debug]     = 0;
    g_DiagData[DiagData_LogLevel]  = LogLevel_Info;
    g_DiagData[DiagData_LogModule] = (1 << LogModule_Generic) | (1 << LogModule_Shim);
}

void UsbDiagFinalize()
{
    // nothing for now
}
void SetDiagData(DiagData data, uint32_t value)
{
    g_DiagData[data] = value;
}

uint32_t GetDiagData(DiagData data)
{
    return g_DiagData[data];
}

int GetEndpointIndex(uint8_t address)
{
    int index = address & UsbEndpointAddressMask_EndpointNumber;

    if (address & UsbEndpointAddressMask_DirDeviceToHost)
    {
        index += 16;
    }

    return index;
}

void DumpBuffer(void *buffer, int size)
{
    char *p = (char*)buffer;

    int remain = size;
    int line   = 0;

    while (remain)
    {
        int i;
        int offset = size - remain;

        // offset marker
        NN_USB_LOG_INFO("%04x: ", line++);

        // hex dump
        for (i = 0; i < remain && i < 16; i++)
        {
            NN_SDK_LOG(" %02x", p[offset + i]);

            if (i % 16 == 7)
            {
                NN_SDK_LOG(" ");
            }
        }

        // hex padding
        for (; i < 16; i++)
        {
            NN_SDK_LOG("   ");

            if (i % 16 == 7)
            {
                NN_SDK_LOG(" ");
            }
        }

        // delimiter between hex and char
        NN_SDK_LOG(" | ");

        // char dump
        for (i = 0; i < remain && i < 16; i++)
        {
            char c = p[offset + i];
            if (c >= ' ' && c <= '~')
            {
                NN_SDK_LOG("%c", c);
            }
            else
            {
                NN_SDK_LOG(".");
            }

            if (i % 16 == 7)
            {
                NN_SDK_LOG(" ");
            }
        }

        // char padding
        for (; i < 16; i++)
        {
            NN_SDK_LOG(" ");

            if (i % 16 == 7)
            {
                NN_SDK_LOG(" ");
            }
        }

        NN_SDK_LOG("\n");

        remain = (remain >= 16) ? (remain - 16) : 0;
    }
}

void DumpCtrlRequest(UsbCtrlRequest *pCtrl)
{
    NN_USB_LOG_INFO("SETUP:\n");
    NN_USB_LOG_INFO("    bmRequestType: %02x\n", pCtrl->bmRequestType);
    NN_USB_LOG_INFO("    bRequest     : %02x\n", pCtrl->bRequest);
    NN_USB_LOG_INFO("    wValue       : %04x\n", pCtrl->wValue);
    NN_USB_LOG_INFO("    wIndex       : %04x\n", pCtrl->wIndex);
    NN_USB_LOG_INFO("    wLength      : %04x\n", pCtrl->wLength);
}

void DumpDeviceDescriptor(UsbDeviceDescriptor *pDesc)
{
    NN_USB_LOG_INFO("Device Descriptor:\n");
    NN_USB_LOG_INFO("    bLength           : %02x\n", pDesc->bLength);
    NN_USB_LOG_INFO("    bDescriptorType   : %02x\n", pDesc->bDescriptorType);
    NN_USB_LOG_INFO("    bcdUSB            : %04x\n", pDesc->bcdUSB);
    NN_USB_LOG_INFO("    bDeviceClass      : %02x\n", pDesc->bDeviceClass);
    NN_USB_LOG_INFO("    bDeviceSubClass   : %02x\n", pDesc->bDeviceSubClass);
    NN_USB_LOG_INFO("    bDeviceProtocol   : %02x\n", pDesc->bDeviceProtocol);
    NN_USB_LOG_INFO("    bMaxPacketSize0   : %02x\n", pDesc->bMaxPacketSize0);
    NN_USB_LOG_INFO("    idVendor          : %04x\n", pDesc->idVendor);
    NN_USB_LOG_INFO("    idProduct         : %04x\n", pDesc->idProduct);
    NN_USB_LOG_INFO("    bcdDevice         : %04x\n", pDesc->bcdDevice);
    NN_USB_LOG_INFO("    iManufacturer     : %02x\n", pDesc->iManufacturer);
    NN_USB_LOG_INFO("    iProduct          : %02x\n", pDesc->iProduct);
    NN_USB_LOG_INFO("    iSerialNumber     : %02x\n", pDesc->iSerialNumber);
    NN_USB_LOG_INFO("    bNumConfigurations: %02x\n", pDesc->bNumConfigurations);
}

void DumpInterfaceDescriptor(UsbInterfaceDescriptor *pDesc)
{
    NN_USB_LOG_INFO("    bLength           : %02x\n", pDesc->bLength);
    NN_USB_LOG_INFO("    bDescriptorType   : %02x\n", pDesc->bDescriptorType);
    NN_USB_LOG_INFO("    bInterfaceNumber  : %02x\n", pDesc->bInterfaceNumber);
    NN_USB_LOG_INFO("    bAlternateSetting : %02x\n", pDesc->bAlternateSetting);
    NN_USB_LOG_INFO("    bNumEndpoints     : %02x\n", pDesc->bNumEndpoints);
    NN_USB_LOG_INFO("    bInterfaceClass   : %02x\n", pDesc->bInterfaceClass);
    NN_USB_LOG_INFO("    bInterfaceSubClass: %02x\n", pDesc->bInterfaceSubClass);
    NN_USB_LOG_INFO("    bInterfaceProtocol: %02x\n", pDesc->bInterfaceProtocol);
    NN_USB_LOG_INFO("    iInterfacd        : %02x\n", pDesc->iInterface);
}

void DumpEndpointDescriptor(UsbEndpointDescriptor *pDesc)
{
    NN_USB_LOG_INFO("    bLength           : %02x\n", pDesc->bLength);
    NN_USB_LOG_INFO("    bDescriptorType   : %02x\n", pDesc->bDescriptorType);
    NN_USB_LOG_INFO("    bEndpointAddress  : %02x\n", pDesc->bEndpointAddress);
    NN_USB_LOG_INFO("    bmAttributes      : %02x\n", pDesc->bmAttributes);
    NN_USB_LOG_INFO("    bwMaxPacketSize   : %04x\n", pDesc->wMaxPacketSize);
    NN_USB_LOG_INFO("    bInterval         : %02x\n", pDesc->bInterval);
}


} // end of namespace detail
} // end of namespace usb
} // end of namespace nn
