﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/usb/usb_Limits.h>

#include "usb_SmmuMapManager.h"

namespace nn { namespace usb { namespace detail {


//#define NN_USB_HS_LOG_SMMU NN_USB_LOG_INFO
#define NN_USB_HS_LOG_SMMU(format,...)

Result SmmuMapManager::Initialize(nn::dd::DeviceName dmaDeviceName,
                                  nn::dd::DeviceVirtualAddress baseIoVa,
                                  size_t ioVaHeapSize)
{
    Result result = ResultSuccess();

    do
    {
        m_DeviceName = dmaDeviceName;
        m_IoVaManager.Initialize(baseIoVa, ioVaHeapSize);
        NN_USB_BREAK_UPON_ERROR(nn::dd::CreateDeviceAddressSpace(&m_DeviceAddressSpace,
                                                                 0x100000000ull));
        NN_USB_BREAK_UPON_ERROR(nn::dd::AttachDeviceAddressSpace(&m_DeviceAddressSpace,
                                                                 m_DeviceName));
    }while (false);

    return result;
}

Result SmmuMapManager::Finalize()
{
    Result result = ResultSuccess();

    nn::dd::DetachDeviceAddressSpace(&m_DeviceAddressSpace, m_DeviceName);
    nn::dd::DestroyDeviceAddressSpace(&m_DeviceAddressSpace);
    m_IoVaManager.Finalize();

    return result;
}

Result SmmuMapManager::Map(nn::dd::DeviceVirtualAddress *pOutIoVa,
                           nn::dd::ProcessHandle processHandle,
                           uint64_t processAddress, size_t size,
                           size_t alignment,
                           nn::dd::MemoryPermission cpuPermission,
                           nn::dd::MemoryPermission devicePermission)
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);
    nn::dd::DeviceVirtualAddress ioVa = 0ULL;
    Result result = ResultSuccess();
    size_t alignmentR;

    NN_UNUSED(cpuPermission);

    /*
     * Check / fix the alignment.
     *
     * Inappropriate alignment leads to abort, we don't want that because those
     * parameters could be provided by class driver.
     */
    if (!NN_USB_IS_DMA_ALIGNED(processAddress) || !NN_USB_IS_DMA_ALIGNED(size))
    {
        return ResultAlignmentError();
    }
    alignmentR = NN_USB_ROUNDUP_SIZE(alignment, HwLimitDmaBufferAlignmentSize);

    // Allocate IoVa space
    ioVa = m_IoVaManager.Allocate(size, alignmentR, 0, processAddress,
                                  static_cast<uint64_t>(processHandle), 0);

    if (ioVa == 0)
    {
        result = ResultIoVaError();
    }
    else
    {
        result = nn::dd::MapDeviceAddressSpaceNotAligned(&m_DeviceAddressSpace,
                                                         processHandle,
                                                         processAddress,
                                                         size, ioVa,
                                                         devicePermission);
    }

    if (result.IsSuccess())
    {
        if (pOutIoVa != nullptr)
        {
            *pOutIoVa = ioVa;
        }
    }
    else
    {
        if (ioVa != 0)
        {
            m_IoVaManager.Free(ioVa);
            ioVa = 0;
        }
    }

    NN_USB_HS_LOG_SMMU("SmmuMapManager::Map(outVa=%p, procHandle=0x%x, "
                       "procVa=%p, size=0x%x, align=0x%x cpuPer=0x%x, "
                       "dmaPer=0x%x) = %d:%d\n",
                       ioVa, processHandle, processAddress, size, alignment,
                       cpuPermission, devicePermission,
                       result.GetModule(), result.GetDescription());
    return result;
}

Result SmmuMapManager::Map(nn::dd::DeviceVirtualAddress *pOutIoVa,
                           nn::dd::ProcessHandle processHandle,
                           uint64_t processAddress, size_t size,
                           nn::dd::MemoryPermission cpuPermission,
                           nn::dd::MemoryPermission devicePermission)
{
    return Map(
        pOutIoVa,
        processHandle, processAddress, size, HwLimitDmaBufferAlignmentSize,
        cpuPermission, devicePermission
    );
}


Result SmmuMapManager::Map(nn::dd::DeviceVirtualAddress *pOutIoVa,
                           void* pBuffer,  size_t size,
                           nn::dd::MemoryPermission cpuPermission, nn::dd::MemoryPermission devicePermission)
{
    return Map(pOutIoVa, nn::dd::GetCurrentProcessHandle(), reinterpret_cast<uint64_t>(pBuffer), size,
               cpuPermission, devicePermission);
}

Result SmmuMapManager::Map(nn::dd::DeviceVirtualAddress *pOutIoVa, void* pBuffer, size_t size)
{
    return Map(pOutIoVa, nn::dd::GetCurrentProcessHandle(), reinterpret_cast<uint64_t>(pBuffer), size,
               nn::dd::MemoryPermission_ReadWrite, nn::dd::MemoryPermission_ReadWrite);
}

Result SmmuMapManager::Map(nn::dd::DeviceVirtualAddress *pOutIoVa, void* pBuffer, size_t size, size_t alignment)
{
    return Map(pOutIoVa, nn::dd::GetCurrentProcessHandle(), reinterpret_cast<uint64_t>(pBuffer), size, alignment,
               nn::dd::MemoryPermission_ReadWrite, nn::dd::MemoryPermission_ReadWrite);
}

Result SmmuMapManager::Map(void* pBuffer, size_t size)
{
    return Map(nullptr, nn::dd::GetCurrentProcessHandle(), reinterpret_cast<uint64_t>(pBuffer), size,
               nn::dd::MemoryPermission_ReadWrite, nn::dd::MemoryPermission_ReadWrite);
}

Result SmmuMapManager::Unmap(nn::dd::ProcessHandle procHandle, uint64_t procVa, uint32_t size)
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);
    Result result = ResultSuccess();
    IoVaAllocationSummary summary;

    NN_USB_RETURN_UPON_ERROR(m_IoVaManager.LookupByProcVa(0, procHandle, procVa, &summary));

    NN_USB_ABORT_UNLESS(size == summary.requestedSize);

    nn::dd::UnmapDeviceAddressSpace(&m_DeviceAddressSpace,
                                    procHandle,
                                    procVa,
                                    size,
                                    summary.ioVa);

    // Abort if failure, since it suggests corrupted IoVa table
    NN_USB_ABORT_UPON_ERROR(m_IoVaManager.Free(summary.ioVa));

    NN_USB_HS_LOG_SMMU("SmmuMapManager::Unmap(procVa=%p, size=0x%x) = %d:%d\n",
                       procVa, size,  result.GetModule(), result.GetDescription());

    return result;
}

Result SmmuMapManager::Unmap(void *pBuffer, uint32_t size)
{
    nn::dd::ProcessHandle procHandle = nn::dd::GetCurrentProcessHandle();
    uint64_t procVa = reinterpret_cast<uint64_t>(pBuffer);

    return Unmap(procHandle, procVa, size);
}

Result SmmuMapManager::Unmap(void *pBuffer)
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);
    Result result = ResultSuccess();
    nn::dd::ProcessHandle procHandle = nn::dd::GetCurrentProcessHandle();
    uint64_t procVa = reinterpret_cast<uint64_t>(pBuffer);
    IoVaAllocationSummary summary;

    NN_USB_RETURN_UPON_ERROR(m_IoVaManager.LookupByProcVa(0, procHandle, procVa, &summary));

    nn::dd::UnmapDeviceAddressSpace(&m_DeviceAddressSpace,
                                    static_cast<nn::dd::ProcessHandle>(summary.context),
                                    reinterpret_cast<uint64_t>(pBuffer),
                                    summary.requestedSize,
                                    summary.ioVa);

    // Abort if failure, since it suggests corrupted IoVa table
    NN_USB_ABORT_UPON_ERROR(m_IoVaManager.Free(summary.ioVa));

    return result;
}

Result SmmuMapManager::UnmapByIoVa(nn::dd::DeviceVirtualAddress ioVa, uint32_t size)
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);
    Result result = ResultSuccess();
    IoVaAllocationSummary summary;

    NN_USB_RETURN_UPON_ERROR(m_IoVaManager.LookupByIoVa(ioVa, &summary));
    NN_USB_ABORT_UNLESS(size == summary.requestedSize);

    nn::dd::UnmapDeviceAddressSpace(&m_DeviceAddressSpace,
                                    static_cast<nn::dd::ProcessHandle>(summary.context),
                                    reinterpret_cast<uint64_t>(summary.procVa),
                                    summary.requestedSize,
                                    summary.ioVa);

    // Abort if failure, since it suggests corrupted IoVa table
    NN_USB_ABORT_UPON_ERROR(m_IoVaManager.Free(summary.ioVa));

    NN_USB_HS_LOG_SMMU("SmmuMapManager::UnmapByIoVa(ioVa=%p, size=0x%x) = %d:%d\n",
                       (void*)ioVa, size,  result.GetModule(), result.GetDescription());

    return result;
}

Result SmmuMapManager::UnmapByIoVa(nn::dd::DeviceVirtualAddress ioVa)
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);
    Result result = ResultSuccess();
    IoVaAllocationSummary summary;
    NN_USB_RETURN_UPON_ERROR(m_IoVaManager.LookupByIoVa(ioVa, &summary));
    nn::dd::UnmapDeviceAddressSpace(&m_DeviceAddressSpace,
                                    static_cast<nn::dd::ProcessHandle>(summary.context),
                                    reinterpret_cast<uint64_t>(summary.procVa),
                                    summary.requestedSize,
                                    summary.ioVa);

    // Abort if failure, since it suggests corrupted IoVa table
    NN_USB_ABORT_UPON_ERROR(m_IoVaManager.Free(summary.ioVa));

    NN_USB_HS_LOG_SMMU("SmmuMapManager::UnmapByIoVa(ioVa=%p) = %d:%d\n",
                       (void*)ioVa,  result.GetModule(), result.GetDescription());

    return result;
}

nn::dd::DeviceVirtualAddress SmmuMapManager::GetIoVa(nn::dd::ProcessHandle procHandle, uint64_t procVa)
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);
    Result result;
    nn::dd::DeviceVirtualAddress ioVa = 0;
    if((result=m_IoVaManager.GetIoVa(0, procHandle, procVa, &ioVa)).IsFailure())
    {
        NN_USB_LOG_WARN("SmmuMapManager::GetIoVa(0x%llx) failed with status %d:%d.\n",
                        ioVa, result.GetModule(), result.GetDescription());
    }
    return ioVa;
}

nn::dd::DeviceVirtualAddress SmmuMapManager::GetIoVa(void *pBuffer)
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);
    nn::dd::ProcessHandle procHandle = nn::dd::GetCurrentProcessHandle();
    uint64_t procVa = reinterpret_cast<uint64_t>(pBuffer);

    return GetIoVa(procHandle, procVa);
}

void* SmmuMapManager::AllocMemory(nn::dd::DeviceVirtualAddress *pOutIoVa, size_t size, const char *objectName)
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);
    size_t roundedUpSize = NN_USB_ROUNDUP_SIZE(size, HwLimitDmaBufferAlignmentSize);
    void* pBuffer = UsbMemoryAllocAligned(roundedUpSize,
                                          HwLimitDmaBufferAlignmentSize,
                                          objectName);

    if (pBuffer == nullptr)
    {
        return nullptr;
    }

    Result result = Map(pOutIoVa, pBuffer, roundedUpSize);
    if (result.IsFailure())
    {
        NN_USB_LOG_WARN("SmmuMapManager::AllocMemory(%d, %s) failed with status %d:%d.\n",
                        size, objectName, result.GetModule(), result.GetDescription());
        UsbMemoryFree(pBuffer, objectName);

        return nullptr;
    }

    return pBuffer;
}

void* SmmuMapManager::CallocMemory(nn::dd::DeviceVirtualAddress *pOutIoVa, size_t size, const char *objectName)
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);
    size_t roundedUpSize = NN_USB_ROUNDUP_SIZE(size, HwLimitDmaBufferAlignmentSize);
    void* pBuffer = UsbMemoryCallocAligned(roundedUpSize,
                                           HwLimitDmaBufferAlignmentSize,
                                           objectName);

    if (pBuffer == nullptr)
    {
        return nullptr;
    }

    Result result = Map(pOutIoVa, pBuffer, roundedUpSize);
    if (result.IsFailure())
    {
        NN_USB_LOG_WARN("SmmuMapManager::CallocMemory(%d, %s) failed with status %d:%d.\n",
                        size, objectName, result.GetModule(), result.GetDescription());
        UsbMemoryFree(pBuffer, objectName);

        return nullptr;
    }

    nn::dd::FlushDataCache(pBuffer, roundedUpSize);

    return pBuffer;
}

Result SmmuMapManager::FreeMemory(void *pBuffer, const char *objectName)
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);
    Result result;
    if((result = Unmap(pBuffer)).IsSuccess())
    {
        UsbMemoryFree(pBuffer, objectName);
    }
    return result;
}

Result SmmuMapManager::FreeMemory(void *pBuffer, size_t size, const char *objectName)
{
    NN_UNUSED(size);
    return FreeMemory(pBuffer, objectName);
}

} // end of namespace detail
} // end of namespace usb0
} // end of namespace nn
