﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

/**
 * @file    usb_Platform.h
 * @brief   Hardware Platform Abstraction
 */

#include <nn/os.h>
#include <nn/usb/usb_Types.h>
#include <nn/usb/usb_Result.h>
#include <nn/usb/usb_Limits.h>

#include "usb_Complex.h"
#include "usb_Memory.h"
#include "usb_Util.h"

namespace nn {
namespace usb {
namespace detail {

class UsbPlatform
{
public:
    enum SocName {
        SocName_Tx1,
        SocName_Mariko,
    };

    struct Config {
        SocName   socName;
        bool      isRootPortPowerControlled;
        bool      isUsb30Enabled;
        int32_t   hsCurrLevelOffset;
        uint32_t  portCount;

        struct {
            uint32_t   capability;
            uint32_t   hsLane;
            uint32_t   ssLane;
        } port[HwLimitMaxPortCount];
    };

public:
    UsbPlatform() NN_NOEXCEPT
        : m_ComplexListMutex(false)
        , m_BreakEvent(nn::os::EventClearMode_ManualClear)
        , m_ContinueEvent(nn::os::EventClearMode_ManualClear)
        , m_IsrMutex(false)
        , m_IsrSemaphore(0, 1)
        , m_pRequestHolder(nullptr)
        , m_pFreeHolder(nullptr)

    {
        // do nothing
    }

    ~UsbPlatform() NN_NOEXCEPT
    {
        auto itor = m_ComplexList.begin();

        while (itor != m_ComplexList.end())
        {
            DelComplex(*itor++);
        }
    }

    virtual Result      Initialize() NN_NOEXCEPT;
    virtual Result      Finalize() NN_NOEXCEPT;

    const Config& GetConfig() const NN_NOEXCEPT;

    UsbComplex *GetComplex(ComplexId complexId) NN_NOEXCEPT;

    Result      RequestIrq(nn::os::MultiWaitHolderType *holder,
                           IrqHandlerType *handler) NN_NOEXCEPT;
    Result      FreeIrq(nn::os::MultiWaitHolderType *holder) NN_NOEXCEPT;

protected:
    Result      AddComplex(UsbComplex *pComplex) NN_NOEXCEPT;
    Result      DelComplex(UsbComplex *pComplex) NN_NOEXCEPT;

private:
    static void IsrThreadEntry(void *arg) NN_NOEXCEPT
    {
        UsbPlatform *pThis = reinterpret_cast<UsbPlatform*>(arg);

        pThis->Isr();
    }

    void Isr() NN_NOEXCEPT;

    void     LoadPlatformConfig() NN_NOEXCEPT;
    SocName  GetSocName() const NN_NOEXCEPT;
    bool     IsRootPortPowerControlled() const NN_NOEXCEPT;
    bool     IsUsb30Enabled() const NN_NOEXCEPT;
    int32_t  GetDriveStrengthOffset() const NN_NOEXCEPT;
    uint32_t GetPortCount() const NN_NOEXCEPT;
    uint32_t GetRoleCapability(uint8_t portNumber) const NN_NOEXCEPT;
    uint32_t GetSpeedCapability(uint8_t portNumber) const NN_NOEXCEPT;
    uint32_t GetHsLane(uint8_t portNumber) const NN_NOEXCEPT;
    uint32_t GetSsLane(uint8_t portNumber) const NN_NOEXCEPT;

private:
    StdList<UsbComplex*>               m_ComplexList;
    nn::os::Mutex                      m_ComplexListMutex;

    // ISR thread
    nn::os::ThreadType                 m_IsrThread;
    NN_OS_ALIGNAS_THREAD_STACK uint8_t m_IsrThreadStack[UsbLimitIsrThreadStackSize];

    // ISR thread control
    nn::os::MultiWaitType              m_MultiWait;
    nn::os::Event                      m_BreakEvent;
    nn::os::MultiWaitHolderType        m_BreakHolder;
    nn::os::Event                      m_ContinueEvent;
    nn::os::MultiWaitHolderType        m_ContinueHolder;
    nn::os::Mutex                      m_IsrMutex;
    nn::os::Semaphore                  m_IsrSemaphore;
    nn::os::MultiWaitHolderType       *m_pRequestHolder;
    nn::os::MultiWaitHolderType       *m_pFreeHolder;

    // Platform Config
    Config                             m_Config;
};

} // end of namespace detail
} // end of namespace usb
} // end of namespace nn
