﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <cstring>

#include <nn/os.h>
#include <nn/nn_Abort.h>
#include <nn/nn_SdkLog.h>

#include <nn/lmem/lmem_Common.h>
#include <nn/lmem/lmem_ExpHeap.h>

#include <nn/sf/sf_ExpHeapAllocator.h>

#include "usb_Memory.h"

#if 1

#define NN_USB_TRACE_ALLOC(...)
#define NN_USB_TRACE_FREE(...)
#define NN_USB_HEAP_CHECK()

#else

#define NN_USB_ENABLE_HEAP_CHECK

#define NN_USB_TRACE_ALLOC(size, align, objectName) \
    NN_SDK_LOG("#### Alloc size %d, align %d, %s\n",size,align,objectName)

#define NN_USB_TRACE_FREE(objectName) \
    NN_SDK_LOG("#### Free size  %s\n",objectName)

#define NN_USB_HEAP_CHECK() NN_ABORT_UNLESS(lmem::CheckExpHeap(s_HeapHandle, 0))

#endif

namespace nn { namespace usb { namespace detail {

static std::aligned_storage<2 * 1024 * 1024>::type   s_HeapMemory;
static nn::lmem::HeapHandle                          s_HeapHandle;
static nn::sf::ExpHeapAllocator                      s_HeapAllocator;

void UsbMemoryInit()
{
    s_HeapHandle = lmem::CreateExpHeap(&s_HeapMemory,
                                       sizeof(s_HeapMemory),
                                       nn::lmem::CreationOption_ThreadSafe);
    s_HeapAllocator.Attach(s_HeapHandle);
}

void UsbMemoryFinalize()
{
    s_HeapAllocator.Detach();
    lmem::DestroyExpHeap(s_HeapHandle);
    s_HeapHandle = NULL;
}

nn::sf::ExpHeapAllocator* UsbMemoryGetAllocator()
{
    return &s_HeapAllocator;
}

void* UsbMemoryAlloc(size_t size)
{
    NN_USB_HEAP_CHECK();
    void *ptr = lmem::AllocateFromExpHeap(s_HeapHandle, size);

    NN_SDK_ASSERT_NOT_NULL(ptr);

    return ptr;
}

void* UsbMemoryAlloc(size_t size, const char *objectName)
{
    NN_USB_TRACE_ALLOC(size, 1, objectName);
    return UsbMemoryAlloc(size);
}

void* UsbMemoryAllocAligned(size_t size, size_t alignment)
{
    NN_USB_HEAP_CHECK();
    void *ptr = lmem::AllocateFromExpHeap(s_HeapHandle, size, alignment);

    NN_SDK_ASSERT_NOT_NULL(ptr);

    return ptr;
}

void* UsbMemoryAllocAligned(size_t size, size_t alignment, const char *objectName)
{
    NN_USB_TRACE_ALLOC(size, alignment, objectName);
    return UsbMemoryAllocAligned(size, alignment);
}

void* UsbMemoryCalloc(size_t size, const char *objectName)
{
    void *pMem = UsbMemoryAlloc(size, objectName);
    if (pMem)
    {
        std::memset(pMem, 0, size);
    }
    return pMem;
}

void* UsbMemoryCallocAligned(size_t size, size_t alignment, const char *objectName)
{
    void *pMem = UsbMemoryAllocAligned(size, alignment, objectName);
    if (pMem)
    {
        std::memset(pMem, 0, size);
    }
    return pMem;
}

void* UsbMemoryRealloc(void* addr, size_t newSize)
{
    if (addr == nullptr)
    {
        return UsbMemoryAlloc(newSize);
    }

    if (newSize == 0)
    {
        UsbMemoryFree(addr);
        return nullptr;
    }

    void *pMem = UsbMemoryAlloc(newSize);
    if (pMem)
    {
        size_t oldSize = lmem::GetExpHeapBlockSize(addr);
        std::memcpy(pMem, addr, oldSize > newSize ? newSize : oldSize);
        UsbMemoryFree(addr);
    }

    return pMem;
}

void* UsbMemoryRealloc(void* addr, size_t newSize, const char *objectname)
{
    if (addr == nullptr)
    {
        return UsbMemoryAlloc(newSize, objectname);
    }

    if (newSize == 0)
    {
        UsbMemoryFree(addr);
        return nullptr;
    }

    void *pMem = UsbMemoryAlloc(newSize, objectname);
    if (pMem)
    {
        size_t oldSize = lmem::GetExpHeapBlockSize(addr);
        std::memcpy(pMem, addr, oldSize > newSize ? newSize : oldSize);
        UsbMemoryFree(addr);
    }

    return pMem;
}

void* UsbMemoryReallocAligned(void* addr, size_t newSize, size_t alignment)
{
    if (addr == nullptr)
    {
        return UsbMemoryAllocAligned(newSize, alignment);
    }

    if (newSize == 0)
    {
        UsbMemoryFree(addr);
        return nullptr;
    }

    void *pMem = UsbMemoryAllocAligned(newSize, alignment);
    if (pMem)
    {
        size_t oldSize = lmem::GetExpHeapBlockSize(addr);
        std::memcpy(pMem, addr, oldSize > newSize ? newSize : oldSize);
        UsbMemoryFree(addr);
    }

    return pMem;
}

void* UsbMemoryReallocAligned(void* addr, size_t newSize, size_t alignment, const char *objectname)
{
    if (addr == nullptr)
    {
        return UsbMemoryAllocAligned(newSize, alignment, objectname);
    }

    if (newSize == 0)
    {
        UsbMemoryFree(addr);
        return nullptr;
    }

    void *pMem = UsbMemoryAllocAligned(newSize, alignment, objectname);
    if (pMem)
    {
        size_t oldSize = lmem::GetExpHeapBlockSize(addr);
        std::memcpy(pMem, addr, oldSize > newSize ? newSize : oldSize);
        UsbMemoryFree(addr);
    }

    return pMem;
}

void UsbMemoryFree(void *pMem)
{
    NN_USB_HEAP_CHECK();

    lmem::FreeToExpHeap(s_HeapHandle, pMem);
}

void UsbMemoryFree(void *pMem, const char *objectName)
{
    NN_USB_TRACE_FREE(objectName);
    UsbMemoryFree(pMem);
}

size_t UsbMemoryGetTotalFreeSize()
{
    size_t totalFreeSize = 0;
    if (s_HeapHandle != NULL)
    {
        totalFreeSize = lmem::GetExpHeapTotalFreeSize(s_HeapHandle);
    }
    return totalFreeSize;
}


} // end of namespace detail
} // end of namespace usb
} // end of namespace nn
