﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

/**
 * @file    usb_Complex.h
 * @brief   USB Complex Abstraction
 */

#include <functional>

#include <nn/os.h>
#include <nn/usb/usb_Result.h>
#include <nn/usb/usb_Types.h>
#include <nn/usb/usb_PmTypes.h>

#include "usb_Util.h"
#include "usb_Memory.h"
#include "usb_Controller.h"

namespace nn {
namespace usb {
namespace detail {

typedef std::function<void()> IrqHandlerType;

class UsbPlatform;

class UsbComplex
{
public:
    const nn::usb::ComplexId     id;

public:
    explicit UsbComplex(nn::usb::ComplexId complexId) NN_NOEXCEPT
        : id(complexId)
        , m_pPlatform(nullptr)
        , m_ResetEvent(nn::os::EventClearMode_ManualClear)
    {
        // do nothing
    }

    virtual ~UsbComplex() NN_NOEXCEPT
    {
        // do nothing
    }

    virtual Result Initialize(UsbPlatform *pPlatform) NN_NOEXCEPT;
    virtual Result Finalize() NN_NOEXCEPT;
    virtual Result Enable() NN_NOEXCEPT = 0;
    virtual Result Disable() NN_NOEXCEPT = 0;

    virtual Result CreatePort(uint32_t hsLane, uint32_t ssLane, uint32_t capability) NN_NOEXCEPT = 0;
    virtual Result DestroyPort(uint32_t hsLane, uint32_t ssLane, uint32_t capability) NN_NOEXCEPT = 0;

    virtual UsbChargerType ChargerDetection() NN_NOEXCEPT = 0;

    Result         AddController(UsbController *pController,
                                 UsbController::Config* pConfig) NN_NOEXCEPT;
    Result         DelController(UsbController *pController) NN_NOEXCEPT;
    UsbController *GetController(uint32_t capability) NN_NOEXCEPT;
    UsbPlatform   *GetPlatform() const NN_NOEXCEPT;

    nn::os::Event *GetResetEvent() NN_NOEXCEPT;
    void           RequestReset() NN_NOEXCEPT;

protected:
    UsbPlatform                       *m_pPlatform;
    StdList<UsbController*>            m_ControllerList;

private:
    nn::os::Event                      m_ResetEvent;
};

} // end of namespace detail
} // end of namespace usb
} // end of namespace nn
