﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/fs.h>
#include <nn/fs/fs_Bis.h>
#include <nn/fs/fs_IStorage.h>

namespace nn { namespace updater {

template <typename T> struct OffsetSize
{
    T type;
    uint64_t offset;
    size_t size;
};

class BisAccessor
{
    NN_DISALLOW_COPY(BisAccessor);

public:
    Result Initialize() NN_NOEXCEPT;
    void Finalize() NN_NOEXCEPT;
    static int GetBlockSize() NN_NOEXCEPT { return 512; } // hard-coded..

protected:
    explicit BisAccessor(fs::BisPartitionId id) NN_NOEXCEPT;

    Result Read(void* pOutBuffer, size_t bufferSize, uint64_t address) NN_NOEXCEPT;
    Result Write(uint64_t address, const void* pData, size_t dataSize) NN_NOEXCEPT;
    Result Write(uint64_t address, size_t size, const char* filename, void* workBuffer, size_t bufferSize) NN_NOEXCEPT;
    Result Invalidate(uint64_t address, size_t size, void* workBuffer, size_t bufferSize) NN_NOEXCEPT;

    Result CalculateHash(void* pOutHash, size_t hashSize, uint64_t address, size_t size, size_t hashRegionSize, void* workBuffer, size_t bufferSize) NN_NOEXCEPT;

private:

    fs::BisPartitionId m_PartitionId;
    std::unique_ptr<fs::IStorage> m_Storage;
};


template <typename TYPE> class TAccessor : public BisAccessor
{
    NN_DISALLOW_COPY(TAccessor);

public:
    TAccessor(fs::BisPartitionId id, const OffsetSize<TYPE>* tables, int numTables) NN_NOEXCEPT:
        BisAccessor(id),
        m_Tables(tables),
        m_NumTables(numTables)
    {
    }
    Result Read(void* pOutBuffer, size_t* pOutSize, size_t bufferSize, TYPE type) NN_NOEXCEPT
    {
        auto table = Find(type);

        NN_ABORT_UNLESS(bufferSize >= table->size, "lack of buffer");
        NN_RESULT_DO(BisAccessor::Read(pOutBuffer, table->size, table->offset));
        *pOutSize = table->size;

        NN_RESULT_SUCCESS;
    }
    Result Write(const void* pData, size_t dataSize, TYPE type) NN_NOEXCEPT
    {
        auto table = Find(type);

        NN_ABORT_UNLESS(dataSize <= table->size, "too large data");
        NN_ABORT_UNLESS(dataSize % GetBlockSize() == 0, "unaligned data");

        NN_RESULT_DO(BisAccessor::Write(table->offset, pData, dataSize));

        NN_RESULT_SUCCESS;
    }
    Result Write(const char* filename, void* buffer, size_t bufferSize, TYPE type) NN_NOEXCEPT
    {
        auto table = Find(type);

        NN_RESULT_DO(BisAccessor::Write(table->offset, table->size, filename, buffer, bufferSize));

        NN_RESULT_SUCCESS;
    }
    Result Invalidate(void* workBuffer, size_t bufferSize, TYPE type) NN_NOEXCEPT
    {
        auto table = Find(type);

        NN_RESULT_DO(BisAccessor::Invalidate(table->offset, table->size, workBuffer, bufferSize));

        NN_RESULT_SUCCESS;
    }
    Result CalculateHash(void* pOutHash, size_t hashSize, size_t hashRegionSize, void* workBuffer, size_t bufferSize, TYPE type) NN_NOEXCEPT
    {
        auto table = Find(type);
        NN_ABORT_UNLESS(hashRegionSize <= table->size, "too large hash region");

        NN_RESULT_DO(BisAccessor::CalculateHash(pOutHash, hashSize, table->offset, table->size, hashRegionSize, workBuffer, bufferSize));

        NN_RESULT_SUCCESS;
    }

private:
    const OffsetSize<TYPE>* Find(TYPE type) NN_NOEXCEPT
    {
        for (int i = 0; i < m_NumTables; ++i)
        {
            if(m_Tables[i].type == type)
            {
                return &(m_Tables[i]);
            }
        }
        NN_ABORT("Invalid type %d", type);
        return nullptr;
    }
    const OffsetSize<TYPE>* m_Tables;
    int              m_NumTables;
};

}}
