﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>
#include <algorithm>

#include <nn/result/result_HandlingUtility.h>
#include <nn/nn_Abort.h>
#include <nn/nn_SdkAssert.h>

#include <nn/fs.h>

#include <nn/crypto/crypto_Sha256Generator.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/util/util_BitUtil.h>

#include <nn/updater/updater_Result.h>
#include "updater_BisAccessor.h"

namespace nn { namespace updater {

BisAccessor::BisAccessor(fs::BisPartitionId id) NN_NOEXCEPT:
    m_PartitionId(id)
{
}

Result BisAccessor::Initialize() NN_NOEXCEPT
{
    NN_RESULT_DO(fs::OpenBisPartition(&m_Storage, m_PartitionId));
    NN_RESULT_SUCCESS;
}
void BisAccessor::Finalize() NN_NOEXCEPT
{
    // nothing to do
}

Result BisAccessor::Read(void* pOutBuffer, size_t bufferSize, uint64_t address) NN_NOEXCEPT
{
    NN_SDK_ASSERT(
        util::is_aligned(address, GetBlockSize()),
        "Invalid Alignment address"
    );
    NN_RESULT_DO(
        m_Storage->Read(static_cast<uint32_t>(address), pOutBuffer, bufferSize)
    );
    NN_RESULT_SUCCESS;
}

Result BisAccessor::Write(uint64_t address, const void* pData, size_t dataSize) NN_NOEXCEPT
{
    NN_SDK_ASSERT(
        util::is_aligned(address, GetBlockSize()),
        "Invalid Alignment address"
    );
    NN_RESULT_DO(
        m_Storage->Write(static_cast<uint32_t>(address), pData, dataSize)
    );
    NN_RESULT_SUCCESS;
}
Result BisAccessor::Write(uint64_t address, size_t size, const char* filename, void* workBuffer, size_t bufferSize) NN_NOEXCEPT
{
    NN_SDK_ASSERT(
        util::is_aligned(address, GetBlockSize()),
        "Invalid Alignment address"
    );
    NN_SDK_ASSERT(
        util::is_aligned(bufferSize, GetBlockSize()),
        "Inavlid Alignment buffer size"
    );

    fs::FileHandle handle;

    NN_RESULT_TRY(fs::OpenFile(&handle, filename, fs::OpenMode_Read))
        NN_RESULT_CATCH(fs::ResultPathNotFound) {NN_RESULT_THROW(updater::ResultInvalidBootImagePackage());}
    NN_RESULT_END_TRY
    NN_UTIL_SCOPE_EXIT{ fs::CloseFile(handle); };

    size_t writtenSize = 0;
    for(;;)
    {
        // TODO: 毎回 0 クリアではなく、最後だけ 0 クリアにすると若干早い
        std::memset(workBuffer, 0, bufferSize);
        size_t readSize;
        NN_RESULT_DO(fs::ReadFile(&readSize, handle, writtenSize, workBuffer, bufferSize, fs::ReadOption()));
        NN_ABORT_UNLESS(writtenSize + readSize <= size, "Too large size data");

        size_t alignedReadSize = util::align_up(readSize, GetBlockSize());
        NN_RESULT_DO(Write(address + writtenSize, workBuffer, alignedReadSize));

        if(readSize != bufferSize)
        {
            break;
        }
        writtenSize += readSize;
    }
    NN_RESULT_SUCCESS;
}
Result BisAccessor::Invalidate(uint64_t address, size_t size, void* workBuffer, size_t bufferSize) NN_NOEXCEPT
{
    NN_SDK_ASSERT(
        util::is_aligned(address, GetBlockSize()),
        "Invalid Alignment address"
    );
    NN_SDK_ASSERT(
        util::is_aligned(bufferSize, GetBlockSize()),
        "Inavlid Alignment buffer size"
    );
    std::memset(workBuffer, 0, bufferSize);

    size_t writtenSize = 0;
    while(writtenSize < size)
    {
        size_t writeSize = std::min(bufferSize, size - writtenSize);
        NN_RESULT_DO(Write(address + writtenSize, workBuffer, writeSize));
        writtenSize += writeSize;
    }
    NN_RESULT_SUCCESS;
}

Result BisAccessor::CalculateHash(void* pOutHash, size_t hashSize, uint64_t address, size_t size, size_t hashRegionSize, void* workBuffer, size_t bufferSize) NN_NOEXCEPT
{
    NN_UNUSED(hashSize);
    NN_SDK_ASSERT(
        util::is_aligned(address, GetBlockSize()),
        "Invalid Alignment address"
    );
    NN_SDK_ASSERT(
        util::is_aligned(bufferSize, GetBlockSize()),
        "Inavlid Alignment buffer size"
    );

    size_t readSize = 0;
    crypto::Sha256Generator generator;
    generator.Initialize();
    while(readSize < hashRegionSize)
    {
        size_t tryReadSize = std::min(bufferSize, size - readSize);
        size_t tryUpdateSize = std::min(tryReadSize, hashRegionSize - readSize);
        NN_RESULT_DO(Read(workBuffer, tryReadSize, address + readSize));
        generator.Update(workBuffer, tryUpdateSize);

        readSize += tryReadSize;
    }
    generator.GetHash(pOutHash, hashSize);

    NN_RESULT_SUCCESS;
}


}}
