﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdio>
#include <cstdlib>
#include <cstring>

#include <nn/nn_SdkAssert.h>
#include <nn/result/result_HandlingUtility.h>

#include <nn/util/util_ScopeExit.h>
#include <nn/util/util_BitUtil.h>
#include <nn/crypto/crypto_Sha256Generator.h>

#include <nn/fs.h>
#include <nn/fs/fs_SystemData.h>

#include <nn/updater/updater.h>
#include <nn/ncm/ncm_Service.h>
#include <nn/ncm/ncm_ContentMetaDatabase.h>

#include "updater_BisSave.h"
#include "updater_Common.h"
#include "updater_PartitionAccessor.h"
#include "updater_UpdateTypeResolver.h"

namespace nn { namespace updater {

namespace
{
Result ReadPackage2AndCalculateHash(void* pOutHash, size_t hashSize, size_t hashRegionSize, void* workBuffer, size_t workSize, Package2Index index) NN_NOEXCEPT
{
    Package2Accessor accessor(index);
    NN_RESULT_DO(accessor.Initialize());
    NN_UTIL_SCOPE_EXIT{ accessor.Finalize(); };
    NN_RESULT_DO(accessor.CalculateHash(pOutHash, hashSize, hashRegionSize, workBuffer, workSize, Package2Type::Package2));

    NN_RESULT_SUCCESS;
}

Result WritePackage2(void* workBuffer, size_t workSize, Package2Index index, BootImageUpdateType bootImageUpdateType) NN_NOEXCEPT
{
    Package2Accessor accessor(index);
    NN_RESULT_DO(accessor.Initialize());
    NN_UTIL_SCOPE_EXIT{ accessor.Finalize(); };
    NN_RESULT_DO(accessor.Invalidate(workBuffer, workSize, Package2Type::Package2));
    NN_RESULT_DO(accessor.Write(ResolvePackage2Path(bootImageUpdateType), workBuffer, workSize, Package2Type::Package2));

    NN_RESULT_SUCCESS;
}

Result CompareHash(const void* hashA, const void* hashB, size_t hashSize) NN_NOEXCEPT
{
    if (std::memcmp(hashA, hashB, hashSize) != 0)
    {
        NN_RESULT_THROW(ResultRequiresUpdateBootImages());
    }

    NN_RESULT_SUCCESS;
}

Result SaveVerifyingRequired(TargetBootMode mode, void* workBuffer, size_t workSize, bool verifyingRequired) NN_NOEXCEPT
{
    NN_RESULT_DO(VerifyBuffer(workBuffer, workSize));

    BisSave save;
    NN_RESULT_DO(save.Initialize(workBuffer, workSize));
    NN_UTIL_SCOPE_EXIT { save.Finalize(); };

    NN_RESULT_DO(save.Load());
    save.SetVerifyingRequiredFlag(mode, verifyingRequired);
    NN_RESULT_DO(save.Save());

    NN_RESULT_SUCCESS;
}

ncm::ContentMetaType GetContentMetaType(TargetBootMode mode) NN_NOEXCEPT
{
    switch(mode)
    {
    case TargetBootMode::Normal:
        return ncm::ContentMetaType::BootImagePackage;
        break;
    case TargetBootMode::Safe:
        return ncm::ContentMetaType::BootImagePackageSafe;
        break;
    default:
        NN_ABORT("Unexpected mode to update\n");
        break;
    }
    return ncm::ContentMetaType::Unknown;
}

Result VerifyBootImagesAndUpdateIfNeededDetail(bool* pOutUpdated, TargetBootMode mode, void* workBuffer, size_t workSize, BootImageUpdateType bootImageUpdateType) NN_NOEXCEPT
{
    ncm::SystemDataId id;
    NN_RESULT_DO(GetBootImagePackageId(&id, mode, workBuffer, workSize));

    auto result = VerifyBootImages(id, mode, workBuffer, workSize, bootImageUpdateType);
    if (result.IsSuccess())
    {
        NN_RESULT_DO(MarkVerified(mode, workBuffer, workSize));
    }
    else if (ResultRequiresUpdateBootImages::Includes(result))
    {
        // 更新が必要
        *pOutUpdated = true;
        NN_RESULT_DO(UpdateBootImagesFromPackage(id, mode, workBuffer, workSize, bootImageUpdateType));
        NN_RESULT_DO(MarkVerified(mode, workBuffer, workSize));
    }
    else
    {
        NN_RESULT_THROW(result);
    }
    NN_RESULT_SUCCESS;
}


}

/**
 * 更新順は下記のとおり
 * - package1 normal sub
 * - package2 normal sub
 * - bct normal sub
 * - bct normal main
 * - package2 normal main
 * - package1 normal main
 * bct は
 * 0x0210 - 0x030F 署名検証鍵 Modulus
 * 0x0320 - 0x041F 署名
 * 0x0510 - 0x287F 署名対象
 * というフォーマットなので、「0 クリア」->「4KB (0x1000)ずつ書き込み」であれば
 * 中途半端な状態で検証が通るということはない
 * package1, package2 はサイズが変わることがあるので、先に全域 0 クリアしている
 */
Result UpdateBootImagesFromPackage(ncm::SystemDataId dataId, void* workBuffer, size_t workSize, BootImageUpdateType bootImageUpdateType) NN_NOEXCEPT
{
    // ワークバッファの確認
    NN_RESULT_DO(VerifyBuffer(workBuffer, workSize));

    NN_RESULT_TRY(fs::MountSystemData(GetMountName(), dataId))
        NN_RESULT_CATCH(fs::ResultTargetNotFound) {NN_RESULT_THROW(ResultBootImagePackageNotFound());}
    NN_RESULT_END_TRY
    NN_UTIL_SCOPE_EXIT{ fs::Unmount(GetMountName()); };

    {
        BootPartition1Accessor accessor;
        NN_RESULT_DO(accessor.Initialize());
        NN_UTIL_SCOPE_EXIT{ accessor.Finalize(); };

        NN_RESULT_DO(accessor.Invalidate(workBuffer, workSize, BootPartition1Type::Package1NormalSub));
        NN_RESULT_DO(accessor.Write(ResolvePackage1Path(bootImageUpdateType), workBuffer, workSize, BootPartition1Type::Package1NormalSub));

        NN_RESULT_DO(WritePackage2(workBuffer, workSize, Package2Index::NormalSub, bootImageUpdateType));

        {
            // このスコープ内でのみ、ワークバッファを bct 用バッファとして扱う
            BctBuffer* ptr = reinterpret_cast<BctBuffer*>(workBuffer);

            size_t size;
            NN_RESULT_DO(ReadFile(&size, ptr->bct, BctSize, ResolveBctPath(bootImageUpdateType)));
            if (NeedsUpdateSecureInfo(bootImageUpdateType))
            {
                NN_RESULT_DO(accessor.UpdateSecureInfoAuto(ptr->bct, BctSize, ptr->eks, sizeof(ptr->eks)));
            }

            NN_RESULT_DO(accessor.Write(ptr->bct, BctSize, BootPartition1Type::BctNormalSub_BCT2));
            NN_RESULT_DO(accessor.Write(ptr->bct, BctSize, BootPartition1Type::BctNormalMain_BCT0));
        }

        // 更新中の package2 normal main を参照されないよう、先に bl normal main をクリアする
        NN_RESULT_DO(accessor.Invalidate(workBuffer, workSize, BootPartition1Type::Package1NormalMain));

        NN_RESULT_DO(WritePackage2(workBuffer, workSize, Package2Index::NormalMain, bootImageUpdateType));
        NN_RESULT_DO(accessor.Write(ResolvePackage1Path(bootImageUpdateType), workBuffer, workSize, BootPartition1Type::Package1NormalMain));
    }


    NN_RESULT_SUCCESS;
}

Result VerifyBootImages(ncm::SystemDataId dataId, void* workBuffer, size_t workSize, BootImageUpdateType bootImageUpdateType) NN_NOEXCEPT
{
    // ワークバッファの確認
    NN_RESULT_DO(VerifyBuffer(workBuffer, workSize));

    NN_RESULT_TRY(fs::MountSystemData(GetMountName(), dataId))
        NN_RESULT_CATCH(fs::ResultTargetNotFound) {NN_RESULT_THROW(ResultBootImagePackageNotFound());}
    NN_RESULT_END_TRY
    NN_UTIL_SCOPE_EXIT{ fs::Unmount(GetMountName()); };

    {
        char hashFile[crypto::Sha256Generator::HashSize];
        char hashWritten[crypto::Sha256Generator::HashSize];
        size_t size;

        BootPartition1Accessor accessor;
        NN_RESULT_DO(accessor.Initialize());
        NN_UTIL_SCOPE_EXIT{ accessor.Finalize(); };

        // bct のチェック
        {
            // このスコープ内でのみ、ワークバッファを bct 用バッファとして扱う
            BctBuffer* ptr = reinterpret_cast<BctBuffer*>(workBuffer);

            NN_RESULT_DO(ReadFile(&size, ptr->bct, BctSize, ResolveBctPath(bootImageUpdateType)));
            if (NeedsUpdateSecureInfo(bootImageUpdateType))
            {
                NN_RESULT_DO(accessor.UpdateSecureInfoAuto(ptr->bct, BctSize, ptr->eks, sizeof(ptr->eks)));
            }

            crypto::GenerateSha256Hash(hashFile, sizeof(hashFile), ptr->bct, BctSize);
        }
        NN_RESULT_DO(accessor.CalculateHash(hashWritten, sizeof(hashWritten), BctSize, workBuffer, workSize, BootPartition1Type::BctNormalMain_BCT0));
        NN_RESULT_DO(CompareHash(hashFile, hashWritten, sizeof(hashFile)));

        NN_RESULT_DO(accessor.CalculateHash(hashWritten, sizeof(hashWritten), BctSize, workBuffer, workSize, BootPartition1Type::BctNormalSub_BCT2));
        NN_RESULT_DO(CompareHash(hashFile, hashWritten, sizeof(hashFile)));

        // package1 のチェック
        NN_RESULT_DO(ReadFileAndCalculateHash(&size, hashFile, sizeof(hashFile), ResolvePackage1Path(bootImageUpdateType), workBuffer, workSize));

        NN_RESULT_DO(accessor.CalculateHash(hashWritten, sizeof(hashWritten), size, workBuffer, workSize, BootPartition1Type::Package1NormalMain));
        NN_RESULT_DO(CompareHash(hashFile, hashWritten, sizeof(hashFile)));

        NN_RESULT_DO(accessor.CalculateHash(hashWritten, sizeof(hashWritten), size, workBuffer, workSize, BootPartition1Type::Package1NormalSub));
        NN_RESULT_DO(CompareHash(hashFile, hashWritten, sizeof(hashFile)));

        // pakcage2 のチェック
        NN_RESULT_DO(ReadFileAndCalculateHash(&size, hashFile, sizeof(hashFile), ResolvePackage2Path(bootImageUpdateType), workBuffer, workSize));

        NN_RESULT_DO(ReadPackage2AndCalculateHash(hashWritten, sizeof(hashWritten), size, workBuffer, workSize, Package2Index::NormalMain));
        NN_RESULT_DO(CompareHash(hashFile, hashWritten, sizeof(hashFile)));

        NN_RESULT_DO(ReadPackage2AndCalculateHash(hashWritten, sizeof(hashWritten), size, workBuffer, workSize, Package2Index::NormalSub));
        NN_RESULT_DO(CompareHash(hashFile, hashWritten, sizeof(hashFile)));
    }

    NN_RESULT_SUCCESS;
}

/**
 * 更新順は下記のとおり
 * - package1 safe sub
 * - package2 safe sub
 * - bct safe sub
 * - bct safe main
 * - package2 safe main
 * - package1 safe main
 */
Result UpdateBootImagesForSafeFromPackage(ncm::SystemDataId dataId, void* workBuffer, size_t workSize, BootImageUpdateType bootImageUpdateType) NN_NOEXCEPT
{
    // ワークバッファの確認
    NN_RESULT_DO(VerifyBuffer(workBuffer, workSize));

    NN_RESULT_TRY(fs::MountSystemData(GetMountName(), dataId))
        NN_RESULT_CATCH(fs::ResultTargetNotFound) {NN_RESULT_THROW(ResultBootImagePackageNotFound());}
    NN_RESULT_END_TRY
    NN_UTIL_SCOPE_EXIT{ fs::Unmount(GetMountName()); };

    {
        BootPartition1Accessor accessor1;
        BootPartition2Accessor accessor2;
        NN_RESULT_DO(accessor1.Initialize());
        NN_UTIL_SCOPE_EXIT{ accessor1.Finalize(); };
        NN_RESULT_DO(accessor2.Initialize());
        NN_UTIL_SCOPE_EXIT{ accessor2.Finalize(); };

        NN_RESULT_DO(accessor2.Invalidate(workBuffer, workSize, BootPartition2Type::Package1SafeSub));
        NN_RESULT_DO(accessor2.Write(ResolvePackage1Path(bootImageUpdateType), workBuffer, workSize, BootPartition2Type::Package1SafeSub));

        NN_RESULT_DO(WritePackage2(workBuffer, workSize, Package2Index::SafeSub, bootImageUpdateType));

        {
            // このスコープ内でのみ、ワークバッファを bct 用バッファとして扱う
            BctBuffer* ptr = reinterpret_cast<BctBuffer*>(workBuffer);

            size_t size;
            NN_RESULT_DO(ReadFile(&size, ptr->bct, BctSize, ResolveBctPath(bootImageUpdateType)));
            if (NeedsUpdateSecureInfo(bootImageUpdateType))
            {
                NN_RESULT_DO(accessor1.UpdateSecureInfoAuto(ptr->bct, BctSize, ptr->eks, sizeof(ptr->eks)));
            }

            NN_RESULT_DO(accessor1.Write(ptr->bct, BctSize, BootPartition1Type::BctSafeSub_BCT3));
            NN_RESULT_DO(accessor1.Write(ptr->bct, BctSize, BootPartition1Type::BctSafeMain_BCT1));
        }

        // 更新中の package2 safe main を参照されないよう、先に bl safe main をクリアする
        NN_RESULT_DO(accessor2.Invalidate(workBuffer, workSize, BootPartition2Type::Package1SafeMain));

        NN_RESULT_DO(WritePackage2(workBuffer, workSize, Package2Index::SafeMain, bootImageUpdateType));
        NN_RESULT_DO(accessor2.Write(ResolvePackage1Path(bootImageUpdateType), workBuffer, workSize, BootPartition2Type::Package1SafeMain));
    }

    NN_RESULT_SUCCESS;
}

Result VerifyBootImagesForSafe(ncm::SystemDataId dataId, void* workBuffer, size_t workSize, BootImageUpdateType bootImageUpdateType) NN_NOEXCEPT
{
    // ワークバッファの確認
    NN_RESULT_DO(VerifyBuffer(workBuffer, workSize));

    NN_RESULT_TRY(fs::MountSystemData(GetMountName(), dataId))
        NN_RESULT_CATCH(fs::ResultTargetNotFound) {NN_RESULT_THROW(ResultBootImagePackageNotFound());}
    NN_RESULT_END_TRY
    NN_UTIL_SCOPE_EXIT{ fs::Unmount(GetMountName()); };

    {
        char hashFile[crypto::Sha256Generator::HashSize];
        char hashWritten[crypto::Sha256Generator::HashSize];
        size_t size;

        BootPartition1Accessor accessor1;
        NN_RESULT_DO(accessor1.Initialize());
        NN_UTIL_SCOPE_EXIT{ accessor1.Finalize(); };

        BootPartition2Accessor accessor2;
        NN_RESULT_DO(accessor2.Initialize());
        NN_UTIL_SCOPE_EXIT{ accessor2.Finalize(); };

        // bct のチェック
        {
            // このスコープ内でのみ、ワークバッファを bct 用バッファとして扱う
            BctBuffer* ptr = reinterpret_cast<BctBuffer*>(workBuffer);

            NN_RESULT_DO(ReadFile(&size, ptr->bct, BctSize, ResolveBctPath(bootImageUpdateType)));
            if (NeedsUpdateSecureInfo(bootImageUpdateType))
            {
                NN_RESULT_DO(accessor1.UpdateSecureInfoAuto(ptr->bct, BctSize, ptr->eks, sizeof(ptr->eks)));
            }

            crypto::GenerateSha256Hash(hashFile, sizeof(hashFile), ptr->bct, BctSize);
        }
        NN_RESULT_DO(accessor1.CalculateHash(hashWritten, sizeof(hashWritten), BctSize, workBuffer, workSize, BootPartition1Type::BctSafeMain_BCT1));
        NN_RESULT_DO(CompareHash(hashFile, hashWritten, sizeof(hashFile)));

        NN_RESULT_DO(accessor1.CalculateHash(hashWritten, sizeof(hashWritten), BctSize, workBuffer, workSize, BootPartition1Type::BctSafeSub_BCT3));
        NN_RESULT_DO(CompareHash(hashFile, hashWritten, sizeof(hashFile)));

        // package1 のチェック
        NN_RESULT_DO(ReadFileAndCalculateHash(&size, hashFile, sizeof(hashFile), ResolvePackage1Path(bootImageUpdateType), workBuffer, workSize));

        NN_RESULT_DO(accessor2.CalculateHash(hashWritten, sizeof(hashWritten), size, workBuffer, workSize, BootPartition2Type::Package1SafeMain));
        NN_RESULT_DO(CompareHash(hashFile, hashWritten, sizeof(hashFile)));

        NN_RESULT_DO(accessor2.CalculateHash(hashWritten, sizeof(hashWritten), size, workBuffer, workSize, BootPartition2Type::Package1SafeSub));
        NN_RESULT_DO(CompareHash(hashFile, hashWritten, sizeof(hashFile)));

        // pakcage2 のチェック
        NN_RESULT_DO(ReadFileAndCalculateHash(&size, hashFile, sizeof(hashFile), ResolvePackage2Path(bootImageUpdateType), workBuffer, workSize));

        NN_RESULT_DO(ReadPackage2AndCalculateHash(hashWritten, sizeof(hashWritten), size, workBuffer, workSize, Package2Index::SafeMain));
        NN_RESULT_DO(CompareHash(hashFile, hashWritten, sizeof(hashFile)));

        NN_RESULT_DO(ReadPackage2AndCalculateHash(hashWritten, sizeof(hashWritten), size, workBuffer, workSize, Package2Index::SafeSub));
        NN_RESULT_DO(CompareHash(hashFile, hashWritten, sizeof(hashFile)));
    }

    NN_RESULT_SUCCESS;
}

Result UpdateBootImagesFromPackage(ncm::SystemDataId dataId, TargetBootMode mode, void* workBuffer, size_t workSize, BootImageUpdateType bootImageUpdateType) NN_NOEXCEPT
{
    switch(mode)
    {
    case TargetBootMode::Normal:
        NN_RESULT_DO(UpdateBootImagesFromPackage(dataId, workBuffer, workSize, bootImageUpdateType));
        break;
    case TargetBootMode::Safe:
        NN_RESULT_DO(UpdateBootImagesForSafeFromPackage(dataId, workBuffer, workSize, bootImageUpdateType));
        break;
    default:
        NN_ABORT("Unexpected mode to update\n");
        break;
    }
    NN_RESULT_SUCCESS;
}
Result VerifyBootImages(ncm::SystemDataId dataId, TargetBootMode mode, void* workBuffer, size_t workSize, BootImageUpdateType bootImageUpdateType) NN_NOEXCEPT
{
    switch(mode)
    {
    case TargetBootMode::Normal:
        NN_RESULT_DO(VerifyBootImages(dataId, workBuffer, workSize, bootImageUpdateType));
        break;
    case TargetBootMode::Safe:
        NN_RESULT_DO(VerifyBootImagesForSafe(dataId, workBuffer, workSize, bootImageUpdateType));
        break;
    default:
        NN_ABORT("Unexpected mode to update\n");
        break;
    }
    NN_RESULT_SUCCESS;
}

Result MarkVerifyingRequired(TargetBootMode mode, void* workBuffer, size_t workSize) NN_NOEXCEPT
{
    return SaveVerifyingRequired(mode, workBuffer, workSize, true);
}
Result MarkVerified(TargetBootMode mode, void* workBuffer, size_t workSize) NN_NOEXCEPT
{
    return SaveVerifyingRequired(mode, workBuffer, workSize, false);
}
Result CheckVerifyingRequired(VerifyingRequiredFlags* pOutFlags, void* workBuffer, size_t workSize) NN_NOEXCEPT
{
    // 安全側に倒すために、デフォルトは true
    pOutFlags->normalRequired = true;
    pOutFlags->safeRequired = true;

    NN_RESULT_DO(VerifyBuffer(workBuffer, workSize));

    BisSave save;
    NN_RESULT_DO(save.Initialize(workBuffer, workSize));
    NN_UTIL_SCOPE_EXIT { save.Finalize(); };

    NN_RESULT_DO(save.Load());
    pOutFlags->normalRequired = save.GetVerifyingRequiredFlag(TargetBootMode::Normal);
    pOutFlags->safeRequired = save.GetVerifyingRequiredFlag(TargetBootMode::Safe);

    NN_RESULT_SUCCESS;
}

Result VerifyBootImagesAndUpdateIfNeeded(bool* pOutNormalUpdated, bool* pOutSafeUpdated, void* workBuffer, size_t workSize, BootImageUpdateType bootImageUpdateType) NN_NOEXCEPT
{
    *pOutNormalUpdated = false;
    *pOutSafeUpdated = false;

    NN_RESULT_DO(VerifyBuffer(workBuffer, workSize));

    VerifyingRequiredFlags flags;
    NN_RESULT_DO(CheckVerifyingRequired(&flags, workBuffer, workSize));

    if (flags.normalRequired == false && flags.safeRequired == false)
    {
        // 検証フラグが両方とも立っていなかった
        NN_RESULT_SUCCESS;
    }
    ncm::Initialize();
    NN_UTIL_SCOPE_EXIT{ncm::Finalize();};

    if (flags.normalRequired)
    {
        NN_RESULT_TRY(VerifyBootImagesAndUpdateIfNeededDetail(pOutNormalUpdated, TargetBootMode::Normal, workBuffer, workSize, bootImageUpdateType))
            NN_RESULT_CATCH(ResultBootImagePackageNotFound)
            {
                // 開発中などないこともあるので、何もしない
                // TODO: フラグは立ったままになるがよいか
            }
        NN_RESULT_END_TRY
    }
    if (flags.safeRequired)
    {
        NN_RESULT_TRY(VerifyBootImagesAndUpdateIfNeededDetail(pOutSafeUpdated, TargetBootMode::Safe, workBuffer, workSize, bootImageUpdateType))
            NN_RESULT_CATCH(ResultBootImagePackageNotFound)
            {
                // 開発中などないこともあるので、何もしない
                // TODO: フラグは立ったままになるがよいか
            }
        NN_RESULT_END_TRY
    }
    NN_RESULT_SUCCESS;
}
Result GetBootImagePackageId(ncm::SystemDataId *pOutId, TargetBootMode mode, void* workBuffer, size_t workSize) NN_NOEXCEPT
{
    const int CountNum = 64;
    NN_ABORT_UNLESS(workSize >= sizeof(ncm::ContentMetaKey) * CountNum, "Out of buffer");

    // updater ライブラリ内で db インスタンスを作っていいことは確認済み
    ncm::ContentMetaDatabase db;
    NN_RESULT_DO(ncm::OpenContentMetaDatabase(&db, ncm::StorageId::BuildInSystem));
    ncm::ContentMetaKey* keys = reinterpret_cast<ncm::ContentMetaKey*>(workBuffer);
    ncm::ContentMetaType type = GetContentMetaType(mode);

    auto count = db.ListContentMeta(keys, CountNum , type);
    NN_RESULT_THROW_UNLESS(count.total > 0, ResultBootImagePackageNotFound());
    if (count.total == 1)
    {
        pOutId->value = keys[0].id;
    }
    else
    {
        // ncm が返すリストが、ID が小さいものから順に返すのに頼る
        // まず、exFAT が有効なものの中で ID 最小のものを探す
        for (int i = 0; i < count.total; ++i)
        {
            Bit8 attributes;
            NN_RESULT_DO(db.GetAttributes(&attributes, keys[i]));

            if (attributes & ncm::ContentMetaAttribute_IncludesExFatDriver)
            {
                pOutId->value = keys[i].id;
                NN_RESULT_SUCCESS;
            }
        }

        // 見つからない場合は最初のものを返す
        pOutId->value = keys[0].id;
    }

    NN_RESULT_SUCCESS;
}

BootImageUpdateType GetBootImageUpdateType(int bootimageUpdateType) NN_NOEXCEPT
{
    switch(bootimageUpdateType)
    {
    case 0:
        return BootImageUpdateType::Original;
    case 1:
        return BootImageUpdateType::OdinMariko;
    default: NN_UNEXPECTED_DEFAULT;
    }
}

BootImageUpdateType GetBootImageUpdateType(spl::HardwareType hardwareType) NN_NOEXCEPT
{
    switch(hardwareType)
    {
    case spl::HardwareType_Icosa:
        return updater::BootImageUpdateType::Original;
    case spl::HardwareType_Iowa:
        return updater::BootImageUpdateType::OdinMariko;
    case spl::HardwareType_Hoag:
        return updater::BootImageUpdateType::OdinMariko;
    default: NN_UNEXPECTED_DEFAULT;
    }
}


}}
