﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/TargetConfigs/build_Base.h>

#include <nn/uart/uart_PortTypes.h>

#if   defined(NN_BUILD_CONFIG_HARDWARE_BDSLIMX6)
    #include "uart_TargetSpec-hardware.bdsl-imx6.h"
#elif defined(NN_BUILD_CONFIG_HARDWARE_JETSONTK1)
    #include "uart_TargetSpec-hardware.jetson-tk1.h"
#elif defined(NN_BUILD_CONFIG_HARDWARE_JETSONTK2) || defined(NN_BUILD_CONFIG_HARDWARE_NX)
    #include "uart_TargetSpec-soc.tegra-x1.h"
#else
    #error NN_BUILD_CONFIG_HARDWARE_ not selected or supported
#endif

namespace nn {
namespace uart {
namespace driver {
namespace detail {

NN_FORCEINLINE
bool IsSupportedBaudRateImpl(int portIndex, BaudRate baudRate) NN_NOEXCEPT
{
    for (const BaudRateSupportInfo& supportInfo : SupportedBaudRateList)
    {
        if (supportInfo.portIndex == portIndex &&
            supportInfo.baudRate == baudRate)
        {
            return true;
        }
    }
    return false;
}

NN_FORCEINLINE
bool IsSupportedFlowControlModeImpl(int portIndex, FlowControlMode flowControlMode) NN_NOEXCEPT
{
    for (const FlowControlModeSupportInfo& supportInfo : SupportedFlowControlModeList)
    {
        if (supportInfo.portIndex == portIndex &&
            supportInfo.flowControlMode == flowControlMode)
        {
            return true;
        }
    }
    return false;
}

NN_FORCEINLINE
bool IsSupportedPortEventImpl(int portIndex, PortEventType portEvent) NN_NOEXCEPT
{
    for (const auto& supportInfo : SupportedPortEventList)
    {
        if (supportInfo.portIndex == portIndex &&
            supportInfo.eventType == portEvent)
        {
            return true;
        }
    }

    return false;
}

// ポートが利用可能か調べる
NN_FORCEINLINE
bool IsAvailablePortIndex(int portIndex) NN_NOEXCEPT
{
    // 範囲外のポート番号は利用不可
    if (portIndex < 0 || portIndex >= PortCountMax)
    {
        return false;
    }

    // 指定されたポートに対応するボーレートが定義されていれば利用可能とする
    for (const auto& supportInfo : SupportedBaudRateList)
    {
        if (supportInfo.portIndex == portIndex)
        {
            return true;
        }
    }

    return false;
}

// 公開 API の PortName を内部で使用するポート番号に変換する
NN_FORCEINLINE
int ConvertPortNameToPortIndex(PortName name) NN_NOEXCEPT
{
    // サポートしているポートかどうかは別途チェックされるため、
    // ここではサポート外のポートであってもリストにあれば返す
    for (const auto& portMap : PortMapCombinationList)
    {
        if (portMap.publicPortName == name)
        {
            return portMap.portIndex;
        }
    }

    return UnsupportedPortIndex;
}

} // detail
} // driver
} // uart
} // nn
