﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/uart/uart_Result.h>

namespace nn {
namespace uart {
namespace driver {
namespace detail {

/**
 * @brief ポートステータスを表す列挙体。
 * 定義値の大小がステータス保持の優先度を表す。
 * より優先度の高いステータスが発生した場合に、LastStatus を上書きする
 *
 * ステータス保持の優先順位（値の小さいほうが高優先度）:
 *   1. BREAK の受信（現在使用される経路なし）
 *   2. 通信経路上の想定外のエラー（現在使用される経路なし）
 *   3. 内部バッファ不足によるデータ欠落
 *   4. FIFO オーバーランによるデータ欠落
 *   5. 通信経路上で混入したデータ不整合（パリティ・フレームエラー）
 *   6. その他の不明なエラー
 *   7. 成功（成功ステータスは必ず最低優先度である）
 */
enum class PortStatusType
{
    Break               = 1,
    UnexpectedBusError  = 2,
    OutOfBuffer         = 3,
    HardwareOverrun     = 4,
    FrameError          = 5,
    ParityError         = 6,
    Unknown             = 1023,
    Success             = 1024
};

/**
 * @brief ポートのエラー状態を適切な優先順位で保持し、返すクラス
 */
class PortStatus
{
    NN_DISALLOW_COPY(PortStatus);

public:
    PortStatus() NN_NOEXCEPT : m_Status(PortStatusType::Success) {}

    void Clear() NN_NOEXCEPT
    {
        m_Status = PortStatusType::Success;
    }
    PortStatusType GetAndClear() NN_NOEXCEPT
    {
        auto status = m_Status;
        Clear();
        return status;
    }
    nn::Result GetAndClearAsResult() NN_NOEXCEPT
    {
        auto status = GetAndClear();
        return ToResult(status);
    }
    void SetStatus(PortStatusType newStatus) NN_NOEXCEPT
    {
        // 優先度のより高い（値が小さい）ステータスコードの場合にのみ上書きする
        if (newStatus < m_Status)
        {
            m_Status = newStatus;
        }
    }

private:
    nn::Result ToResult(PortStatusType status) const NN_NOEXCEPT
    {
        switch (status)
        {
        case PortStatusType::Success:
            return nn::ResultSuccess();

        case PortStatusType::OutOfBuffer:
            return nn::uart::ResultOutOfBuffer();

        case PortStatusType::HardwareOverrun:
            return nn::uart::ResultHardwareOverrun();

        case PortStatusType::FrameError:
            return nn::uart::ResultFrameError();

        case PortStatusType::ParityError:
            return nn::uart::ResultParityError();

        case PortStatusType::Break:
        case PortStatusType::UnexpectedBusError:
        case PortStatusType::Unknown:
        default:
            NN_UNEXPECTED_DEFAULT;
        }
    }

private:
    PortStatusType  m_Status;
};

} // detail
} // driver
} // uart
} // nn
