﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief   UART ドライバのポート毎の機能提供クラス
 */

#include <mutex>

#include <nn/nn_Abort.h>
#include <nn/nn_SdkAssert.h>

#include <nn/os.h>
#include <nn/util/util_ScopeExit.h>

#include <nn/uart/uart_PortTypes.h>
#include <nn/uart/uart_Result.h>
#include <nn/uart/detail/uart_Log.h>
#include "uart_PortSessionImpl.h"
#include "uart_PortAccessor.h"
#include "uart_CircularBuffer.h"
#include "uart_Command.h"

namespace {

template <typename UInt>
NN_FORCEINLINE
bool IsAlignedWithMemoryPageSize(UInt param) NN_NOEXCEPT
{
    return (param & (nn::os::MemoryPageSize - 1)) == 0 ? true : false;
}

}

namespace nn {
namespace uart {
namespace driver {
namespace detail {

void PortSessionImpl::ConfigurePort(const PortConfigType& portConfig) NN_NOEXCEPT
{
    GlobalLock lock(this);

    NN_SDK_ASSERT(!m_IsOpen);

    m_BaudRate = portConfig._baudRate;
    m_FlowControlMode = portConfig._flowControlMode;
    m_IsInvertTx = portConfig._isInvertTx;
    m_IsInvertRx = portConfig._isInvertRx;
    m_IsInvertRts = portConfig._isInvertRts;
    m_IsInvertCts = portConfig._isInvertCts;
    m_ErrorStatus.Clear();

    // バッファ指定の事前条件
    NN_SDK_REQUIRES(portConfig._sendBufferLength > 0 ||
                    portConfig._receiveBufferLength > 0);

    // 内部送信バッファの指定あり
    if (portConfig._sendBufferLength > 0)
    {
        NN_SDK_REQUIRES_NOT_NULL(portConfig._sendBuffer);
        NN_SDK_REQUIRES(IsAlignedWithMemoryPageSize<uintptr_t>(reinterpret_cast<uintptr_t>(portConfig._sendBuffer)));
        NN_SDK_REQUIRES(IsAlignedWithMemoryPageSize<size_t>(portConfig._sendBufferLength));
        m_SendBuffer.Initialize(portConfig._sendBuffer, portConfig._sendBufferLength);
    }

    // 内部受信バッファの指定あり
    if (portConfig._receiveBufferLength > 0)
    {
        NN_SDK_REQUIRES_NOT_NULL(portConfig._receiveBuffer);
        NN_SDK_REQUIRES(IsAlignedWithMemoryPageSize<uintptr_t>(reinterpret_cast<uintptr_t>(portConfig._receiveBuffer)));
        NN_SDK_REQUIRES(IsAlignedWithMemoryPageSize<size_t>(portConfig._receiveBufferLength));
        m_ReceiveBuffer.Initialize(portConfig._receiveBuffer, portConfig._receiveBufferLength);
    }
}

void PortSessionImpl::DeconfigurePort() NN_NOEXCEPT
{
    GlobalLock lock(this);

    NN_SDK_ASSERT(!m_IsOpen);

    // 以下、ConfigurePort と逆順に終了していく

    if (m_ReceiveBuffer.IsInitialized())
    {
        m_ReceiveBuffer.Finalize();
    }
    if (m_SendBuffer.IsInitialized())
    {
        m_SendBuffer.Finalize();
    }
}

void PortSessionImpl::Open() NN_NOEXCEPT
{
    NN_SDK_ASSERT(!m_IsOpen);

    m_PortAccessor.Initialize(m_PortIndex);

    // 割り込みハンドラをスレッドから呼び出すよう登録
    nn::uart::driver::detail::InterruptNotifier::GetInstance().SetHandler(
        m_PortIndex,
        HandleInterruptCallback,
        reinterpret_cast<uintptr_t>(this)
        );

    m_PortAccessor.EnablePowerAndClock(m_BaudRate);
    m_PortAccessor.Setup(m_BaudRate, m_FlowControlMode, m_IsInvertTx, m_IsInvertRx, m_IsInvertRts, m_IsInvertCts);

    m_PortAccessor.Start();

#ifdef NN_DETAIL_UART_ENABLE_DMA
    // DMA の割り込みハンドラの登録
    for (int i = 0; i < DmaChannelIndex_Max; ++i)
    {
        m_DmaInterruptArgs[i].pSessionImpl = this;
        m_DmaInterruptArgs[i].channelIndex = i;
        nn::uart::driver::detail::InterruptNotifier::GetInstance().SetDmaHandler(
                                        m_PortIndex,
                                        i,
                                        HandleDmaInterruptCallback,
                                        reinterpret_cast<uintptr_t>(&m_DmaInterruptArgs[i]));
    }
#endif
    // 受信可能通知の割り込みは、受信バッファが一杯でないときにだけ ON にする（つまり初期状態 ON）
    // 送信可能通知の割り込みは、送信バッファが空でないときにだけ ON にする（つまり初期状態 OFF）
    if (m_ReceiveBuffer.IsInitialized())
    {
        m_PortAccessor.SetReceiveInterruptEnabled(true);
    }

    m_IsOpen = true;
}

void PortSessionImpl::Close() NN_NOEXCEPT
{
    // クリーンアップでも呼びうるため多重呼び出しを許可
    if (m_IsOpen)
    {
        // 以下、Open と逆順に終了していく

        m_IsOpen = false;
        m_IsOpenSuspended = false;

        m_PortAccessor.SetRtsToHigh(true);

        // Rx の FIFO は一番先に clear する。(FIFO にデータが残ったまま割り込みを落とすと SpuriousInterrupt が発生し続ける)
        int flushRxFifoCount = 0;
        while (!m_PortAccessor.FlushRxFifo(m_BaudRate))
        {
            flushRxFifoCount++;
            NN_ABORT_UNLESS_LESS_EQUAL(flushRxFifoCount, MaxFlushRetryCount);
        }

        m_PortAccessor.SetReceiveInterruptEnabled(false);
        m_PortAccessor.SetTransmitterReadyInterruptEnabled(false);

        m_PortAccessor.Stop();

        // 割り込みハンドラの解除
        nn::uart::driver::detail::InterruptNotifier::GetInstance().SetHandler(m_PortIndex, nullptr, 0);

        // Tx の FIFO は割り込みを止めて、これ以上データが書かれない状態で clear する。
        int flushTxFifoCount = 0;
        while (!m_PortAccessor.FlushTxFifo(m_BaudRate))
        {
            flushTxFifoCount++;
            NN_ABORT_UNLESS_LESS_EQUAL(flushTxFifoCount, MaxFlushRetryCount);
        }

        m_PortAccessor.Finalize();
    }
}

void PortSessionImpl::Suspend() NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_IsOpen);

    if (!m_IsOpenSuspended)
    {
        m_IsOpenSuspended = true;

        m_IsForceRtsHighBeforeSleep = m_PortAccessor.IsForceRtsHigh();
        m_PortAccessor.SetRtsToHigh(true);

        m_PortAccessor.SetReceiveInterruptEnabled(false);
        m_PortAccessor.SetTransmitterReadyInterruptEnabled(false);
#ifdef NN_DETAIL_UART_ENABLE_DMA
        m_PortAccessor.SuspendDmaTransfer();
#endif
    }
}

void PortSessionImpl::DisablePowerAndClock() NN_NOEXCEPT
{
    m_PortAccessor.DisablePowerAndClock();
}

void PortSessionImpl::Resume() NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_IsOpen);

    if (m_IsOpenSuspended)
    {
        if (m_ReceiveBuffer.IsInitialized())
        {
            m_ReceiveBuffer.Empty(false);
        }
        if (m_SendBuffer.IsInitialized())
        {
            m_SendBuffer.Empty(false);
        }

        m_PortAccessor.EnablePowerAndClock(m_BaudRate);
        m_PortAccessor.Setup(m_BaudRate, m_FlowControlMode, m_IsInvertTx, m_IsInvertRx, m_IsInvertRts, m_IsInvertCts);
#ifdef NN_DETAIL_UART_ENABLE_DMA
        m_PortAccessor.ResumeDmaTransfer();

        // DMA の割り込みハンドラの再登録
        // スレッドが走ってないことのチェック
        NN_SDK_ASSERT(!InterruptNotifier::GetInstance().IsThreadRunning());
        for (int i = 0; i < DmaChannelIndex_Max; ++i)
        {
            m_DmaInterruptArgs[i].pSessionImpl = this;
            m_DmaInterruptArgs[i].channelIndex = i;
            // Resume 時にはまだスレッドが走ってないので、そのまま Link する
            nn::uart::driver::detail::InterruptNotifier::GetInstance().LinkDmaHandler(
                m_PortIndex,
                i,
                HandleDmaInterruptCallback,
                reinterpret_cast<uintptr_t>(&m_DmaInterruptArgs[i]));
        }
#endif

        // 受信可能通知の割り込みは、受信バッファが一杯でないときにだけ ON にする（つまり初期状態 ON）
        // 送信可能通知の割り込みは、送信バッファが空でないときにだけ ON にする（つまり初期状態 OFF）
        if (m_ReceiveBuffer.IsInitialized())
        {
            m_PortAccessor.SetReceiveInterruptEnabled(true);
        }
        m_PortAccessor.SetRtsToHigh(m_IsForceRtsHighBeforeSleep);

        m_IsOpenSuspended = false;
    }
}

void PortSessionImpl::HandleInterruptCallback(uintptr_t arg) NN_NOEXCEPT
{
    auto pSessionImpl = reinterpret_cast<PortSessionImpl*>(arg);
    pSessionImpl->HandleInterrupt();
}

void PortSessionImpl::HandleInterrupt() NN_NOEXCEPT
{
    //NN_DETAIL_UART_INFO("Uart Interrupt Occured\n");
    {
        std::lock_guard<StaticMutex> lock(m_MutexSend);

        if ( m_PortAccessor.IsTransmitterReadyInterruptEnabled() && m_SendBuffer.IsInitialized())
        {
            // 送信バッファが空なら割り込みを無効にする
            if (m_SendBuffer.IsEmpty())
            {
                m_PortAccessor.SetTransmitterReadyInterruptEnabled(false);
            }
            else
            {
                // Data might be ready on send buffer
                m_PortAccessor.SendDataFromCircularBuffer(m_ErrorStatus, m_SendBuffer);

                // 送信バッファが空になったら割り込みを無効にする
                if (m_SendBuffer.IsEmpty())
                {
                    //NN_DETAIL_UART_INFO("SetDisable SendBuffer by empty\n");
                    m_PortAccessor.SetTransmitterReadyInterruptEnabled(false);
                }
            }
            HandleSendBufferEvent();
        }
    }

    {
        std::lock_guard<StaticMutex> lock(m_MutexReceive);

        if (m_ReceiveBuffer.IsInitialized() &&
            (m_PortAccessor.IsReceiveTimeout() || m_PortAccessor.IsReceiveEnd() || (m_PortAccessor.IsReceiveReadyInterruptEnabled() && m_PortAccessor.IsReceiveReady())))
        {
            // 受信バッファがいっぱいになったら、空きが出来るまで割り込みを無効にする
            if (m_ReceiveBuffer.IsFull())
            {
                m_PortAccessor.SetReceiveInterruptEnabled(false);

                // この時 ReceiveEnd が立っていた場合はクリアとイベントのシグナルを行う
                if (m_PortAccessor.IsReceiveEnd())
                {
                    m_PortAccessor.ClearReceiveEndInterrupt();
                    if (m_PortEvent.receiveEnd.pEvent != nullptr && !m_ReceiveBuffer.IsEmpty())
                    {
                        //NN_DETAIL_UART_INFO("HandleReceiveEndEvent -- no data left, signaling app\n");
                        nn::os::SignalSystemEvent(m_PortEvent.receiveEnd.pEvent);
                    }
                }
            }
            else
            {
                // Receive data might be ready, get and put it to circular receive buffer until it is empty

                /* If FIFO has more than 4 bytes then it will just trigger DMA receive.
                   If FIFO has less than 4 bytes then it will copy data by PIO.
                   The problem of this design is that it takes 4 more charactor cycles to trigger EORD.
                 */
                m_PortAccessor.ReceiveDataToCircularBuffer(m_ErrorStatus, m_ReceiveBuffer);

                // 受信バッファがいっぱいになったら、空きができるまで一旦割り込みを無効にする
                if (m_ReceiveBuffer.IsFull())
                {
                    m_PortAccessor.SetReceiveInterruptEnabled(false);
                }
            }
            HandleReceiveBufferEvent();
            HandleReceiveEndEvent();
        }
    }
}

#ifdef NN_DETAIL_UART_ENABLE_DMA

void PortSessionImpl::HandleDmaInterruptCallback(uintptr_t arg) NN_NOEXCEPT
{
    auto* pArgs = reinterpret_cast<DmaInterruptArgs*>(arg);
    pArgs->pSessionImpl->HandleDmaInterrupt(pArgs->channelIndex);
}

void PortSessionImpl::HandleDmaInterrupt(int channelIndex) NN_NOEXCEPT
{
    // DMA の割り込みハンドラを呼ぶ
    if(channelIndex == DmaChannelIndex_Send)
    {
        std::lock_guard<StaticMutex> lock(m_MutexSend);

        //NN_DETAIL_UART_INFO("%s() handle DMA interrupt channelIndex=DmaChannelIndex_Send\n", NN_CURRENT_FUNCTION_NAME);
        // Send 側は完了通知を受けて、UART 側の割り込みを Enable にするだけ
        m_PortAccessor.HandleSendDmaInterrupt();
    }
    else
    {
        std::lock_guard<StaticMutex> lock(m_MutexReceive);

        //NN_DETAIL_UART_INFO("%s() handle DMA interrupt channelIndex=DmaChannelIndex_Receive\n", NN_CURRENT_FUNCTION_NAME);
        // Receive 側は完了通知を受けて、完了していたら、DMA のバッファから Circular バッファへデータをコピーする
        if (m_PortAccessor.HandleReceiveDmaInterrupt(m_ErrorStatus, m_ReceiveBuffer))
        {
            HandleReceiveBufferEvent();
            HandleReceiveEndEvent();
            m_PortAccessor.SetReceiveInterruptEnabled(true);
            return;
        }

        HandleReceiveBufferEvent();
        HandleReceiveEndEvent();
    }
}

#endif

void PortSessionImpl::HandleSendBufferEvent() NN_NOEXCEPT
{
    //NN_DETAIL_UART_INFO("HandleSendBufferEvent(%d)\n", m_PortIndex);

    // SendBufferEmpty
    if (m_PortEvent.sendBufferEmpty.pEvent != nullptr)
    {
        // 送信バッファが空になったら SendBufferEmpty イベントをシグナル化
        if (m_SendBuffer.IsEmpty())
        {
            nn::os::SignalSystemEvent(m_PortEvent.sendBufferEmpty.pEvent);
        }
    }

    // SendBufferReady
    if (m_PortEvent.sendBufferReady.pEvent != nullptr)
    {
        // 送信バッファの空きが閾値以上になったら SendBufferReady イベントをシグナル化
        if (m_SendBuffer.GetWritableLength() >= m_PortEvent.sendBufferReady.threshold)
        {
            nn::os::SignalSystemEvent(m_PortEvent.sendBufferReady.pEvent);
        }
    }
}

void PortSessionImpl::HandleReceiveBufferEvent() NN_NOEXCEPT
{
    //NN_DETAIL_UART_INFO("HandleReceiveBufferEvent(%d)\n", m_PortIndex);

    auto& eventHolder = m_PortEvent.receiveBufferReady;

    if (eventHolder.pEvent == nullptr)
    {
        return;
    }

    // 受信済みデータサイズが閾値以上になったら ReceiveBufferReady イベントをシグナル化
    if (m_ReceiveBuffer.GetReadableLength() >= eventHolder.threshold)
    {
        nn::os::SignalSystemEvent(eventHolder.pEvent);
    }
}

void PortSessionImpl::HandleReceiveEndEvent() NN_NOEXCEPT
{
    //NN_DETAIL_UART_INFO("HandleReceiveEndEvent\n");
    if(m_PortAccessor.IsReceiveEndInterruptEnabled())
    {
        if (m_PortAccessor.WasReceiveEndOccured())
        {
            if (m_PortEvent.receiveEnd.pEvent != nullptr)
            {
                //NN_DETAIL_UART_INFO("HandleReceiveEndEvent -- no data left, signaling app\n");
                nn::os::SignalSystemEvent(m_PortEvent.receiveEnd.pEvent);
            }
            m_PortAccessor.ResetReceiveEndOccured();
        }
    }
}

void PortSessionImpl::Send(CommandSend& command) NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_IsOpen);

    std::lock_guard<StaticMutex> lock(m_MutexSend);
    auto& param = command.GetParam();

    NN_SDK_REQUIRES(m_SendBuffer.IsInitialized());

    auto doneBytes = m_SendBuffer.Write(param.in.data, param.in.dataBytes);
    param.out.doneBytes = doneBytes;

    // 送りたいデータがある間だけ割り込みを ON にする
    if (!m_SendBuffer.IsEmpty())
    {
        m_PortAccessor.SetTransmitterReadyInterruptEnabled(true);
    }
}

void PortSessionImpl::Receive(CommandReceive& command) NN_NOEXCEPT
{
    std::lock_guard<StaticMutex> lock(m_MutexReceive);
    auto& param = command.GetParam();
    size_t doneBytes;

    //NN_DETAIL_UART_INFO("Receive, param.in.dataBytes = %d\n", param.in.dataBytes);

    NN_SDK_ASSERT(m_IsOpen);

    NN_SDK_REQUIRES(m_ReceiveBuffer.IsInitialized());

    // 受信中のエラーの有無を読み出す
    auto result = m_ErrorStatus.GetAndClearAsResult();

    if(result.IsSuccess())
    {
        // 循環バッファのデータを読み出す
        doneBytes = m_ReceiveBuffer.Read(param.in.data, param.in.dataBytes);

        // もし DMA Buffer の中にコピーされてないデータがある場合、取り出しを試みる。
        if (m_PortAccessor.IsRestDataExist())
        {
            if (m_PortAccessor.MoveRestDataToCircularBuffer(m_ErrorStatus, m_ReceiveBuffer))
            {
                // この時、DMA の転送単位よりも小さな data が FIFO に入っていた場合、残ったままになる可能性があるため、
                // ReceiveDataToCircularBuffer を呼んでおく
                if (m_ReceiveBuffer.IsWritableSpaceExist(1))
                {
                    m_PortAccessor.ReceiveDataToCircularBufferByPio(m_ErrorStatus, m_ReceiveBuffer);
                }

                // データが入ったので、イベントをシグナルしておく
                HandleReceiveBufferEvent();
                if (m_PortEvent.receiveEnd.pEvent != nullptr)
                {
                    //NN_DETAIL_UART_INFO("HandleReceiveEndEvent -- no data left, signaling app\n");
                    nn::os::SignalSystemEvent(m_PortEvent.receiveEnd.pEvent);
                }
            }
        }
    }
    else
    {
        // エラー時は循環バッファの中身はすべて空にしておく
        // SF の仕様上、ResultSuccess() 以外では doneByte に有効な値が入らないため
        m_ReceiveBuffer.Empty(false);

        // 今回の受信もできなかったことにする
        doneBytes = 0;
    }

    param.out.doneBytes = doneBytes;
    param.out.result = result;

    // 受信する余地がある間だけ割り込みを ON にする
    if (!m_ReceiveBuffer.IsFull() && !m_PortAccessor.IsRestDataExist())
    {
        m_PortAccessor.SetReceiveInterruptEnabled(true);
    }
}

bool PortSessionImpl::BindEvent(nn::os::SystemEventType* pEvent,
                                PortEventType eventType,
                                size_t threshold) NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_IsOpen);
    NN_SDK_REQUIRES_NOT_NULL(pEvent);

    NN_ABORT_UNLESS(IsSupportedPortEventImpl(GetPortIndex(), eventType));

    EventHolder* pEventHolder;
    switch (eventType)
    {
    case PortEventType_SendBufferEmpty:
        {
            // SendBufferEmpty は threshold を無視する
            NN_SDK_REQUIRES(m_SendBuffer.IsInitialized());
            pEventHolder = &m_PortEvent.sendBufferEmpty;
            threshold    = 0;
        }
        break;

    case PortEventType_SendBufferReady:
        {
            NN_SDK_REQUIRES(m_SendBuffer.IsInitialized());
            NN_SDK_REQUIRES(threshold >= 1 &&
                            threshold <= m_SendBuffer.GetBufferLength());
            pEventHolder = &m_PortEvent.sendBufferReady;
        }
        break;

    case PortEventType_ReceiveBufferReady:
        {
            NN_SDK_REQUIRES(m_ReceiveBuffer.IsInitialized());
            NN_SDK_REQUIRES(threshold >= 1 &&
                            threshold <= m_ReceiveBuffer.GetBufferLength());
            pEventHolder = &m_PortEvent.receiveBufferReady;
        }
        break;

    case PortEventType_ReceiveEnd:
        {
            // 割り込み条件が固定のため、threshold は無視する
            pEventHolder = &m_PortEvent.receiveEnd;
            threshold    = 0;
        }
        break;

    default:
        NN_ABORT("Unsupported event type (%d)\n", eventType);
    }

    if (pEventHolder->pEvent != nullptr)
    {
        // 既にイベントが関連付けられている
        return false;
    }

    // SystemEvent をプロセス間イベントとして初期化
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::CreateSystemEvent(pEvent, nn::os::EventClearMode_ManualClear, true));

    pEventHolder->pEvent    = pEvent;
    pEventHolder->threshold = threshold;

    switch (eventType)
    {
    case PortEventType_ReceiveEnd:
        // 割り込みを有効化
        m_PortAccessor.SetReceiveEndInterruptEnabled(true);
        break;

    // 以下のケースは既に条件を満たしている可能性があるため、イベントの発生確認を行う
    case PortEventType_ReceiveBufferReady:
        HandleReceiveBufferEvent();
        break;

    case PortEventType_SendBufferEmpty:
    case PortEventType_SendBufferReady:
        HandleSendBufferEvent();
        break;

    default: NN_UNEXPECTED_DEFAULT;
    }

    return true;
}

bool PortSessionImpl::UnbindEvent(nn::os::SystemEventType* pEvent) NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_IsOpen);
    NN_SDK_REQUIRES_NOT_NULL(pEvent);

    bool isUnbound = false;

    // 指定されたイベントの関連付けをすべて解除する
    if (m_PortEvent.sendBufferEmpty.pEvent == pEvent)
    {
        m_PortEvent.sendBufferEmpty.pEvent = nullptr;
        isUnbound = true;
    }

    if (m_PortEvent.sendBufferReady.pEvent == pEvent)
    {
        m_PortEvent.sendBufferReady.pEvent = nullptr;
        isUnbound = true;
    }

    if (m_PortEvent.receiveBufferReady.pEvent == pEvent)
    {
        m_PortEvent.receiveBufferReady.pEvent = nullptr;
        isUnbound = true;
    }

    if (m_PortEvent.receiveEnd.pEvent == pEvent)
    {
        // 割り込みも止める
        m_PortAccessor.SetReceiveEndInterruptEnabled(false);
        m_PortEvent.receiveEnd.pEvent = nullptr;
        isUnbound = true;
    }

    return isUnbound;
}

} // detail
} // driver
} // uart
} // nn
