﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/util/util_BitPack.h>

#include "uart_CircularBuffer.h"
#include "uart_PortStatus.h"
#include "uart_TargetSpec.h"

#ifdef NN_DETAIL_UART_ENABLE_DMA
#include "uart_DmaAccessor-soc.tegra.h"
#endif

namespace nn {
namespace uart {
namespace driver {
namespace detail {

class PortAccessor
{
    NN_DISALLOW_COPY(PortAccessor);
    NN_DISALLOW_MOVE(PortAccessor);

public:
    PortAccessor() NN_NOEXCEPT :
        m_Reg(),
        m_PortIndex(),
        m_IsInvertTxRts(false),
        m_IsUse2StopBitOver3MBaud(false),
        m_IsForceRtsHigh(false),
        m_WasReceiveEnd(false)
    {}

    void Initialize(int portIdx) NN_NOEXCEPT;
    void Finalize() NN_NOEXCEPT;

    void EnablePowerAndClock(int baudRate) NN_NOEXCEPT;
    void DisablePowerAndClock() NN_NOEXCEPT;
    void Setup(int baudRate, FlowControlMode flowControlMode, bool isInvertTx, bool isInvertRx, bool isInvertRts, bool isInvertCts) NN_NOEXCEPT;
    void Start() NN_NOEXCEPT;
    void Stop() NN_NOEXCEPT;

    bool IsTransmitterReady() const NN_NOEXCEPT;
    bool IsReceiveFifoEmpty() const NN_NOEXCEPT;
    bool IsReceiveReady() const NN_NOEXCEPT;
    bool IsReceiveEnd() const NN_NOEXCEPT;
    bool IsReceiveTimeout() const NN_NOEXCEPT;
    bool WasReceiveEndOccured() NN_NOEXCEPT
    {
        return m_WasReceiveEnd;
    }

    void ResetReceiveEndOccured() NN_NOEXCEPT
    {
        m_WasReceiveEnd = false;
    }

    bool IsReceiveReadyInterruptEnabled() const NN_NOEXCEPT;
    bool IsReceiveEndInterruptEnabled() const NN_NOEXCEPT;
    void SetReceiveReadyInterruptEnabled(bool enable) NN_NOEXCEPT;
    void SetReceiveTimeoutInterruptEnabled(bool enable) NN_NOEXCEPT;
    void SetReceiveEndInterruptEnabled(bool enable) NN_NOEXCEPT;
    void SetReceiveInterruptEnabled(bool enable) NN_NOEXCEPT;
    void ClearReceiveEndInterrupt() NN_NOEXCEPT;
    bool MoveRestDataToCircularBuffer(PortStatus& outErrorStatus, CircularBuffer& receiveBuffer) NN_NOEXCEPT;
    void SetRtsToHigh(bool enable) NN_NOEXCEPT;
    bool IsRestDataExist() NN_NOEXCEPT;
    bool IsForceRtsHigh() NN_NOEXCEPT
    {
        return m_IsForceRtsHigh;
    }

    bool FlushTxFifo(int baudrate) NN_NOEXCEPT;
    bool FlushRxFifo(int baudrate) NN_NOEXCEPT;

    void SendByte(char data) NN_NOEXCEPT;
    bool IsTransmitterReadyInterruptEnabled() const NN_NOEXCEPT;
    void SetTransmitterReadyInterruptEnabled(bool enable) NN_NOEXCEPT;
    PortStatusType ReceiveByte(char* data) NN_NOEXCEPT;
    void SendDataFromCircularBuffer(PortStatus& outErrorStatus, CircularBuffer& sendBuffer) NN_NOEXCEPT;
    void ReceiveDataToCircularBuffer(PortStatus& outErrorStatus, CircularBuffer& receiveBuffer) NN_NOEXCEPT;
    void ReceiveDataToCircularBufferByPio(PortStatus& outErrorStatus, CircularBuffer& receiveBuffer) NN_NOEXCEPT;

    bool IsAnyInterruptEnabled() NN_NOEXCEPT;
    bool IsAnyInterruptOccured() NN_NOEXCEPT;

    void HandleSendDmaInterrupt() NN_NOEXCEPT;
    bool HandleReceiveDmaInterrupt(PortStatus& outErrorStatus, CircularBuffer& receiveBuffer) NN_NOEXCEPT;

    void SuspendDmaTransfer() NN_NOEXCEPT;
    void ResumeDmaTransfer() NN_NOEXCEPT;

    /**
     * @brief デバッグ用関数。関連レジスタの情報をダンプする。
     */
    void DumpRegisters() NN_NOEXCEPT;

private:

    /**
    * @brief Flush 処理内でレジスタが変わるのを待つタイムアウト
    */
    const TimeSpan FlushRegisterCheckTimeout = nn::TimeSpan::FromMilliSeconds(100);


    /**
     * @brief レジスタのビット定義 ([Ref1] 32.4)
     */
    struct UartRegBit
    {
        typedef nn::util::BitPack32::Field< 0, 8, char > Thr;
        typedef nn::util::BitPack32::Field< 0, 8, char > Rbr;
        typedef nn::util::BitPack32::Field< 0, 8, char > Dll;

        typedef nn::util::BitPack32::Field< 0, 8, char > Dlm;
        typedef nn::util::BitPack32::Field< 5, 1, bool > IerEord;
        typedef nn::util::BitPack32::Field< 4, 1, bool > IerRxTimeout;
        typedef nn::util::BitPack32::Field< 3, 1, bool > IerMsi;
        typedef nn::util::BitPack32::Field< 2, 1, bool > IerRxs;
        typedef nn::util::BitPack32::Field< 1, 1, bool > IerThr;
        typedef nn::util::BitPack32::Field< 0, 1, bool > IerRhr;

        typedef nn::util::BitPack32::Field< 6, 2, int >  IirEnFifo;
        typedef nn::util::BitPack32::Field< 0, 4, int >  IirId;
        typedef nn::util::BitPack32::Field< 6, 2, int >  FcrRxTrig;
        typedef nn::util::BitPack32::Field< 4, 2, int >  FcrTxTrig;
        typedef nn::util::BitPack32::Field< 3, 1, bool > FcrDma;
        typedef nn::util::BitPack32::Field< 2, 1, bool > FcrTxClr;
        typedef nn::util::BitPack32::Field< 1, 1, bool > FcrRxClr;
        typedef nn::util::BitPack32::Field< 0, 1, bool > FcrEnFifo;

        typedef nn::util::BitPack32::Field< 7, 1, bool > LcrDlab;
        typedef nn::util::BitPack32::Field< 6, 1, bool > LcrSetBrk;
        typedef nn::util::BitPack32::Field< 5, 1, bool > LcrSetP;
        typedef nn::util::BitPack32::Field< 4, 1, bool > LcrEven;
        typedef nn::util::BitPack32::Field< 3, 1, bool > LcrPar;
        typedef nn::util::BitPack32::Field< 2, 1, bool > LcrStop;
        typedef nn::util::BitPack32::Field< 0, 2, int >  LcrWdSize;

        typedef nn::util::BitPack32::Field< 6, 1, bool > McrRtsEn;
        typedef nn::util::BitPack32::Field< 5, 1, bool > McrCtsEn;
        typedef nn::util::BitPack32::Field< 4, 1, bool > McrLoopbk;
        typedef nn::util::BitPack32::Field< 3, 1, bool > McrOut2;
        typedef nn::util::BitPack32::Field< 2, 1, bool > McrOut1;
        typedef nn::util::BitPack32::Field< 1, 1, bool > McrRts;
        typedef nn::util::BitPack32::Field< 0, 1, bool > McrDtr;

        typedef nn::util::BitPack32::Field< 9, 1, bool > LsrRxFifoEmpty;
        typedef nn::util::BitPack32::Field< 8, 1, bool > LsrTxFifoFull;
        typedef nn::util::BitPack32::Field< 7, 1, bool > LsrFifoE;
        typedef nn::util::BitPack32::Field< 6, 1, bool > LsrTmty;
        typedef nn::util::BitPack32::Field< 5, 1, bool > LsrThre;
        typedef nn::util::BitPack32::Field< 4, 1, bool > LsrBrk;
        typedef nn::util::BitPack32::Field< 3, 1, bool > LsrFerr;
        typedef nn::util::BitPack32::Field< 2, 1, bool > LsrPerr;
        typedef nn::util::BitPack32::Field< 1, 1, bool > LsrOvrf;
        typedef nn::util::BitPack32::Field< 0, 1, bool > LsrRdr;

        typedef nn::util::BitPack32::Field< 4, 1, bool > MsrCts;

        typedef nn::util::BitPack32::Field< 3, 1, bool > IrdaCsrRts;
        typedef nn::util::BitPack32::Field< 2, 1, bool > IrdaCsrCts;
        typedef nn::util::BitPack32::Field< 1, 1, bool > IrdaCsrTxd;
        typedef nn::util::BitPack32::Field< 0, 1, bool > IrdaCsrRxd;
    };

    /**
     * @brief ベースレジスタからのオフセット参照のための構造体
     */
    struct UartRegTable
    {
        volatile uint32_t thrRbrDll;        // 0x00 THR (WO): Transmit Holding Register
                                            //      RBR (RO): Receiver Buffer Register
                                            //      DLL (RW): Divisor Latch LSByte
        volatile uint32_t ierDlm;           // 0x04 Interrupt Enable and Divisor Latch MSByte
        volatile uint32_t iirFcr;           // 0x08 IIR (RO): Interrupt Identification Register
                                            //      FCR (WO): FIFO Control Register
        volatile uint32_t lcr;              // 0x0c Line Control Register
        volatile uint32_t mcr;              // 0x10 Modem Control Register
        volatile uint32_t lsr;              // 0x14 Line Status Register
        volatile uint32_t msr;              // 0x18 Modem Status Register
        volatile uint32_t spr;              // 0x1c Scratchpad Register
        volatile uint32_t irdaCsr;          // 0x20 IrDA Pulse Coding CSR
        volatile uint32_t rxFifoCfg;        // 0x24
        volatile uint32_t mie;              // 0x28 Modem Interrupt Enable Register
                 uint8_t  padding1[0x10];   // 0x2c - 0x3b
        volatile uint32_t asr;              // 0x3c Auto Sense Baud Register
    };

    /**
     * @brief バス／ポート毎に異なるレジスタのベースアドレスおよびビットシフトを保持するための構造体
     */
    struct RegInfo
    {
        // UART Register ([Ref1] 32.4)
        UartRegTable*       uart;

        // Clock source register ([Ref1] 5.7.64, 5.7.65, 5.7.70, 5.7.75)
        volatile uint32_t*  clockSource;

        // Clock enable register ([Ref1] 5.7.7 - 5.7.9)
        volatile uint32_t*  clockEnable;
        uint32_t            clockEnableBitMask;

        // Reset register ([Ref1] 5.7.4 - 5.7.6)
        volatile uint32_t*  reset;
        uint32_t            resetBitMask;

        RegInfo() NN_NOEXCEPT :
            uart(nullptr),
            clockSource(nullptr),
            clockEnable(nullptr),
            clockEnableBitMask(0),
            reset(nullptr),
            resetBitMask(0) {}
    };

private:
    void SetBaseAddr(int portIdx) NN_NOEXCEPT;
    PortStatusType GetReceiveStatus() NN_NOEXCEPT;
#if defined(NN_BUILD_CONFIG_SPEC_NX)
    void ResetController(uint32_t* pOutClockHz, uint32_t expectedClockHz) NN_NOEXCEPT;
#else
    void ResetControllerWithoutPcv(int clockSelect) NN_NOEXCEPT;
#endif
    void SendDataFromCircularBufferByPio(PortStatus& outErrorStatus, CircularBuffer& sendBuffer) NN_NOEXCEPT;

private:
    RegInfo         m_Reg;
    int             m_PortIndex;
    bool            m_IsInvertTxRts;
    bool            m_IsUse2StopBitOver3MBaud;
    bool            m_IsForceRtsHigh;
    bool            m_WasReceiveEnd;

#ifdef NN_DETAIL_UART_ENABLE_DMA
private:
    DmaAccessor     m_DmaAccessor;
#endif

};

} // detail
} // driver
} // uart
} // nn
