﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief   UART ドライバのレジスタアクセサクラスの実装部(Tegra ファミリー向け)
 * @details Jetson TK1/TK2 で使用可能な UART のみを対象とします。
 *          参考文献：
 *            - [Ref1] Tegra K1 Technical Reference Manual, Revision v02p, Jul 15 2014
 */

#include <algorithm>
#include <climits>
#include <cmath>

#include <nn/TargetConfigs/build_Base.h>

#include <nn/nn_Abort.h>
#include <nn/nn_SdkAssert.h>

#include <nn/os.h>
#include <nn/os/os_MemoryAttribute.h>
#include <nn/dd.h>
#if defined(NN_BUILD_CONFIG_SPEC_NX)
#include <nn/pcv/pcv.h>
#endif

#include <nn/uart/uart_PortTypes.h>
#include <nn/uart/detail/uart_Log.h>
#include "uart_PortAccessor.h"
#include "uart_CircularBuffer.h"
#include "uart_DdUtil.h"
#include "uart_PortAccessor-soc.tegra.h"
#include "uart_RegisterUart-soc.tegra.h"
#include "uart_Interrupt.h"

#ifdef NN_DETAIL_UART_ENABLE_DMA
#include "uart_DmaAccessor-soc.tegra.h"
#endif

namespace nn {
namespace uart {
namespace driver {
namespace detail {

namespace {

// TK1/TK2 の物理アドレス
// UART
const nn::dd::PhysicalAddress   UartPhysicalAddress   = 0x70006000ull;
const size_t                    UartAddressSize       = 0x1000;

#ifndef NN_BUILD_CONFIG_SPEC_NX
// CLK_RST
const nn::dd::PhysicalAddress   ClkRstPhysicalAddress = 0x60006000ull;
const size_t                    ClkRstAddressSize     = 0x1000;
#endif

/**
 * @brief ターゲットボーレートと参照クロックを元に divisor を決定する
 */
int CalcDivisor(int baudRate, int refFreq) NN_NOEXCEPT
{
    NN_SDK_ASSERT(baudRate < INT_MAX / 16); // 値域チェック
    int divisor = (refFreq + (16 * baudRate) / 2) / (16 * baudRate);

    return divisor;
}

#if defined(NN_BUILD_CONFIG_SPEC_NX)
nn::pcv::Module ToPcvModule(int portIndex) NN_NOEXCEPT
{
    switch(portIndex)
    {
    case 0:
        return nn::pcv::Module_UartA;
    case 1:
        return nn::pcv::Module_UartB;
    case 2:
        return nn::pcv::Module_UartC;
    case 3:
        return nn::pcv::Module_UartD;
    default: NN_UNEXPECTED_DEFAULT;
    }
}
#endif

}

void PortAccessor::Initialize(int portIdx) NN_NOEXCEPT
{
    m_PortIndex = portIdx;
    SetBaseAddr(portIdx);

#if defined(NN_BUILD_CONFIG_SPEC_NX)
    nn::pcv::Initialize();
#endif  // NN_BUILD_CONFIG_SPEC_NX

#ifdef NN_DETAIL_UART_ENABLE_DMA
    m_DmaAccessor.Initialize(portIdx);
#endif  // NN_DETAIL_UART_ENABLE_DMA
}

void PortAccessor::Finalize() NN_NOEXCEPT
{
#ifdef NN_DETAIL_UART_ENABLE_DMA
    m_DmaAccessor.Finalize();
#endif  // NN_DETAIL_UART_ENABLE_DMA

#if defined(NN_BUILD_CONFIG_SPEC_NX)
    nn::pcv::Finalize();
#endif  // NN_BUILD_CONFIG_SPEC_NX
}

/**
 * @brief ポート毎に固有となる、レジスタアドレス等の情報をセットする。
 */
void PortAccessor::SetBaseAddr(int portIdx) NN_NOEXCEPT
{
    // 対応しているポートか確認
    NN_ABORT_UNLESS_RANGE(portIdx, 0, PortCountMax);

    uintptr_t uartVirtualAddress = GetVirtualAddress(UartPhysicalAddress, UartAddressSize);
#ifndef NN_BUILD_CONFIG_SPEC_NX
    uintptr_t clkRstVirtualAddress = GetVirtualAddress(ClkRstPhysicalAddress, ClkRstAddressSize);
#endif
    // NN_DETAIL_UART_INFO("uartVirtualAddress = %p\n", uartVirtualAddress);
    // NN_DETAIL_UART_INFO("clkRstVirtualAddress = %p\n", clkRstVirtualAddress);

    // UART, CLK_RST レジスタの設定
    const auto& uartRegs   = UartRegisterTable[portIdx];
    m_Reg.uart        = reinterpret_cast<UartRegTable*>     (uartVirtualAddress   + uartRegs.uartAddress);

#ifndef NN_BUILD_CONFIG_SPEC_NX
    const auto& clkRstRegs = ClkRstRegisterTable[portIdx];
    m_Reg.clockSource = reinterpret_cast<volatile uint32_t*>(clkRstVirtualAddress + clkRstRegs.clockSourceAddress);
    m_Reg.clockEnable = reinterpret_cast<volatile uint32_t*>(clkRstVirtualAddress + clkRstRegs.clockEnableAddress);
    m_Reg.clockEnableBitMask =                                                      clkRstRegs.clockEnableBitMask;
    m_Reg.reset       = reinterpret_cast<volatile uint32_t*>(clkRstVirtualAddress + clkRstRegs.resetAddress);
    m_Reg.resetBitMask =                                                            clkRstRegs.resetBitMask;
#endif

#if defined(NN_BUILD_CONFIG_HARDWARE_NX)
    if(portIdx == 1 || portIdx == 2)
    {
        m_IsInvertTxRts = true;
        m_IsUse2StopBitOver3MBaud = true;
    }
#endif

    // DumpRegisters();
}

/**
 * @brief リセットレジスタを使ってバスをリセットする。
 *        リセット信号をアサートしている間にクロックソースを選択し、有効にする。
 *        一連の手順は [Ref1] 5.7.1 をベースにしている。
 */
#if defined(NN_BUILD_CONFIG_SPEC_NX)
void PortAccessor::ResetController(uint32_t* pOutClockHz, uint32_t expectedClockHz) NN_NOEXCEPT
{
    auto module = ToPcvModule(m_PortIndex);

    // Assert reset signal
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::pcv::SetReset(module, true));

    // Set Clock Rate (クロックがオフの場合、ここで自動的に enable される)
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::pcv::SetClockRate(module, expectedClockHz));

    // 誤差をチェック
    float            errorRate = 0;
    nn::pcv::ClockHz clock;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::pcv::GetClockRate(&clock, module));
    errorRate = 1 - static_cast<float>(clock) / expectedClockHz;

    // 3% 以上の誤差がある場合は 3% 上げたクロックを指定してみる
    if(errorRate >= 0.03)
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::pcv::SetClockRate(module, expectedClockHz * 1.03));

        // 誤差をチェック
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::pcv::GetClockRate(&clock, module));

        // 再設定しても誤差が 3% 以上ある場合、想定外なので Abort する。
        // UART として保障している 115200 / 1M / 3M のボーレートでは発生しないことを確認済み
        NN_ABORT_UNLESS(fabsf(1 - static_cast<float>(clock) / expectedClockHz) <= 0.03, "[uart] Detect Over 3 percents Clock Error Rate\n");
    }

    *pOutClockHz = clock;

    // De-assert reset signal
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::pcv::SetReset(module, false));
}
#else
void PortAccessor::ResetControllerWithoutPcv(int clockSelect) NN_NOEXCEPT
{
    // Assert reset signal
    SetBitOn32  (m_Reg.reset, m_Reg.resetBitMask);

    // Enable clock
    SetBitOn32  (m_Reg.clockEnable, m_Reg.clockEnableBitMask);

    // Set CLK_SRC
    Modify32    (m_Reg.clockSource, (clockSelect << 29), (0x7 << 29));
    DummyRead(m_Reg.clockSource);
    nn::os::SleepThread( nn::TimeSpan::FromMicroSeconds(2) );

    // De-assert reset signal
    SetBitOff32 (m_Reg.reset, m_Reg.resetBitMask);
    DummyRead(m_Reg.clockSource);
}
#endif

/**
 * @brief デバイスをリセットし、クロックの設定を行う
 */
void PortAccessor::EnablePowerAndClock(int baudRate) NN_NOEXCEPT
{
    // TODO: PINMUX の設定が必要？（現状ではデフォルトで UART B が有効な様子）

#if defined(NN_BUILD_CONFIG_SPEC_NX)
    // バスのリセット
    // UART 側の divisor の計算を楽にするため、
    // baudRate の 16 倍のクロックを出してもらうようにクロック周りを設定する
    uint32_t clockRate;
    ResetController(&clockRate, 16 * baudRate);

    // 設定される実際のボーレートと指定したボーレートの表示
    auto divisor = CalcDivisor(baudRate, 16 * baudRate);
    NN_UNUSED(divisor); // Release ビルド対策
    NN_DETAIL_UART_INFO("Port %d: Real Baudrate = %d, (Exected Baudrate = %d)\n", m_PortIndex, clockRate / 16 / divisor, baudRate);

    // 誤差も計算できるが、スタックサイズを増やす必要があるため、コメントアウト
    //NN_DETAIL_UART_INFO("baudrate = %d, Error Rate = %.3f \n", clockRate / 16 / divisor,
    //    fabs(1.00 - static_cast<float>(clockRate) / 16 / divisor / baudRate));
#else
    // PLLP_OUT0 (408 MHz)
    const int clockSelect = ClkRstRegisterSource_PllPOut0;
    ResetControllerWithoutPcv(clockSelect);
#endif
}

/**
 * @brief クロック供給の停止
 */
void PortAccessor::DisablePowerAndClock() NN_NOEXCEPT
{
#if defined(NN_BUILD_CONFIG_SPEC_NX)
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::pcv::SetClockEnabled(ToPcvModule(m_PortIndex), false));
#endif
}

/**
 * @brief レジスタ値の初期化の実施。BaudRate の設定や FIFO の設定などを行う。
 *        最初は一切の割り込みを有効にしない。
 */
void PortAccessor::Setup(int baudRate, FlowControlMode flowControlMode, bool isInvertTx, bool isInvertRx, bool isInvertRts, bool isInvertCts) NN_NOEXCEPT
{
    // ボーレート divisor の設定
#if defined(NN_BUILD_CONFIG_SPEC_NX)
    const int refFreq = 16 * baudRate;
#else
    const int refFreq = 408000000;
#endif
    auto divisor = CalcDivisor(baudRate, refFreq);

    // DLAB を立てて、DLL/DLM に divisor をセットし、DLAB を戻す
    SetBitOn32  (&(m_Reg.uart->lcr),       UartRegBit::LcrDlab::Mask);
    Write32     (&(m_Reg.uart->thrRbrDll), divisor & UartRegBit::Dll::Mask);
    Write32     (&(m_Reg.uart->ierDlm),    (divisor >> 8) & UartRegBit::Dlm::Mask);
    SetBitOff32 (&(m_Reg.uart->lcr),       UartRegBit::LcrDlab::Mask);

    // Disable all modem interrupts
    SetBitOff32 (&(m_Reg.uart->mie), 0xf);

    if(baudRate >= 3000000 && m_IsUse2StopBitOver3MBaud)
    {
        // 対向デバイスの仕様上、ボーレートが 3M 以上の時は 2 Stop bit にする
        // Parity: None | Stop: 2 bit | Word size: 8 bit
        Write32     (&(m_Reg.uart->lcr), (0 << UartRegBit::LcrSetP::Pos) |
                                         (0 << UartRegBit::LcrEven::Pos) |
                                         (0 << UartRegBit::LcrPar::Pos)  |
                                         (1 << UartRegBit::LcrStop::Pos) |
                                         (3 << UartRegBit::LcrWdSize::Pos) );
    }
    else
    {
        // Parity: None | Stop: 1 bit | Word size: 8 bit
        Write32     (&(m_Reg.uart->lcr), (0 << UartRegBit::LcrSetP::Pos) |
                                         (0 << UartRegBit::LcrEven::Pos) |
                                         (0 << UartRegBit::LcrPar::Pos)  |
                                         (0 << UartRegBit::LcrStop::Pos) |
                                         (3 << UartRegBit::LcrWdSize::Pos) );
    }



    switch (flowControlMode)
    {
    case FlowControlMode_None:
        {
            SetBitOff32(&(m_Reg.uart->mcr), (1 << UartRegBit::McrRtsEn::Pos) |
                                             (1 << UartRegBit::McrCtsEn::Pos) );
        }
        break;
    case FlowControlMode_Hardware:
        {
            SetBitOn32 (&(m_Reg.uart->mcr), (1 << UartRegBit::McrRtsEn::Pos) |
                                             (1 << UartRegBit::McrCtsEn::Pos) );
        }
        break;
    default:
        NN_ABORT("Unsupported flow control mode (%d)\n", flowControlMode);
    }

#ifdef NN_DETAIL_UART_ENABLE_DMA
    // RX Trigger Lv: 4 byte | TX Trigger Lv: 16 byte | DMA MODE: 1 | Clear Tx Fifo | Clear Rx Fifo | Enable Fifo
    Write32  (&(m_Reg.uart->iirFcr), (1 << UartRegBit::FcrRxTrig::Pos) |
                                      (0 << UartRegBit::FcrTxTrig::Pos) |
                                      (1 << UartRegBit::FcrDma::Pos)    |
                                      (1 << UartRegBit::FcrTxClr::Pos)  |
                                      (1 << UartRegBit::FcrRxClr::Pos)  |
                                      (1 << UartRegBit::FcrEnFifo::Pos) );
#else
    // RX Trigger Lv: 1 byte | TX Trigger Lv: 16 byte | DMA MODE: 1 | Clear Tx Fifo | Clear Rx Fifo | Enable Fifo
    Write32  (&(m_Reg.uart->iirFcr), (0 << UartRegBit::FcrRxTrig::Pos) |
                                      (0 << UartRegBit::FcrTxTrig::Pos) |
                                      (1 << UartRegBit::FcrDma::Pos)    |
                                      (1 << UartRegBit::FcrTxClr::Pos)  |
                                      (1 << UartRegBit::FcrRxClr::Pos)  |
                                      (1 << UartRegBit::FcrEnFifo::Pos) );
#endif

    // Invert すべきポートの場合、UART_IRDA_CSR_0 レジスタを使用して信号を反転させる
    // (SIR encoding には設定していない)
    if (isInvertTx)
    {
        SetBitOn32(&(m_Reg.uart->irdaCsr), (1 << UartRegBit::IrdaCsrTxd::Pos));
    }
    else
    {
        SetBitOn32(&(m_Reg.uart->irdaCsr), (0 << UartRegBit::IrdaCsrTxd::Pos));
    }

    if (isInvertRx)
    {
        SetBitOn32(&(m_Reg.uart->irdaCsr), (1 << UartRegBit::IrdaCsrRxd::Pos));
    }
    else
    {
        SetBitOn32(&(m_Reg.uart->irdaCsr), (0 << UartRegBit::IrdaCsrRxd::Pos));
    }

    if (isInvertRts)
    {
        SetBitOn32(&(m_Reg.uart->irdaCsr), (1 << UartRegBit::IrdaCsrRts::Pos));
    }
    else
    {
        SetBitOn32(&(m_Reg.uart->irdaCsr), (0 << UartRegBit::IrdaCsrRts::Pos));
    }

    if (isInvertCts)
    {
        SetBitOn32(&(m_Reg.uart->irdaCsr), (1 << UartRegBit::IrdaCsrCts::Pos));
    }
    else
    {
        SetBitOn32(&(m_Reg.uart->irdaCsr), (0 << UartRegBit::IrdaCsrCts::Pos));
    }

    // 念の為、lcr を読んでおく。(read に意味のあるレジスタは読まない)
    DummyRead(&(m_Reg.uart->lcr));
    nn::dd::EnsureMemoryAccess();

#ifdef NN_DETAIL_UART_ENABLE_DMA
    //NN_DETAIL_UART_INFO("UART is using DMA\n");
#else
    NN_DETAIL_UART_INFO("UART is *not* using DMA\n");
#endif
    // DumpRegisters();
}

void PortAccessor::Start() NN_NOEXCEPT
{
#ifdef NN_DETAIL_UART_ENABLE_DMA
    m_DmaAccessor.Start(UartPhysicalAddress);
#endif  // NN_DETAIL_UART_ENABLE_DMA
}

void PortAccessor::Stop() NN_NOEXCEPT
{
#ifdef NN_DETAIL_UART_ENABLE_DMA
    m_DmaAccessor.Stop();
#endif  // NN_DETAIL_UART_ENABLE_DMA
}

bool PortAccessor::IsTransmitterReady() const NN_NOEXCEPT
{
    // LSR.TX_FIFO_FULL
    return !(Read32(&(m_Reg.uart->lsr)) & UartRegBit::LsrTxFifoFull::Mask);
}

bool PortAccessor::IsTransmitterReadyInterruptEnabled() const NN_NOEXCEPT
{
    // IER.IE_THR
    return (Read32(&(m_Reg.uart->ierDlm)) & UartRegBit::IerThr::Mask) ? true : false;
}

void PortAccessor::SetTransmitterReadyInterruptEnabled(bool enable) NN_NOEXCEPT
{
    // IER.IE_THR
    if (enable)
    {
        SetBitOn32(&(m_Reg.uart->ierDlm), UartRegBit::IerThr::Mask);
    }
    else
    {
        SetBitOff32(&(m_Reg.uart->ierDlm), UartRegBit::IerThr::Mask);
    }
    DummyRead(&(m_Reg.uart->ierDlm));
    nn::dd::EnsureMemoryAccess();
}

bool PortAccessor::IsAnyInterruptEnabled() NN_NOEXCEPT
{
    return Read32(&(m_Reg.uart->ierDlm)) != 0x00;
}

bool PortAccessor::IsAnyInterruptOccured() NN_NOEXCEPT
{
    // iir の下位 4 bit を取り出し、 0001 = No interrupt であることをチェックする
    return (Read32(&(m_Reg.uart->iirFcr)) & 0x0f) != 0x01;
}

bool PortAccessor::IsReceiveFifoEmpty() const NN_NOEXCEPT
{
    // LSR.RDR
    return (Read32(&(m_Reg.uart->lsr)) & UartRegBit::LsrRxFifoEmpty::Mask) ? true : false;
}

bool PortAccessor::IsReceiveReady() const NN_NOEXCEPT
{
    // LSR.RDR
    return (Read32(&(m_Reg.uart->lsr)) & UartRegBit::LsrRdr::Mask) ? true : false;
}

bool PortAccessor::IsReceiveTimeout() const NN_NOEXCEPT
{
    // EORD の割り込み ID
    const nn::Bit8 RxtInterruptId = 0x0c;

    // IIR_FCR
    return (Read32(&(m_Reg.uart->iirFcr)) & UartRegBit::IirId::Mask) == RxtInterruptId ? true : false;
}

bool PortAccessor::IsReceiveEnd() const NN_NOEXCEPT
{
    // EORD の割り込み ID
    const nn::Bit8 EordInterruptId = 0x08;

    // IIR_FCR
    return (Read32(&(m_Reg.uart->iirFcr)) & UartRegBit::IirId::Mask) == EordInterruptId ? true : false;
}

bool PortAccessor::IsReceiveReadyInterruptEnabled() const NN_NOEXCEPT
{
    // IER.IE_RHR
    return (Read32(&(m_Reg.uart->ierDlm)) & UartRegBit::IerRhr::Mask) ? true : false;
}

bool PortAccessor::IsReceiveEndInterruptEnabled() const NN_NOEXCEPT
{
    // IER.IE_EORD
    return (Read32(&(m_Reg.uart->ierDlm)) & UartRegBit::IerEord::Mask) ? true : false;
}

void PortAccessor::SetReceiveInterruptEnabled(bool enable) NN_NOEXCEPT
{
    // IER.IE_RHR
    if (enable)
    {
        WriteMasked32(&(m_Reg.uart->ierDlm),
            UartRegBit::IerRhr::Mask | UartRegBit::IerRxTimeout::Mask | UartRegBit::IerEord::Mask,
            UartRegBit::IerRhr::Mask | UartRegBit::IerRxTimeout::Mask | UartRegBit::IerEord::Mask);
    }
    else
    {
        WriteMasked32(&(m_Reg.uart->ierDlm),
            0x00,
            UartRegBit::IerRhr::Mask | UartRegBit::IerRxTimeout::Mask | UartRegBit::IerEord::Mask);
    }
    DummyRead(&(m_Reg.uart->ierDlm));
    nn::dd::EnsureMemoryAccess();
}

void PortAccessor::SetReceiveReadyInterruptEnabled(bool enable) NN_NOEXCEPT
{
    // IER.IE_RHR
    if (enable)
    {
        SetBitOn32(&(m_Reg.uart->ierDlm), UartRegBit::IerRhr::Mask);
    }
    else
    {
        SetBitOff32(&(m_Reg.uart->ierDlm), UartRegBit::IerRhr::Mask);
    }
    DummyRead(&(m_Reg.uart->ierDlm));
    nn::dd::EnsureMemoryAccess();
}

void PortAccessor::SetReceiveTimeoutInterruptEnabled(bool enable) NN_NOEXCEPT
{
    // IER.IE_RX_TIMEOUT
    if (enable)
    {
        SetBitOn32(&(m_Reg.uart->ierDlm), UartRegBit::IerRxTimeout::Mask);
    }
    else
    {
        SetBitOff32(&(m_Reg.uart->ierDlm), UartRegBit::IerRxTimeout::Mask);
    }
    DummyRead(&(m_Reg.uart->ierDlm));
    nn::dd::EnsureMemoryAccess();
}

void PortAccessor::SetReceiveEndInterruptEnabled(bool enable) NN_NOEXCEPT
{
    // IER.IE_EORD
    if (enable)
    {
        SetBitOn32(&(m_Reg.uart->ierDlm), UartRegBit::IerEord::Mask);
    }
    else
    {
        SetBitOff32(&(m_Reg.uart->ierDlm), UartRegBit::IerEord::Mask);
    }
    DummyRead(&(m_Reg.uart->ierDlm));
    nn::dd::EnsureMemoryAccess();
}

void PortAccessor::SetRtsToHigh(bool enable) NN_NOEXCEPT
{
    if (enable)
    {
        m_IsForceRtsHigh = true;
        SetBitOff32(&(m_Reg.uart->mcr), UartRegBit::McrRtsEn::Mask);
        SetBitOff32(&(m_Reg.uart->mcr), UartRegBit::McrRts::Mask);
    }
    else
    {
        m_IsForceRtsHigh = false;
        SetBitOn32(&(m_Reg.uart->mcr), UartRegBit::McrRts::Mask);
        SetBitOn32(&(m_Reg.uart->mcr), UartRegBit::McrRtsEn::Mask);
    }
}

void PortAccessor::ClearReceiveEndInterrupt() NN_NOEXCEPT
{
    // EORD 割り込みをクリアするには IER.IE_EORD をクリアする必要があるので、
    // 一旦 IER.IE_EORD をクリアした後に再セットする
    SetReceiveEndInterruptEnabled(false);
    SetReceiveEndInterruptEnabled(true);
}

bool PortAccessor::FlushTxFifo(int baudrate) NN_NOEXCEPT
{
    // ref. 38.2.20 FIFO Flushing Guidelines

    // 本関数を呼び出す際に、 別スレッドで TX にデータを書かないこと。

    //Program TX_CLR to 1 (UART_IIR_FCR_0[2] - TX_FLUSH request)
    WriteMasked32(&(m_Reg.uart->iirFcr), UartRegBit::FcrTxClr::Mask, UartRegBit::FcrTxClr::Mask);

    // Wait 32 baud cycles
    int waitMicroSecondTime = (32 * 1000 * 1000 / baudrate) + 1;
    nn::os::SleepThread(nn::TimeSpan::FromMicroSeconds(waitMicroSecondTime));

    // Poll for TMTY to be ‘1’ (UART_LSR_0[6])
    auto startTick = nn::os::GetSystemTick();
    while (!((Read32(&(m_Reg.uart->lsr)) & UartRegBit::LsrTmty::Mask) == UartRegBit::LsrTmty::Mask))
    {
        NN_DETAIL_UART_ERROR("Poll for TMTY to be ‘1’ (UART_LSR_0[6])\n");
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(1));
        if ((nn::os::GetSystemTick() - startTick).ToTimeSpan() > FlushRegisterCheckTimeout)
        {
            return false;
        }
    }

    return true;
}

bool PortAccessor::FlushRxFifo(int baudrate) NN_NOEXCEPT
{
    // ref. 38.2.20 FIFO Flushing Guidelines

    bool wasRtsHigh = m_IsForceRtsHigh;

    if (!wasRtsHigh)
    {
        // Program rts_enable to 0 (MCR[6])
        SetRtsToHigh(true);
    }

    // Make sure MCR[1] is 0
    auto startTick = nn::os::GetSystemTick();
    while (!((Read32(&(m_Reg.uart->mcr)) & UartRegBit::McrRts::Mask) == 0x00))
    {
        NN_DETAIL_UART_ERROR("Make sure MCR[1] is 0\n");
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(1));
        if ((nn::os::GetSystemTick() - startTick).ToTimeSpan() > FlushRegisterCheckTimeout)
        {
            return false;
        }
    }

    // Wait for 1 character time
    // 1 character は startbit(1) + 8 bit + 2stopbit(2) + parity = 13 bit を最大値として計算
    int waitMicroSecondTime = (13 * 1000 * 1000 / baudrate) + 1;
    nn::os::SleepThread(nn::TimeSpan::FromMicroSeconds(waitMicroSecondTime));

    //Program RX_CLR to 1 (UART_IIR_FCR_0[1] - RX_FLUSH request)
    WriteMasked32(&(m_Reg.uart->iirFcr), UartRegBit::FcrRxClr::Mask, UartRegBit::FcrRxClr::Mask);

    // Poll for RDR to be ‘0’ (UART_LSR_0[0])
    startTick = nn::os::GetSystemTick();
    while (!((Read32(&(m_Reg.uart->lsr)) & UartRegBit::LsrRdr::Mask) == 0x00))
    {
        NN_DETAIL_UART_ERROR("Poll for RDR to be ‘0’ (UART_LSR_0[0])\n");
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(1));
        if ((nn::os::GetSystemTick() - startTick).ToTimeSpan() > FlushRegisterCheckTimeout)
        {
            return false;
        }
    }

    if (!wasRtsHigh)
    {
        SetRtsToHigh(false);
    }

    return true;
}

void PortAccessor::SendByte(char data) NN_NOEXCEPT
{
    Write32(&(m_Reg.uart->thrRbrDll), (data << UartRegBit::Thr::Pos) & UartRegBit::Thr::Mask);
    DummyRead(&(m_Reg.uart->ierDlm));
}

PortStatusType PortAccessor::ReceiveByte(char *data) NN_NOEXCEPT
{
    uint32_t val = Read32(&(m_Reg.uart->thrRbrDll));
    auto status = GetReceiveStatus();

    *data = val & UartRegBit::Rbr::Mask;
    return status;
}

PortStatusType PortAccessor::GetReceiveStatus() NN_NOEXCEPT
{
    uint32_t err = Read32(&(m_Reg.uart->lsr));

#if 0 // BRK はサポートしていないので立たない
    if (err & UartRegBit::LsrBrk::Mask)
    {
        return PortStatusType::Break;
    }
    else
#endif
#if 0 // パリティは今サポートしていないので立たない
    if (err & UartRegBit::LsrPerr::Mask)
    {
        return PortStatusType::ParityError;
    }
    else
#endif
    if (err & UartRegBit::LsrFerr::Mask)
    {
        return PortStatusType::FrameError;
    }
    else if (err & UartRegBit::LsrOvrf::Mask)
    {
        return PortStatusType::HardwareOverrun;
    }

    return PortStatusType::Success;
}

void PortAccessor::SendDataFromCircularBufferByPio(PortStatus& outErrorStatus, CircularBuffer& sendBuffer) NN_NOEXCEPT
{
    NN_UNUSED(outErrorStatus);

    //NN_DETAIL_UART_INFO("BytesToSendByPIO = %d\n", sendBuffer.GetReadableLength() );

    // PIO なので効率度外視で 1 バイトずつ送る
    char sendingData;
    size_t sendingSize;
    while( IsTransmitterReady() &&
           (sendingSize = sendBuffer.Read(&sendingData, 1)) > 0 )
    {
        SendByte(sendingData);
    }
}

void PortAccessor::SendDataFromCircularBuffer(PortStatus& outErrorStatus, CircularBuffer& sendBuffer) NN_NOEXCEPT
{
#ifdef NN_DETAIL_UART_ENABLE_DMA

    if(!m_DmaAccessor.IsDmaComplete(DmaChannelIndex_Send))
    {
        // 前回の DMA 送信が完了していない場合は送信関連の割り込みを一旦 disable にして抜ける
        //NN_DETAIL_UART_INFO("Disable Interupt\n");
        SetTransmitterReadyInterruptEnabled(false);
        return;
    }

    // 送信可能なバイト数が DMA のアラインメント未満の場合は PIO
    if (sendBuffer.GetReadableLength() < DmaBufferAlign)
    {
        SendDataFromCircularBufferByPio(outErrorStatus, sendBuffer);
        m_DmaAccessor.SetSendDmaTransferSize(0);
        return;
    }

    // Uart の Transmitter が Ready なら Dma 転送を試みる
    if(IsTransmitterReady())
    {
        // TransmitterReady の Interrupt は disable にしておく
        SetTransmitterReadyInterruptEnabled(false);
        m_DmaAccessor.RequestSendDataByDma(sendBuffer);
    }
    else
    {
        m_DmaAccessor.SetSendDmaTransferSize(0);
    }
#else
    SendDataFromCircularBufferByPio(outErrorStatus, sendBuffer);
#endif  // NN_DETAIL_UART_ENABLE_DMA
}

void PortAccessor::ReceiveDataToCircularBufferByPio(PortStatus& outErrorStatus, CircularBuffer& receiveBuffer) NN_NOEXCEPT
{
    const size_t ReceiveChunkSize = 16;
    char receivedData[ReceiveChunkSize];
    size_t receivedSize = 0;

    while (!IsReceiveFifoEmpty())
    {
        if (receiveBuffer.IsWritableSpaceExist(1))
        {
            auto result = PortStatusType::Success;

            result = ReceiveByte(&receivedData[receivedSize]);

            if (result == PortStatusType::Success)
            {

                bool wasOverwritten = false;
                size_t written = receiveBuffer.Overwrite(&wasOverwritten, &receivedData[receivedSize], 1);
                NN_UNUSED(written);
                if (wasOverwritten) // Receive buffer was full, mark error status
                {
                    outErrorStatus.SetStatus(PortStatusType::OutOfBuffer);
                }
            }
            else
            {
                // Record last error to return to the next caller (do not overwrite the first error till it is handled)
                outErrorStatus.SetStatus(result);
                return;
            }
        }
        else
        {
            // CircularBuffer がいっぱいなのでこれ以上の取り出し処理は行わない
            NN_DETAIL_UART_INFO("CircularBuffer is out of buffer, disable interrupt(%d)\n", __LINE__);
            SetReceiveInterruptEnabled(false);
            return;
        }
    }
}

bool PortAccessor::MoveRestDataToCircularBuffer(PortStatus& outErrorStatus, CircularBuffer& receiveBuffer) NN_NOEXCEPT
{
#ifdef NN_DETAIL_UART_ENABLE_DMA
    return m_DmaAccessor.MoveRestDataToCircularBuffer(outErrorStatus, receiveBuffer);
#else
    // DMA 対応してないときは FIFO にたまっているだけなはずなので、常に true を返す
    return true;
#endif
}

bool PortAccessor::IsRestDataExist() NN_NOEXCEPT
{
#ifdef NN_DETAIL_UART_ENABLE_DMA
    return m_DmaAccessor.IsRestDataExist();
#else
    // DMA 対応してないときは FIFO にたまっているだけなはずなので、常に false を返す
    return false;
#endif
}

void PortAccessor::ReceiveDataToCircularBuffer(PortStatus& outErrorStatus, CircularBuffer& receiveBuffer) NN_NOEXCEPT
{
#ifdef NN_DETAIL_UART_ENABLE_DMA
    //NN_DETAIL_UART_INFO("%s()\n", NN_CURRENT_FUNCTION_NAME);

    // IIR_FCR レジスタの読み出し
    uint32_t iir = Read32(&(m_Reg.uart->iirFcr)) & 0x0f;

    const nn::Bit8 ReceiveTimeoutId = 0x0c;
    const nn::Bit8 ReceiveEndId     = 0x08;

    if (iir == ReceiveTimeoutId || iir == ReceiveEndId)
    {
        // TimeOut や ReceiveEnd の場合は DMA 転送を止めて、データを取り出す。
        // TORIAEZU:関数の名前はもう少し考える (この関数内で CircluarBuffer へのコピーも行う)
        if (!m_DmaAccessor.AbortReceiveDataTransfer(outErrorStatus, receiveBuffer))
        {
            // CircularBuffer がいっぱいなのでこれ以上の取り出し処理は行わない
            NN_DETAIL_UART_INFO("CircularBuffer is out of buffer, disable interrupt(%d)\n", __LINE__);
            SetReceiveInterruptEnabled(false);
            return;
        }
        else
        {
            // FIFO からデータの読み出しが完了し、この時点で ReceiveEnd が立っていたなら、Clear する
            if (iir == ReceiveEndId)
            {
                // 要因のクリアはここで行う
                ClearReceiveEndInterrupt();

                // ここでフラグを立てて、あとでシグナル
                // 要因が立ったら再度シグナルされるのでシステムイベントのクリアは最後でよい
                m_WasReceiveEnd = true;
            }
        }

        if (IsRestDataExist())
        {
            return;
        }

        // TimeOut 時に DMA でデータを取り出した後も FIFO が空になっていなければ、残りは PIO で 1 byte ずつ取り出す。
        char receivedData;

        while (!IsReceiveFifoEmpty())
        {
            auto result = PortStatusType::Success;

            if (receiveBuffer.IsWritableSpaceExist(1))
            {
                result = ReceiveByte(&receivedData);
                if (result == PortStatusType::Success)
                {

                    bool wasOverwritten = false;
                    size_t written = receiveBuffer.Overwrite(&wasOverwritten, &receivedData, 1);
                    NN_UNUSED(written);
                    if (wasOverwritten) // Receive buffer was full, mark error status
                    {
                        outErrorStatus.SetStatus(PortStatusType::OutOfBuffer);
                    }

                 }
                 else
                 {
                    //XXX: can overwrite the previous error
                    outErrorStatus.SetStatus(result);
                }
            }
            else
            {
                NN_DETAIL_UART_INFO("CircularBuffer is out of buffer, disable interrupt(%d)\n", __LINE__);
                // CircularBuffer がいっぱいなのでこれ以上の取り出し処理は行わない
                SetReceiveReadyInterruptEnabled(false);
                return;
            }
        }
    }
    else if (IsReceiveReady())
    {
        //NN_DETAIL_UART_INFO("ReceiveReady\n");

        if(!m_DmaAccessor.IsDmaComplete(DmaChannelIndex_Receive))
        {
            // 前回の DMA 送信が完了していない場合は受信関連の割り込みを一旦 disable にして抜ける
            // ここでは ReceiveEnd だけ受けられるようにしておく
            SetReceiveReadyInterruptEnabled(false);
            SetReceiveTimeoutInterruptEnabled(false);
            return;
        }

        // 前回の DMA が完了していた場合、新しく DMA を行う
        if (!IsRestDataExist())
        {
            // ReceiveReady の Interrupt は disable にしておく
            SetReceiveReadyInterruptEnabled(false);

            m_DmaAccessor.RequestReceiveDataByDma();
        }
        else
        {
            NN_DETAIL_UART_INFO("Rest data exist, not DMA start\n");
        }
    }

#else
    ReceiveDataToCircularBufferByPio(outErrorStatus, receiveBuffer);
#endif  // NN_DETAIL_UART_ENABLE_DMA
}

#ifdef NN_DETAIL_UART_ENABLE_DMA

void PortAccessor::HandleSendDmaInterrupt() NN_NOEXCEPT
{
    m_DmaAccessor.RunInterruptHandler(DmaChannelIndex_Send);

    // Send 側の完了通知なら TransmitterReady を Enable にする。
    if(m_DmaAccessor.IsDmaComplete(DmaChannelIndex_Send))
    {
        SetTransmitterReadyInterruptEnabled(true);
    }
}

bool PortAccessor::HandleReceiveDmaInterrupt(PortStatus& outErrorStatus, CircularBuffer& receiveBuffer) NN_NOEXCEPT
{
    m_DmaAccessor.RunInterruptHandler(DmaChannelIndex_Receive);

    if(m_DmaAccessor.IsDmaComplete(DmaChannelIndex_Receive))
    {
        // データを CircularBuffer へ移す。
        if (!m_DmaAccessor.CopyDataFromDmaBufferToCircularBuffer(outErrorStatus, receiveBuffer))
        {
            // CircularBuffer がいっぱいでデータが移せない状態なので、割り込みを disable にする
            NN_DETAIL_UART_INFO("CircularBuffer is out of buffer, disable interrupt(%d)\n", __LINE__);
            SetReceiveInterruptEnabled(false);

            return false;
        }
    }
    else
    {
        // 基本的にここにはこないはず
        NN_DETAIL_UART_ERROR("Can not AcquireReceiveSemaphore\n");
    }

    return true;
}

void PortAccessor::SuspendDmaTransfer() NN_NOEXCEPT
{
    // Send / Receive 両方の Transfer を Suspend する
    m_DmaAccessor.SuspendDmaTransfer();
}

void PortAccessor:: ResumeDmaTransfer() NN_NOEXCEPT
{
    // Send / Receive 両方の Transger を Resume する
    m_DmaAccessor.ResumeDmaTransfer(UartPhysicalAddress);
}
#endif

void PortAccessor::DumpRegisters() NN_NOEXCEPT
{
    NN_DETAIL_UART_INFO("\n\n*** UART Register info ***\n\n");

#if 0 // PINMUX 状態の確認用
    const nn::dd::PhysicalAddress   ApbMiscPhysicalAddress   = 0x070000000ull;
    const size_t                    ApbMiscAddressSize       = 0x4000;

    uintptr_t apbMiscVirtualAddress = nn::dd::QueryIoMappingAddress(ApbMiscPhysicalAddress, ApbMiscAddressSize);
    if (apbMiscVirtualAddress == 0)
    {
        // PINMUX のレジスタアドレスがマッピングされていないので、ダンプは諦める
        NN_DETAIL_UART_INFO("Cannot dump PINMUX registers as APB MISC (0x%llx) are not mapped. "
                   "Make sure the capability setting is properly set for this process.\n", ApbMiscPhysicalAddress);
    }
    else
    {
        NN_DETAIL_UART_INFO("[Pinmux info (UARTB)] \n\n");

        // 0 for UARTA, 1 for UARTB @ bit 0:1
        NN_DETAIL_UART_INFO("  pinmux1  = 0x%08x\n", *reinterpret_cast<volatile uint32_t*>(apbMiscVirtualAddress + 0x316c));
        NN_DETAIL_UART_INFO("  pinmux2  = 0x%08x\n", *reinterpret_cast<volatile uint32_t*>(apbMiscVirtualAddress + 0x3170));
        NN_DETAIL_UART_INFO("\n");

        NN_DETAIL_UART_INFO("[Pinmux info (UARTD)] \n\n");

        // 2 for UARTD @ bit 0:1
        NN_DETAIL_UART_INFO("  pinmux1  = 0x%08x\n", *reinterpret_cast<volatile uint32_t*>(apbMiscVirtualAddress + 0x3020));
        NN_DETAIL_UART_INFO("  pinmux2  = 0x%08x\n", *reinterpret_cast<volatile uint32_t*>(apbMiscVirtualAddress + 0x3024));
        NN_DETAIL_UART_INFO("  pinmux3  = 0x%08x\n", *reinterpret_cast<volatile uint32_t*>(apbMiscVirtualAddress + 0x3028));
        NN_DETAIL_UART_INFO("  pinmux4  = 0x%08x\n", *reinterpret_cast<volatile uint32_t*>(apbMiscVirtualAddress + 0x302c));

        // 0 for UARTD @ bit 0:1
        NN_DETAIL_UART_INFO("  pinmux5  = 0x%08x\n", *reinterpret_cast<volatile uint32_t*>(apbMiscVirtualAddress + 0x3230));
        NN_DETAIL_UART_INFO("  pinmux6  = 0x%08x\n", *reinterpret_cast<volatile uint32_t*>(apbMiscVirtualAddress + 0x3234));
        NN_DETAIL_UART_INFO("  pinmux7  = 0x%08x\n", *reinterpret_cast<volatile uint32_t*>(apbMiscVirtualAddress + 0x3238));
        NN_DETAIL_UART_INFO("  pinmux8  = 0x%08x\n", *reinterpret_cast<volatile uint32_t*>(apbMiscVirtualAddress + 0x323c));

        // 3 for UARTD @ bit 0:1
        NN_DETAIL_UART_INFO("  pinmux9  = 0x%08x\n", *reinterpret_cast<volatile uint32_t*>(apbMiscVirtualAddress + 0x3314));
        NN_DETAIL_UART_INFO("  pinmux10 = 0x%08x\n", *reinterpret_cast<volatile uint32_t*>(apbMiscVirtualAddress + 0x3318));
        NN_DETAIL_UART_INFO("\n");
    }
#endif

    NN_DETAIL_UART_INFO("[Register locations] \n\n");
    NN_DETAIL_UART_INFO("  uart               @ 0x%p\n",     m_Reg.uart);
#ifndef NN_BUILD_CONFIG_SPEC_NX
    NN_DETAIL_UART_INFO("  clockSource        @ 0x%p\n",     m_Reg.clockSource);
    NN_DETAIL_UART_INFO("  clockEnable        @ 0x%p\n",     m_Reg.clockEnable);
    NN_DETAIL_UART_INFO("  clockEnableBitMask = 0x%08x\n",   m_Reg.clockEnableBitMask);
    NN_DETAIL_UART_INFO("  reset              @ 0x%p\n",     m_Reg.reset);
    NN_DETAIL_UART_INFO("  resetBitMask       = 0x%08x\n\n", m_Reg.resetBitMask);
#endif
    NN_DETAIL_UART_INFO("[Register values] \n\n");
#ifndef NN_BUILD_CONFIG_SPEC_NX
    uint32_t clockSource = *m_Reg.clockSource;
    uint32_t clockEnable = *m_Reg.clockEnable;
    uint32_t reset = *m_Reg.reset;
    bool isClockEnabled = ((clockEnable & m_Reg.clockEnableBitMask) > 0);
    bool isInReset = ((reset & m_Reg.resetBitMask) > 0);

    NN_UNUSED(isClockEnabled); // Escape warning for Release build
    NN_UNUSED(isInReset); // Escape warning for Release build
    NN_UNUSED(clockSource); // Escape warning for Release build
    NN_DETAIL_UART_INFO("  clockSource        = 0x%08x (CLK_SRC:%d, DIV_ENB:%d, CLK_DIVISOR:%d)\n",
                clockSource,
                (clockSource >> 29) & 0x7,
                (clockSource >> 24) & 0x1,
                (clockSource >>  0) & 0xffff);
    NN_DETAIL_UART_INFO("  clockEnable        = 0x%08x (associated bit is %s)\n",
                clockEnable, isClockEnabled ? "true" : "false");
    NN_DETAIL_UART_INFO("  reset              = 0x%08x (associated bit is %s)\n\n",
                reset, isInReset ? "true" : "false");
#endif
    // RxFIFO からの読み出しで副作用があるため回避
    // NN_DETAIL_UART_INFO("  uart->thrRbrDll    = 0x%08x\n", m_Reg.uart->thrRbrDll);
    NN_DETAIL_UART_INFO("  uart->ierDlm       = 0x%08x\n", m_Reg.uart->ierDlm);
    NN_DETAIL_UART_INFO("  uart->iirFcr       = 0x%08x\n", m_Reg.uart->iirFcr);
    NN_DETAIL_UART_INFO("  uart->lcr          = 0x%08x\n", m_Reg.uart->lcr);
    NN_DETAIL_UART_INFO("  uart->mcr          = 0x%08x\n", m_Reg.uart->mcr);
    NN_DETAIL_UART_INFO("  uart->lsr          = 0x%08x\n", m_Reg.uart->lsr);
    NN_DETAIL_UART_INFO("  uart->msr          = 0x%08x\n", m_Reg.uart->msr);
    NN_DETAIL_UART_INFO("  uart->spr          = 0x%08x\n", m_Reg.uart->spr);
    NN_DETAIL_UART_INFO("  uart->irdaCsr      = 0x%08x\n", m_Reg.uart->irdaCsr);
    NN_DETAIL_UART_INFO("  uart->rxFifoCfg    = 0x%08x\n", m_Reg.uart->rxFifoCfg);
    NN_DETAIL_UART_INFO("  uart->mie          = 0x%08x\n", m_Reg.uart->mie);
    NN_DETAIL_UART_INFO("  uart->asr          = 0x%08x\n\n", m_Reg.uart->asr);

    NN_DETAIL_UART_INFO("**************************\n\n");
}

} // detail
} // driver
} // uart
} // nn
