﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief   温度計測ライブラリをサービスフレームのインターフェイスの差を吸収する実装です。
 */

#include <mutex>

#include <nn/nn_SdkAssert.h>

#include <nn/os/os_Mutex.h>
#include <nn/sf/sf_Types.h>
#include <nn/sf/sf_ObjectFactory.h>
#include <nn/result/result_HandlingUtility.h>

#include <nn/ts.h>
#include <nn/ts/ts_ShimInternal.h>

#include "ts_CreateServerProxy.h"

namespace nn {
namespace ts {

namespace {

// Shim ライブラリ実装用のサービスオブジェクトへの共有ポインタ
// DFC と HIPC とで共通でよい。
::nn::sf::SharedPointer<server::IMeasurementServer> g_MeasurementServer;

// Initialize の参照カウント
int g_InitializeCount = 0;

// 参照カウントを守る Mutex
struct StaticMutex
{
    nn::os::MutexType mutex;
    void lock() NN_NOEXCEPT
    {
        nn::os::LockMutex(&mutex);
    }
    void unlock() NN_NOEXCEPT
    {
        nn::os::UnlockMutex(&mutex);
    }
} g_InitializeCountMutex = { NN_OS_MUTEX_INITIALIZER(false) };

}


void Initialize() NN_NOEXCEPT
{
    ::std::lock_guard<StaticMutex> lock(g_InitializeCountMutex);

    if(g_InitializeCount == 0)
    {
        NN_SDK_ASSERT(!g_MeasurementServer);
        g_MeasurementServer = ::nn::ts::CreateMeasurementServerProxy();
    }

    g_InitializeCount++;
}


void InitializeWith(::nn::sf::SharedPointer<server::IMeasurementServer> measurementServer) NN_NOEXCEPT
{
    ::std::lock_guard<StaticMutex> lock(g_InitializeCountMutex);

    NN_SDK_ASSERT(g_InitializeCount == 0);

    g_MeasurementServer = measurementServer;

    g_InitializeCount++;
}


::nn::sf::SharedPointer<server::IMeasurementServer> GetInternalMeasurementServer() NN_NOEXCEPT
{
    return g_MeasurementServer;
}


void Finalize() NN_NOEXCEPT
{
    ::std::lock_guard<StaticMutex> lock(g_InitializeCountMutex);

    NN_SDK_ASSERT(g_InitializeCount > 0);

    g_InitializeCount--;

    if(g_InitializeCount == 0)
    {
        NN_SDK_ASSERT(g_MeasurementServer);
        // 共有ポインタへの nullptr の代入で解放できる。
        g_MeasurementServer = nullptr;
    }
}


LimitTemperature GetTemperatureRange(Location location) NN_NOEXCEPT
{
    LimitTemperature outValue;
    auto result = g_MeasurementServer->GetTemperatureRange(&outValue, static_cast<::std::uint8_t>(location));
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    return outValue;
}


nn::Result GetTemperature(int* pOutTemperature, Location location) NN_NOEXCEPT
{
    return g_MeasurementServer->GetTemperature(pOutTemperature, static_cast<::std::uint8_t>(location));
}

nn::Result GetTemperatureMilliC(int* pOutTemperatureMilliC, Location location) NN_NOEXCEPT
{
    return g_MeasurementServer->GetTemperatureMilliC(pOutTemperatureMilliC, static_cast<::std::uint8_t>(location));
}

void SetMeasurementMode(Location location, MeasurementMode measurementMode) NN_NOEXCEPT
{
    auto result = g_MeasurementServer->SetMeasurementMode(static_cast<::std::uint8_t>(location), static_cast<::std::uint8_t>(measurementMode));
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
}

} // ts
} // nn
