﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <climits>
#include <mutex>

#include <nn/os.h>
#include <nn/nn_Common.h>

#include <nn/nn_SdkAssert.h>
#include <nn/result/result_HandlingUtility.h>

#include "ts_ResourceManager.h"
#include "ts_ISensorController.h"
#include "ts_SensorControllerImpl.h"
#include "ts_Devices.h"


namespace nn {
namespace ts {
namespace impl {
namespace detail {

namespace {

#if defined(NN_BUILD_CONFIG_HARDWARE_NX)
//SocCpuController                g_SocCpuController;
//SocGpuController                g_SocGpuController;
//SocPllController                g_SocPllController;
//SocMemController                g_SocMemController;
ThermalSensorInternalController g_ThermalSensorInternalController;
ThermalSensorExternalController g_ThermalSensorExternalController;
//BatteryController               g_BatteryController;

ISensorController* const        g_SensorController[] = {
//                                    &g_SocCpuController,
//                                    &g_SocGpuController,
//                                    &g_SocPllController,
//                                    &g_SocMemController,
                                    &g_ThermalSensorInternalController,
                                    &g_ThermalSensorExternalController,
//                                    &g_BatteryController,
                                };
#else
    #error NN_BUILD_CONFIG_HARDWARE_ not selected or supported
#endif

const int           InvalidIndex = -1;

int                 g_InitializeCount = 0;

// 参照カウントを守る Mutex
struct StaticMutex
{
    nn::os::MutexType mutex;
    void lock() NN_NOEXCEPT
    {
        nn::os::LockMutex(&mutex);
    }
    void unlock() NN_NOEXCEPT
    {
        nn::os::UnlockMutex(&mutex);
    }
} g_InitializeCountMutex = { NN_OS_MUTEX_INITIALIZER(false) };

}


namespace {
nn::Result InitializeSensorController() NN_NOEXCEPT
{
    for (int i = 0; i < NumberOfLocation; i++)
    {
        NN_RESULT_DO(g_SensorController[i]->Initialize());
    }
    return nn::ResultSuccess();
}


nn::Result FinalizeSensorController() NN_NOEXCEPT
{
    for (int i = 0; i < NumberOfLocation; i++)
    {
        NN_RESULT_DO(g_SensorController[i]->Finalize());
    }
    return nn::ResultSuccess();
}


int LocationToIndex(Location location) NN_NOEXCEPT
{
    for (int i = 0; i < NumberOfLocation; i++)
    {
        if (LocationDeviceCombinationList[i].location == location)
        {
            return i;
        }
    }
    return InvalidIndex;
}


//Location IndexToLocation(int idx) NN_NOEXCEPT
//{
//    return LocationDeviceCombinationList[idx].location;
//}

} // namespace


nn::Result InitializeResourceManager() NN_NOEXCEPT
{
    std::lock_guard<StaticMutex> lock(g_InitializeCountMutex);

    NN_SDK_ASSERT(g_InitializeCount < INT_MAX); // 値域チェック

    g_InitializeCount++;

    // 初回のみの処理
    if (g_InitializeCount == 1)
    {
        NN_RESULT_DO(InitializeSensorController());
    }
    return nn::ResultSuccess();
}



nn::Result FinalizeResourceManager() NN_NOEXCEPT
{
    std::lock_guard<StaticMutex> lock(g_InitializeCountMutex);

    NN_SDK_ASSERT(g_InitializeCount > INT_MIN); // 値域チェック

    g_InitializeCount--;

    // Initialize が呼ばれた回数分 Finalize された場合の処理
    if (g_InitializeCount == 0)
    {
        NN_RESULT_DO(FinalizeSensorController());
    }
    return nn::ResultSuccess();
}



ISensorController& GetSensorController(Location location) NN_NOEXCEPT
{
    return *g_SensorController[LocationToIndex(location)];
}



bool IsLocationValid(Location location) NN_NOEXCEPT
{
    return (LocationToIndex(location) != InvalidIndex ? true : false);
}


}}}} // nn::ts::impl::detail
