﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "tmipc_thread.h"
#include "tmipc_result.h"
#include "../tmagent.h"

//==============================================================================
namespace tmipc {
//==============================================================================

Mutex::Mutex( )
:   m_Created( false )
{
}

//==============================================================================

Mutex::~Mutex( )
{
    // Free resources.
    if( m_Created )
    {
        ::DeleteCriticalSection( &m_CriticalSection );
        m_Created = false;
    }
}

//==============================================================================

void Mutex::Create( )
{
    ASSERT( m_Created == false );

    // Create a critical section.
    ::InitializeCriticalSectionAndSpinCount( &m_CriticalSection, 0x1000 );
    m_Created = true;
}

//==============================================================================

void Mutex::Destroy( )
{
    // Free resources.
    if( m_Created )
    {
        ::DeleteCriticalSection( &m_CriticalSection );
        m_Created = false;
    }
}

//==============================================================================

void Mutex::Lock( )
{
    // Enter the critical section.
    ASSERT( m_Created );
    ::EnterCriticalSection( &m_CriticalSection );
}

//==============================================================================

void Mutex::Unlock()
{
    // Leave the critical section.
    ASSERT( m_Created );
    ::LeaveCriticalSection( &m_CriticalSection );
}

//==============================================================================
//==============================================================================

Event::Event()
:   m_hEvent    ( 0 )
,   m_Created   ( false )
{
}

//==============================================================================

Event::~Event()
{
    if( m_Created )
    {
        Destroy();
    }
}

//==============================================================================

void Event::Create()
{
    ASSERT( !m_Created );
    m_hEvent = ::CreateEvent( NULL, 0, 0, NULL );
    ASSERT( m_hEvent );
    m_Created = true;
}

//==============================================================================

void Event::Destroy()
{
    ASSERT( m_Created );
    ::CloseHandle( m_hEvent );
    m_hEvent = 0;
    m_Created = false;
}

//==============================================================================

void Event::Set( )
{
    ASSERT( m_Created );
    ::SetEvent( m_hEvent );
}

//==============================================================================

void Event::Reset( )
{
    ASSERT( m_Created );
    ::ResetEvent( m_hEvent );
}

//==============================================================================

s32 Event::Wait( s32 Timeout )
{
    ASSERT( m_Created );
    DWORD Result = ::WaitForSingleObject( m_hEvent, Timeout );
    switch( Result )
    {
    case WAIT_TIMEOUT:
        return( TMIPC_RESULT_TIMEOUT );
    case WAIT_OBJECT_0:
        return( TMIPC_RESULT_OK );
    default:
        break;
    }

    return( TMIPC_RESULT_TIMEOUT );
}

//==============================================================================
//==============================================================================

MultiEventWait::MultiEventWait()
{
    m_phEvents       = nullptr;
    m_pEvents        = nullptr;
    m_NumberOfEvents = 0;
}

//==============================================================================

MultiEventWait::~MultiEventWait()
{
    tma::s_Deallocate( m_phEvents, sizeof( HANDLE ) * m_NumberOfEvents );
    m_phEvents = nullptr;
    tma::s_Deallocate( m_pEvents, sizeof( Event* ) * m_NumberOfEvents );
    m_pEvents = nullptr;
    m_NumberOfEvents = 0;
}

//==============================================================================

// A [very] common case: exactly two Events to wait for.
s32 MultiEventWait::SetTwoEvents( Event* pEvent1, Event* pEvent2)
{
    s32 Result = TMIPC_RESULT_OK;

    // Validate the parameters.
    if( Result == TMIPC_RESULT_OK )
    {
        if( (pEvent1 == nullptr) || (pEvent2 == nullptr) )
        {
//TODO: this should be: tmapi::result::RESULT_INVALID_PARAMETER.
            Result = TMIPC_RESULT_FAILED;
        }
    }

    // Set up the array(s) to be exactly two elements for each parameter.
    if( Result == TMIPC_RESULT_OK )
    {
        tma::s_Deallocate( m_pEvents, sizeof( Event* ) * (m_NumberOfEvents) );
        m_pEvents = nullptr;

        tma::s_Deallocate( m_phEvents, sizeof( HANDLE ) * (m_NumberOfEvents) );
        m_phEvents = nullptr;

        m_NumberOfEvents = 2;

        void* pMem = tma::s_Allocate( sizeof(Event*) * m_NumberOfEvents );
        m_pEvents = new (pMem) Event*[m_NumberOfEvents];
        m_pEvents[0] = pEvent1;
        m_pEvents[1] = pEvent2;

        pMem = tma::s_Allocate( sizeof(HANDLE) * m_NumberOfEvents );
        m_phEvents = new (pMem) HANDLE[m_NumberOfEvents];
        m_phEvents[0] = pEvent1->m_hEvent;
        m_phEvents[1] = pEvent2->m_hEvent;
    }

    return Result;
}

//==============================================================================

// Allow the user to add an arbitrary number of Events to wait for.
s32 MultiEventWait::AddEvent( Event* pEvent )
{
    s32 Result = TMIPC_RESULT_OK;

    // Make sure there are not too many Events.
    if( Result == TMIPC_RESULT_OK )
    {
        if( m_NumberOfEvents >= MAXIMUM_WAIT_OBJECTS )
        {
//TODO: Determine if this is a correct response.  It should be in the tmapi::result enum.
            Result = TMIPC_RESULT_MAX;
        }
    }

    // Add the Event.
    if( Result == TMIPC_RESULT_OK )
    {
        // Allocate and copy the existing Events to the new array
        void* pMem = tma::s_Allocate( sizeof( Event* ) * (m_NumberOfEvents + 1 ) );
        Event** pNewEvents = new (pMem) Event*[m_NumberOfEvents + 1];
        for( s32 Index = 0; Index < m_NumberOfEvents; Index++ )
        {
            pNewEvents[Index] = m_pEvents[Index];
        }

        // Add the new Event to the array.
        pNewEvents[m_NumberOfEvents] = pEvent;
        m_NumberOfEvents++;

        // Replace the existing Event array with the new array.
        tma::s_Deallocate( m_pEvents, sizeof( Event* ) * m_NumberOfEvents - 1 );
        m_pEvents = nullptr;
        m_pEvents = pNewEvents;
    }

    // Add the Event HANDLE.
    if( Result == TMIPC_RESULT_OK )
    {
        // Allocate a new HANDLE array.
        tma::s_Deallocate( m_phEvents, sizeof( HANDLE ) * (m_NumberOfEvents - 1) );
        m_phEvents = nullptr;
        void* pMem = tma::s_Allocate( sizeof( HANDLE ) * m_NumberOfEvents );
        m_phEvents = new (pMem) HANDLE[m_NumberOfEvents];

        // Fill in the Event HANDLE array
        for( s32 Index = 0; Index < m_NumberOfEvents; Index++ )
        {
            m_phEvents[Index] = m_pEvents[Index]->m_hEvent;
        }
    }

    return Result;
}

//==============================================================================

s32 MultiEventWait::Wait( s32 TimeoutMS, s32& SignalledIndex )
{
    s32 Result = TMIPC_RESULT_OK;

    // Default the out parameter.
    SignalledIndex = -1;

    // Ensure there is at least one Event to wait for...
    if( Result == TMIPC_RESULT_OK )
    {
        if( m_NumberOfEvents < 1 )
        {
            Result = TMIPC_RESULT_FAILED;
        }
    }

    // Do the wait.
    if( Result == TMIPC_RESULT_OK )
    {
        // Perform the wait.
        const DWORD WaitResult = ::WaitForMultipleObjects( m_NumberOfEvents, m_phEvents, true, TimeoutMS );

        // Determine the wait results.
        if ((WAIT_OBJECT_0 <= WaitResult) && (WaitResult < (WAIT_OBJECT_0 + m_NumberOfEvents)) )
        {
            SignalledIndex = WaitResult - WAIT_OBJECT_0;
            Result = TMIPC_RESULT_OK;
        }
        else if( WaitResult == WAIT_TIMEOUT )
        {
            Result = TMIPC_RESULT_TIMEOUT;
        }
        else
        {
            Result = TMIPC_RESULT_FAILED;
        }
    }

    return Result;
}

//==============================================================================
//==============================================================================

Thread::Thread()
:   m_hThread   ( INVALID_HANDLE_VALUE )
,   m_ThreadId  ( 0 )
{
}

//==============================================================================

Thread::~Thread()
{
}

//==============================================================================

tmipc::Result Thread::Start( ThreadFunction* pFn, void* pArg, s32 Priority )
{
    // Create a new system thread.
    m_hThread = ::CreateThread( NULL, 0, (LPTHREAD_START_ROUTINE)pFn, pArg, 0, &m_ThreadId );
    ASSERT( m_hThread != INVALID_HANDLE_VALUE );
    ASSERT( m_ThreadId != 0 );
    ::SetThreadPriority( m_hThread, Priority );

    return( TMIPC_RESULT_OK );
}

//==============================================================================

tmipc::Result Thread::Join( s32 Timeout )
{
    // Wait for the thread to exit.
    if( m_hThread != INVALID_HANDLE_VALUE )
    {
        DWORD Result = ::WaitForSingleObject( m_hThread, Timeout );
        switch( Result )
        {
        case WAIT_TIMEOUT:
            {
                return( TMIPC_RESULT_TIMEOUT );
            }
            break;
        case WAIT_OBJECT_0:
            {
                CloseHandle( m_hThread );
                m_hThread = INVALID_HANDLE_VALUE;
                m_ThreadId = 0;
                return( TMIPC_RESULT_OK );
            }
            break;
        default:
            ASSERT( 0 );
            break;
        }
    }

    return( TMIPC_RESULT_FAILED );
}

//==============================================================================

void Thread::SleepMs( s32 Ms )
{
    Sleep( Ms );
}

//==============================================================================
//==============================================================================

ScopedLock::ScopedLock( Mutex& Mutex )
:   m_pMutex( &Mutex )
{
    m_pMutex->Lock();
}

//==============================================================================

ScopedLock::~ScopedLock()
{
    m_pMutex->Unlock();
}

//==============================================================================
//==============================================================================

MsgQueue::MsgQueue()
:   m_hSemaphore( 0     ),
    m_pQueue    ( NULL  ),
    m_Allocated ( false ),
    m_Capacity  ( 0     ),
    m_Count     ( 0     ),
    m_iRead     ( 0     ),
    m_iWrite    ( 0     )
{
}

//==============================================================================

MsgQueue::~MsgQueue()
{
    // Destroy the message queue
    if( m_hSemaphore )
    {
        Destroy();
    }
}

//==============================================================================

void MsgQueue::Create( s32 Capacity )
{
    ASSERT( m_hSemaphore == 0 );
    ASSERT( Capacity > 0 );

    // Save the capacity.
    m_Capacity = Capacity;

    // Create the semaphore.
    m_hSemaphore = ::CreateSemaphore( NULL, 0, m_Capacity, NULL );
    ASSERT( m_hSemaphore );

    // Create the mutex.
    m_Mutex.Create();

    // Allocate storage.
    void* pMem = tma::s_Allocate( sizeof( void* ) * m_Capacity );
    m_pQueue = new (pMem) void*[m_Capacity];
    ASSERT( m_pQueue );
    m_Allocated = true;
}

//==============================================================================

void MsgQueue::Create( s32 Capacity, void** pStorage )
{
    ASSERT( m_hSemaphore == 0 );
    ASSERT( Capacity > 0 );

    // Save the capacity.
    m_Capacity = Capacity;

    // Create the semaphore.
    m_hSemaphore = ::CreateSemaphore( NULL, 0, m_Capacity, NULL );
    ASSERT( m_hSemaphore );

    // Create the mutex.
    m_Mutex.Create();

    // Set the queue pointer to the provided storage
    ASSERT( pStorage );
    m_pQueue = pStorage;
    m_Allocated = false;
}

//==============================================================================

void MsgQueue::Destroy()
{
    ASSERT( m_hSemaphore );

    // Close the semaphore
    ::CloseHandle( m_hSemaphore );
    m_hSemaphore = 0;

    // Destroy the mutex
    m_Mutex.Destroy();

    // Delete storage
    if( m_Allocated )
    {
        tma::s_Deallocate( m_pQueue, sizeof( void* ) * m_Capacity );
        m_pQueue = NULL;
        m_Allocated = false;
    }

    // Reset capacity
    m_Capacity = 0;
}

//==============================================================================

bool MsgQueue::IsValid()
{
    return( m_hSemaphore != 0 );
}

//==============================================================================

bool MsgQueue::Send( void* pMessage, bool Block )
{
    bool   Success = false;

    do
    {
        m_Mutex.Lock();

        // Room for another message?
        if( m_Count < m_Capacity )
        {
            // Write to the queue, advance write index and increment count.
            m_pQueue[m_iWrite] = pMessage;
            m_iWrite = (m_iWrite + 1) % m_Capacity;
            m_Count++;

            // Release Seaphore for any waiting threads.
            VERIFY( ::ReleaseSemaphore( m_hSemaphore, 1, NULL ) );

            // Success!
            Success = true;
        }

        m_Mutex.Unlock();

        // If the queue was full and we're blocking then wait 1ms (or longer
        // depending on OS) and try again.
        if( !Success && Block )
        {
            Thread::SleepMs( 1 );
        }
    } while( !Success && Block );

    return( Success );
}

//==============================================================================

bool MsgQueue::Receive( void** ppMessage, bool Block )
{
    ASSERT( ppMessage );

    bool GotMessage = false;

    if( Block )
    {
        // Blocking, wait until we have a valid message.
        while( !GotMessage )
        {
            // Wait on the semaphore to be sure there is a message available.
            ::WaitForSingleObject( m_hSemaphore, INFINITE );

            // Lock the mutex
            m_Mutex.Lock();

            // Check that there is still a message
            if( m_Count > 0 )
            {
                // Get the first message, advance read index and decrement
                // count.
                void* pMessage = m_pQueue[m_iRead];
                m_iRead = (m_iRead + 1) % m_Capacity;
                m_Count--;

                // Save the output.
                *ppMessage = pMessage;
                GotMessage = true;
            }

            // Unlock the mutex.
            m_Mutex.Unlock();
        }
    }
    else
    {
        // Non-Blocking, get a message if one is available.

        // Lock the mutex.
        m_Mutex.Lock();

        // Check that there is still a message.
        if( m_Count > 0 )
        {
            // Decrement the count of the semaphore since we're reading a
            // message.
            ::WaitForSingleObject( m_hSemaphore, 0 );

            // Get the first message, advance read index and decrement count.
            void* pMessage = m_pQueue[m_iRead];
            m_iRead = (m_iRead + 1) % m_Capacity;
            m_Count--;

            // Save the output.
            *ppMessage = pMessage;
            GotMessage = true;
        }

        // Unlock the mutex.
        m_Mutex.Unlock();
    }

    // Return if we got a message or not.
    return GotMessage;
}

//==============================================================================

void* MsgQueue::Receive()
{
    void* pMessage;
    VERIFY( Receive( &pMessage ) );
    return( pMessage );
}

//==============================================================================

s32 MsgQueue::GetCount() const
{
    s32 Count;

    // Get the number of entries in the queue
    Count = m_Count;

    // Count
    return Count;
}

//==============================================================================
} // namespace tmipc
//==============================================================================
