﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "tmipc_task.h"
#include "tmipc_service.h"
#include "tmipc_packet.h"
#include "tmipc_result.h"
#include "tmipc_services_manager.h"
#include "..\DejaInsight.h"

//==============================================================================
namespace tmipc {
//==============================================================================

Task::Task()
:   m_pServicesManager  ( NULL )
,   m_Priority          ( 0 )
,   m_ServiceId         ( 0 )
,   m_TaskId            ( 0 )
,   m_TaskType          ( TaskType_Invalid )
,   m_bNeedPackets      ( false )
,   m_Status            ( StatusPending )
,   m_bExternallyOwned  ( true )
,   m_bCanBePutToSleep  ( true )
{
    m_Event.Create();
}

//==============================================================================

Task::~Task()
{
    m_Event.Destroy();
}

//==============================================================================

void Task::SetCanBePutToSleep( bool bCanBePutToSleep )
{
    m_bCanBePutToSleep = bCanBePutToSleep;
}

//==============================================================================

void Task::SetServicesManager( ServicesManager* pServicesManager )
{
    m_pServicesManager = pServicesManager;
}

//==============================================================================

ServicesManager* Task::GetServicesManager()
{
    return m_pServicesManager ;
}

//==============================================================================

s32 Task::GetStatus() const
{
    return m_Status;
}

//==============================================================================

void Task::SetPriority( s32 Priority )
{
    ASSERT( (Priority >= 0) && (Priority < TMIPC_NUM_PRIORITIES) );
    m_Priority = Priority;
}

//==============================================================================

s32 Task::GetPriority() const
{
    return  m_Priority;
}

//==============================================================================

void Task::SetServiceId( u32 ServiceId )
{
    m_ServiceId = ServiceId;
}

//==============================================================================

u32 Task::GetServiceId() const
{
    return  m_ServiceId;
}

//==============================================================================

void Task::SetTaskId( u32 TaskId )
{
    m_TaskId = TaskId;
}

//==============================================================================

u32 Task::GetTaskId() const
{
    return  m_TaskId;
}

//==============================================================================

void Task::SetTaskType( s32 TaskType )
{
    ASSERT(TaskType >= 0 && TaskType <= tmipc::ProtocolVersion);
    m_TaskType = TaskType;
}

//==============================================================================

s32 Task::GetTaskType() const
{
    return  m_TaskType;
}

//==============================================================================

tmipc::Service* Task::GetService() const
{
    Service* pService = m_pServicesManager->GetService( m_ServiceId );
    return  pService;
}

//==============================================================================

void Task::SetExternallyOwned( bool bState )
{
    m_bExternallyOwned = bState;
}

//==============================================================================

bool Task::GetExternallyOwned() const
{
    return  m_bExternallyOwned;
}

//==============================================================================

bool Task::CanBePutToSleep() const
{
    return  m_bCanBePutToSleep;
}

//==============================================================================

tmipc::Result Task::WaitComplete( s32 Timeout )
{
    DEJA_TRACE( "Task::WaitComplete", "WaitComplete %x", Timeout );

    // Wait for the event to be signaled.
    tmipc::Result Result = static_cast<tmipc::Result>(m_Event.Wait( Timeout ));
    if( Result == TMIPC_RESULT_OK )
    {
        // Completed of Canceled?
        if( m_Status == StatusCanceled )
        {
            Result = TMIPC_RESULT_CANCELED;
        }
    }

    return  Result;
}

//==============================================================================

void Task::Complete()
{
    DEJA_TRACE( "Task::Complete", "Task::Complete" );
//TMA_POWER_TEST_PRINT( "[%s]!!! Service: 0x%08X, Id: %d\n", _BestFunctionName_, GetServiceId(), GetTaskId() );

    // No more packets.
    SetNeedPackets( false );

    // Cache ServiceManager pointer in case this Task is deleted before we use it.
    ServicesManager* pServicesManager = m_pServicesManager;

    // Set status.
    m_Status = StatusComplete;

    // Tick the task.
    pServicesManager->Tick();
}

//==============================================================================

void Task::Cancel()
{
    DEJA_TRACE( "Task::Cancel", "Cancel" );
//TMA_POWER_TEST_PRINT( "[%s]!!! Service: 0x%08X, Id: %d\n", _BestFunctionName_, GetServiceId(), GetTaskId() );

    // No more packets.
    SetNeedPackets( false );

    // Set status.
    m_Status = StatusCanceled;

    // Tick the task.
    m_pServicesManager->Tick();
}

//==============================================================================

void Task::SetNeedPackets( bool bState )
{
//TMA_POWER_TEST_PRINT( "[%s]!!! Service: 0x%08X, Id: %d, Need Packet: '%s'.\n", _BestFunctionName_, GetServiceId(), GetTaskId(), bState ? "Yes" : "No" );
    // Flag that we need packets.
    m_bNeedPackets = bState;

    // Tick the node which will cause packets to be given to tasks needing them.
    m_pServicesManager->Tick();
}

//==============================================================================

bool Task::GetNeedPackets()
{
    return m_bNeedPackets;
}

//==============================================================================

tmipc::Packet* Task::AllocSendPacket( bool bInitiate )
{
    ASSERT( m_pServicesManager );

    // !! DO NOT IGNORE THIS ASSERT !! It means that
    // the new packet will not have a valid task-type
    // identifier and that the packet will be blocked
    // on the receiving end.

    ASSERT( m_TaskType >= 0 && m_TaskType <= tmipc::ProtocolVersion &&
        "The current task does not have a valid type type identifier."
        "If you are creating a new task, be sure to add a new task type identifier to "
        "the 'TaskType' enumeration in tmipc_task.h and initialize the task accordingly.");

    Packet* pPacket = m_pServicesManager->AllocSendPacket();
    ASSERT( pPacket );

    pPacket->SetServiceId( m_ServiceId );
    pPacket->SetTaskId( m_TaskId );
    pPacket->SetTaskType( m_TaskType );
    pPacket->SetInitiate( bInitiate ? 1 : 0 );

    return  pPacket;
}

//==============================================================================

void Task::OnDumpDebugInfo()
{
    NN_SDK_LOG( "[tma][task] GetNeedPackets:        '%s'.\n", GetNeedPackets() ? "Yes" : "No" );
    const char* pStatus{ "Unknown" };
    switch( GetStatus() )
    {
    case StatusPending:  pStatus = "Pending";  break;
    case StatusCanceled: pStatus = "Canceled"; break;
    case StatusComplete: pStatus = "Complete"; break;
    default:             pStatus = "Unknown";  break;
    }
    NN_SDK_LOG( "[tma][task] Status:                '%s'.\n", pStatus );
}

//==============================================================================
} // namespace tmipc
//==============================================================================
