﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "tmipc_packet.h"
#include "tmipc_result.h"
#include "..\tmagent.h"
#include "..\DejaInsight.h"
#include <cstdarg>
#include <nn/util/util_FormatString.h>

//==============================================================================
namespace tmipc {
//==============================================================================

Packet::Packet()
:   m_pBuffer   ( NULL )
,   m_BufferLen ( 0 )
,   m_pHeader   ( NULL )
,   m_pDataStart( NULL )
,   m_pDataEnd  ( NULL )
,   m_pCursor   ( NULL )
{
    DEJA_TRACE( "Packet::Packet", "Packet::Packet" );

    // Allocate storage and intialize all the pointers.
    m_BufferLen = sizeof( Header ) + MAX_PACKET_DATA;
    void* pMem = tma::s_Allocate( sizeof( u8 ) * m_BufferLen );
    m_pBuffer = new (pMem) u8[m_BufferLen];
    m_pHeader    = (Header*)m_pBuffer;
    m_pDataStart = m_pBuffer + sizeof(Header);
    m_pDataEnd   = m_pDataStart + MAX_PACKET_DATA;
    m_pCursor    = m_pDataStart;
    m_pHeader->m_ServiceId  = 0;
    m_pHeader->m_TaskId     = 0;
    m_pHeader->m_TaskType   = TaskType_Invalid;
    m_pHeader->m_Initiate   = 0;
    m_pHeader->m_Version    = 0;
    m_pHeader->m_DataLen    = 0;
    m_pHeader->m_Reserved1  = 0;
    m_pHeader->m_Reserved2  = 0;
    m_pHeader->m_Reserved3  = 0;
    m_pHeader->m_Reserved4  = 0;
}

//==============================================================================

Packet::~Packet()
{
    DEJA_TRACE( "Packet::~Packet", "Packet::~Packet" );
    // Free the storage.
    tma::s_Deallocate( m_pBuffer, m_BufferLen );
    m_BufferLen  = 0;
    m_pBuffer    = NULL;
    m_pHeader    = NULL;
    m_pDataStart = NULL;
    m_pDataEnd   = NULL;
    m_pCursor    = NULL;
}

//==============================================================================

void Packet::SetServiceId( u32 ServiceId )
{
    DEJA_TRACE( "Packet::SetServiceId", "SetServiceId %u", ServiceId );
    ASSERT( m_pHeader );
    m_pHeader->m_ServiceId = ServiceId;
}

//==============================================================================

u32 Packet::GetServiceId() const
{
    ASSERT( m_pHeader );
    return m_pHeader->m_ServiceId;
}

//==============================================================================

void Packet::SetTaskId( u32 TaskId )
{
    DEJA_TRACE( "Packet::SetTaskId", "SetTaskId %p", TaskId );
    ASSERT( m_pHeader );
    m_pHeader->m_TaskId = TaskId;
}

//==============================================================================

u32 Packet::GetTaskId() const
{
    ASSERT( m_pHeader );
    return m_pHeader->m_TaskId;
}

//==============================================================================

void Packet::SetTaskType( s32 TaskType )
{
    DEJA_TRACE( "Packet::SetTaskType", "SetTaskType %d", TaskType );
    ASSERT( m_pHeader );
    m_pHeader->m_TaskType = (s16)TaskType;
}

//==============================================================================

s32 Packet::GetTaskType() const
{
    ASSERT( m_pHeader );
    return m_pHeader->m_TaskType;
}

//==============================================================================

void Packet::SetInitiate( u8 Initiate )
{
    DEJA_TRACE( "Packet::SetInitiate", "SetInitiate %d", Initiate );
    ASSERT( m_pHeader );
    m_pHeader->m_Initiate = Initiate;
}

//==============================================================================

u32 Packet::GetInitiate() const
{
    ASSERT( m_pHeader );
    return m_pHeader->m_Initiate;
}

//==============================================================================

void Packet::SetVersion( u8 Version )
{
    DEJA_TRACE( "Packet::SetVersion", "SetVersion %u", Version );
    ASSERT( m_pHeader );
    m_pHeader->m_Version = Version;
}

//==============================================================================

u32 Packet::GetVersion() const
{
    ASSERT( m_pHeader );
    return m_pHeader->m_Version;
}

//==============================================================================

s32 Packet::GetDataLen()
{
    return m_pHeader->m_DataLen;
}

//==============================================================================

s32 Packet::GetDataLen() const
{
    return m_pHeader->m_DataLen;
}

//==============================================================================

s32 Packet::GetSendLen() const
{
    return (s32)(m_pCursor - m_pBuffer);
}

//==============================================================================

s32 Packet::GetRecvLen() const
{
    return (s32)(m_pDataStart + GetDataLen() - m_pBuffer);
}

//==============================================================================

u8* Packet::GetBuffer()
{
    return m_pBuffer;
}

//==============================================================================

const u8* Packet::GetBuffer() const
{
    return m_pBuffer;
}

//==============================================================================

u8* Packet::GetCursor()
{
    return m_pCursor;
}

//==============================================================================

s32 Packet::ResetCursor()
{
    DEJA_TRACE( "Packet::ResetCursor", "ResetCursor" );
    m_pCursor = m_pDataStart;
    return TMIPC_RESULT_OK;
}

//==============================================================================

s32 Packet::WriteData( const void* pData, s32 nBytes )
{
    DEJA_TRACE( "Packet::WriteData", "WriteData %d bytes", nBytes );
    // Clamp number of bytes to copy and set OVERRUN if necessary.
    s32 result = TMIPC_RESULT_OK;
    s32 nToWrite = nBytes;
    s32 nAvailable = (s32)(m_pDataEnd - m_pCursor);
    if( nBytes > nAvailable )
    {
        nToWrite = nAvailable;
        ASSERT( 0 );
        result = TMIPC_RESULT_PACKET_OVERRUN;
    }

    // Do the write.
    if( nToWrite > 0 )
    {
        memcpy( m_pCursor, pData, nToWrite );
        m_pCursor += nToWrite;
    }

    // Update the header with the new Data Length.
    m_pHeader->m_DataLen = (s32)(m_pCursor - m_pDataStart);

    return result;
}

//==============================================================================

s32 Packet::WriteU8( const u8& V )
{
    return WriteData( &V, sizeof(V) );
}

//==============================================================================

s32 Packet::WriteS8( const s8& V )
{
    return WriteData( &V, sizeof(V) );
}

//==============================================================================

s32 Packet::WriteS16( const s16& V )
{
    return WriteData( &V, sizeof(V) );
}

//==============================================================================

s32 Packet::WriteU16( const u16& V )
{
    return WriteData( &V, sizeof(V) );
}

//==============================================================================

s32 Packet::WriteS32( const s32& V )
{
    return WriteData( &V, sizeof(V) );
}

//==============================================================================

s32 Packet::WriteU32( const u32& V )
{
    return WriteData( &V, sizeof(V) );
}

//==============================================================================

s32 Packet::WriteS64( const s64& V )
{
    return WriteData( &V, sizeof(V) );
}

//==============================================================================

s32 Packet::WriteU64( const u64& V )
{
    return WriteData( &V, sizeof(V) );
}

//==============================================================================

s32 Packet::WriteString( const char* pString )
{
                                            //This "+ 1" is so we include the EOL.
    return WriteData( pString, (s32)strlen(pString) + 1 );
}

//==============================================================================

s32 Packet::WriteString( const std::string& String )
{
    return WriteData( String.c_str(), (s32)String.length() );
}

//==============================================================================

s32 Packet::WriteFormat( const char* pFormat, ... )
{
    va_list Arguments;
    va_start( Arguments, pFormat );
    const s32 kBytesLeft{ MAX_PACKET_DATA - GetDataLen() };
    // +1: The end of string terminator('\0') is not returned to kBytesWritten.
    const s32 kBytesWritten{ nn::util::VSNPrintf( reinterpret_cast<char*>(m_pCursor), kBytesLeft, pFormat, Arguments ) + 1 };
    va_end( Arguments );

    // Force the end of string termination - just in case.
    m_pCursor += kBytesWritten;
    *m_pCursor = '\0';

    // Update the header with the new Data Length.
    m_pHeader->m_DataLen = (s32)(m_pCursor - m_pDataStart);

    return kBytesWritten;
}

//==============================================================================

s32 Packet::ReadData( void* pData, s32 nBytes )
{
    DEJA_TRACE( "Packet::ReadData", "ReadData %d bytes", nBytes );
    // Clamp number of bytes to copy and set OVERRUN if necessary.
    s32 result = TMIPC_RESULT_OK;
    s32 nToRead = nBytes;
    s32 nAvailable = (s32)(m_pDataEnd - m_pCursor);
    if( nBytes > nAvailable )
    {
        nToRead = nAvailable;
        ASSERT( 0 );
        result = TMIPC_RESULT_PACKET_OVERRUN;
    }

    // Do the read.
    if( nToRead > 0 )
    {
        memcpy( pData, m_pCursor, nToRead );
        m_pCursor += nToRead;
    }

    return result;
}

//==============================================================================

s32 Packet::ReadU8( u8& V )
{
    return ReadData( &V, sizeof(V) );
}

//==============================================================================

s32 Packet::ReadS8( s8& V )
{
    return ReadData( &V, sizeof(V) );
}

//==============================================================================

s32 Packet::ReadS16( s16& V )
{
    return ReadData( &V, sizeof(V) );
}

//==============================================================================

s32 Packet::ReadU16( u16& V )
{
    return ReadData( &V, sizeof(V) );
}

//==============================================================================

s32 Packet::ReadS32( s32& V )
{
    return ReadData( &V, sizeof(V) );
}

//==============================================================================

s32 Packet::ReadU32( u32& V )
{
    return ReadData( &V, sizeof(V) );
}

//==============================================================================

s32 Packet::ReadS64( s64& V )
{
    return ReadData( &V, sizeof(V) );
}

//==============================================================================

s32 Packet::ReadU64( u64& V )
{
    return ReadData( &V, sizeof(V) );
}

//==============================================================================

s32 Packet::ReadString( char* pBuffer, s32 bufferLen, s32* pStringLen )
{
    DEJA_TRACE( "Packet::ReadString", "ReadString" );
    s32 result = TMIPC_RESULT_OK;
    s32 nAvailable = (s32)(m_pDataEnd - m_pCursor);
    u8* pWriteTo = (u8*)pBuffer;
    s32 nBytesRead = 0;

    if( bufferLen == 0 )
    {
        result = TMIPC_RESULT_USER_BUFFER_OVERRUN;
    }

    while( bufferLen > nBytesRead )
    {
        // Save this byte
        *pWriteTo = *m_pCursor;

        // Get to the next byte.
        m_pCursor += 1;

        // Update our counters.
        nBytesRead += 1;

        // Done?
        if(*pWriteTo == 0)
        {
            break;
        }

        if( nBytesRead >= nAvailable )
        {
            result = TMIPC_RESULT_PACKET_OVERRUN;
            break;
        }
        if( bufferLen == nBytesRead )
        {
            result = TMIPC_RESULT_USER_BUFFER_OVERRUN;
            break;
        }

        // else
        pWriteTo += 1;
    }



    if( pStringLen != NULL )
    {
        if( result == TMIPC_RESULT_USER_BUFFER_OVERRUN )
        {
            pWriteTo = m_pCursor;
            nBytesRead++;
            while( *pWriteTo != 0 )
            {
                pWriteTo++;
                if( nBytesRead >= nAvailable )
                {
                    result = TMIPC_RESULT_PACKET_OVERRUN;
                    break;
                }
                nBytesRead++;
            }
        }
        *pStringLen = nBytesRead;
    }

    return result;
}

//==============================================================================

s32 Packet::ReadString( std::string& String )
{
    (void)String;
    DEJA_TRACE( "Packet::ReadString", "ReadString std::string" );
    // TODO: Implement strings.
    ASSERT( 0 );
    return TMIPC_RESULT_NOT_IMPLEMENTED;
}

//==============================================================================
} // namespace tmipc
//==============================================================================
