﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

// For access to settings, print serial number at runtime
#include <nn/settings/factory/settings_SerialNumber.h>

#include "tmipc_node_usb_interface.h"
#if defined( TMIPC_TARGET_HORIZON )
#include "tmipc_packet.h"
#include "tmipc_result.h"
#include "../DejaInsight.h"
#include "../thread_tracker.h"
#include "../Version.h"
#include <nn/nn_SystemThreadDefinition.h>
#include <nn/os.h>

//==============================================================================
namespace tmipc {
//==============================================================================

NN_USB_DMA_ALIGN unsigned char              USBInterface::m_ReceiveDataBuffer[m_ReceiveBufferSize];
NN_USB_DMA_ALIGN unsigned char              USBInterface::m_SendDataBuffer[m_SendBufferSize];
NN_USB_DMA_ALIGN unsigned char              USBInterface::m_ReceiveHeaderBuffer[m_HeaderBufferSize];
tmipc::Thread                               USBInterface::m_IndicationPollingThread;        // Indication polling thread.
nn::os::Event                               USBInterface::m_BreakEvent(nn::os::EventClearMode_ManualClear);
NN_OS_ALIGNAS_THREAD_STACK char             USBInterface::m_IndicationPollingThreadStack[m_IndicationPollingThreadStackSize];
bool                                        USBInterface::m_bIsInitialized         = false;
Mutex                                       USBInterface::m_SendMutex;                      // The send function needs to be thread safe.
s32                                         USBInterface::m_ThreadPriority = nn::os::InvalidThreadPriority;
void*                                       USBInterface::m_pEventArgument;
USBInterface::EventCallback                 USBInterface::m_pfEventCallback;
#if ENABLE_BENCHMARK_TESTING
USBInterface::OnPacketReceivedCallback      USBInterface::m_pfOnPacketReceivedCallback = { nullptr };
USBInterface::OnPacketSendCallback          USBInterface::m_pfOnPacketSendCallback = { nullptr };
#endif // ENABLE_BENCHMARK_TESTING

nn::usb::DsClient                           USBInterface::m_DsClient;
nn::usb::DsInterface                        USBInterface::m_DsInterface;
nn::usb::DsEndpoint                         USBInterface::m_DsEndpoints[2];

nn::usb::UsbStringDescriptor                USBInterface::m_LanguageStringDescriptor            = {4,   nn::usb::UsbDescriptorType_String, {0x0409}};
nn::usb::UsbStringDescriptor                USBInterface::m_ManufacturerStringDescriptor        = {18,  nn::usb::UsbDescriptorType_String, {'N', 'i', 'n', 't', 'e', 'n', 'd', 'o'}};
nn::usb::UsbStringDescriptor                USBInterface::m_ProductStringFullSpeedDescriptor    = {40,  nn::usb::UsbDescriptorType_String, {'N', 'i', 'n', 't', 'e', 'n', 'd', 'o', 'S', 'd', 'k', 'D', 'e', 'b', 'u', 'g', 'g', 'e', 'r'}};
nn::usb::UsbStringDescriptor                USBInterface::m_SerialNumberStringDescriptor        = {0,   nn::usb::UsbDescriptorType_String,  {}}; // Will be populated by device serial number
nn::usb::UsbStringDescriptor                USBInterface::m_InterfaceStringDescriptor           = {24,  nn::usb::UsbDescriptorType_String, {'t', 'm', 'a', 'g', 'e', 'n', 't', ' ', 'u', 's', 'b'}};

nn::usb::UsbDeviceDescriptor                USBInterface::m_UsbDeviceDescriptorFs =
{
    nn::usb::UsbDescriptorSize_Device,      // bLength
    nn::usb::UsbDescriptorType_Device,      // bDescriptorType
    0x0110,                                 // bcdUSB 1.1
    0x00,                                   // bDeviceClass
    0x00,                                   // bDeviceSubClass
    0x00,                                   // bDeviceProtocol
    0x40,                                   // bMaxPacketSize0
    0x057e,                                 // idVendor
    0x3000,                                 // idProduct
    0x0100,                                 // bcdDevice
    0x01,                                   // iManufacturer
    0x02,                                   // iProduct
    0x03,                                   // iSerialNumber
    0x01                                    // bNumConfigurations
};


nn::usb::UsbDeviceDescriptor                USBInterface::m_UsbDeviceDescriptorHs =
{
    nn::usb::UsbDescriptorSize_Device,      // bLength
    nn::usb::UsbDescriptorType_Device,      // bDescriptorType
    0x0200,                                 // bcdUSB 2.0
    0x00,                                   // bDeviceClass
    0x00,                                   // bDeviceSubClass
    0x00,                                   // bDeviceProtocol
    0x40,                                   // bMaxPacketSize0
    0x057e,                                 // idVendor
    0x3000,                                 // idProduct
    0x0100,                                 // bcdDevice
    0x01,                                   // iManufacturer
    0x02,                                   // iProduct
    0x03,                                   // iSerialNumber
    0x01                                    // bNumConfigurations
};


nn::usb::UsbDeviceDescriptor                USBInterface::m_UsbDeviceDescriptorSs =
{
    nn::usb::UsbDescriptorSize_Device,      // bLength
    nn::usb::UsbDescriptorType_Device,      // bDescriptorType
    0x0300,                                 // bcdUSB 3.0
    0x00,                                   // bDeviceClass
    0x00,                                   // bDeviceSubClass
    0x00,                                   // bDeviceProtocol
    0x09,                                   // bMaxPacketSize0, SS 512 or 2^9
    0x057e,                                 // idVendor
    0x3000,                                 // idProduct
    0x0100,                                 // bcdDevice
    0x01,                                   // iManufacturer
    0x02,                                   // iProduct
    0x03,                                   // iSerialNumber
    0x01                                    // bNumConfigurations
};


uint8_t                                     USBInterface::m_BinaryObjectStore[] =
{
    0x05,                                   // bLength
    nn::usb::UsbDescriptorType_Bos,         // bDescriptorType
    0x16,0x00,                              // Length of this descriptor and all sub descriptors
    0x02,                                   // Number of device capability descriptors

    // USB 2.0 extension
    0x07,                                   // bLength
    nn::usb::UsbDescriptorType_DeviceCapability, // bDescriptorType
    0x02,                                   // USB 2.0 extension capability type
    0x02,0x00,0x00,0x00,                    // Supported device level features: LPM support

    // SuperSpeed device capability
    0x0A,                                   // bLength
    nn::usb::UsbDescriptorType_DeviceCapability,  // bDescriptorType
    0x03,                                   // SuperSpeed device capability type
    0x00,                                   // Supported device level features
    0x0e,0x00,                              // Speeds supported by the device : SS, HS
    0x03,                                   // Functionality support
    0x00,                                   // U1 Device Exit latency
    0x00,0x00                               // U2 Device Exit latency
};


nn::usb::UsbInterfaceDescriptor             USBInterface::m_UsbInterfaceDescriptor =
{
    nn::usb::UsbDescriptorSize_Interface,   // bLength
    nn::usb::UsbDescriptorType_Interface,   // bDescriptorType
    0,                                      // bInterfaceNumber
    0,                                      // bAlternateSetting
    2,                                      // bNumEndpoints
    0xff,                                   // bInterfaceClass
    0xff,                                   // bInterfaceSubClass
    0xff,                                   // bInterfaceProtocol
    4,                                      // iInterface
};


nn::usb::UsbEndpointDescriptor              USBInterface::m_UsbEndpointDescriptorsFs[2] =
{
    {
        nn::usb::UsbDescriptorSize_Endpoint,            // bLength
        nn::usb::UsbDescriptorType_Endpoint,            // bDescriptorType
        0x81,                                           // bEndpointAddress
        nn::usb::UsbEndpointAttributeMask_XferTypeBulk, // bmAttributes
        64,                                             // wMaxPacketSize
        0                                               // bInterval
    },
    {
        nn::usb::UsbDescriptorSize_Endpoint,            // bLength
        nn::usb::UsbDescriptorType_Endpoint,            // bDescriptorType
        0x01,                                           // bEndpointAddress
        nn::usb::UsbEndpointAttributeMask_XferTypeBulk, // bmAttributes
        64,                                             // wMaxPacketSize
        0                                               // bInterval
    },
};


nn::usb::UsbEndpointDescriptor              USBInterface::m_UsbEndpointDescriptorsHs[2] =
{
    {
        nn::usb::UsbDescriptorSize_Endpoint,            // bLength
        nn::usb::UsbDescriptorType_Endpoint,            // bDescriptorType
        0x81,                                           // bEndpointAddress
        nn::usb::UsbEndpointAttributeMask_XferTypeBulk, // bmAttributes
        512,                                            // wMaxPacketSize
        0                                               // bInterval
    },
    {
        nn::usb::UsbDescriptorSize_Endpoint,            // bLength
        nn::usb::UsbDescriptorType_Endpoint,            // bDescriptorType
        0x01,                                           // bEndpointAddress
        nn::usb::UsbEndpointAttributeMask_XferTypeBulk, // bmAttributes
        512,                                            // wMaxPacketSize
        0                                               // bInterval
    },
};


nn::usb::UsbEndpointDescriptor              USBInterface::m_UsbEndpointDescriptorsSs[2] =
{
    {
        nn::usb::UsbDescriptorSize_Endpoint,            // bLength
        nn::usb::UsbDescriptorType_Endpoint,            // bDescriptorType
        0x81,                                           // bEndpointAddress
        nn::usb::UsbEndpointAttributeMask_XferTypeBulk, // bmAttributes
        1024,                                           // wMaxPacketSize
        0                                               // bInterval
    },
    {
        nn::usb::UsbDescriptorSize_Endpoint,            // bLength
        nn::usb::UsbDescriptorType_Endpoint,            // bDescriptorType
        0x01,                                           // bEndpointAddress
        nn::usb::UsbEndpointAttributeMask_XferTypeBulk, // bmAttributes
        1024,                                           // wMaxPacketSize
        0                                               // bInterval
    },
};


// Use this endpoint companion descriptor for both endpoints :)
nn::usb::UsbEndpointCompanionDescriptor     USBInterface::m_UsbEndpointCompanionDescriptor =
{
    0x06,                                       // bLength
    nn::usb::UsbDescriptorType_EndpointCompanion, // bDescriptorType
    15,                                         // bMaxBurst
    0,                                          // bmAttributes
    0                                           // wBytesPerInterval
};


// ============================================================================
// Listen thread.
s32 USBInterface::IndicationPollingThread( void* lpThis )
{
    DEJA_TRACE( "USBInterface::IndicationPollingThread", "Starting thread" );
TMA_POWER_TEST_PRINT( "[%s] !!! Start.\n", _BestFunctionName_ );

    nn::usb::UsbState  state;

    nn::os::SystemEventType     *pStateChangeEvent;

    nn::os::MultiWaitType        MultiWait;
    nn::os::MultiWaitHolderType  StateChangeHolder;
    nn::os::MultiWaitHolderType  BreakHolder;

    // Get USB state change event
    pStateChangeEvent = m_DsClient.GetStateChangeEvent();

    // Prepare multi-wait
    nn::os::InitializeMultiWait( &MultiWait );
    nn::os::InitializeMultiWaitHolder( &StateChangeHolder, pStateChangeEvent );
    nn::os::LinkMultiWaitHolder( &MultiWait, &StateChangeHolder );
    nn::os::InitializeMultiWaitHolder( &BreakHolder, m_BreakEvent.GetBase() );
    nn::os::LinkMultiWaitHolder( &MultiWait, &BreakHolder );

    // Poll for an Indication status change.
    nn::usb::UsbCtrlRequest     UsbControlRequest;
    uint32_t                    IndicationBitmask = { 0 };
    nn::Result                  NnResult;
    bool                        bQuit = { false };
    while( !bQuit )
    {
        nn::os::MultiWaitHolderType *pHolder = nn::os::WaitAny(&MultiWait);

        // Have we been asked to terminate this thread?
        bQuit = pHolder == &BreakHolder;
        if( !bQuit )
        {
            // else, there is USB state change
            nn::os::ClearSystemEvent(pStateChangeEvent);
            m_DsClient.GetState(&state);

            switch(state)
            {
            case nn::usb::UsbState_Detached:
            case nn::usb::UsbState_Suspended:
                DEJA_CONTEXT( "USBInterface::IndicationPollingThread", "Detected USB DETACHED / SUSPENDED indication\n" );
                if( m_pfEventCallback != nullptr )
                {
                    m_pfEventCallback( tmipc::NodeUSB::USBEvent::Reset, m_pEventArgument );
                }
                break;

            case nn::usb::UsbState_Configured:
                DEJA_CONTEXT( "USBInterface::IndicationPollingThread", "Found USB CONFIGURED indication\n" );
                if( m_pfEventCallback != nullptr )
                {
                    m_pfEventCallback( tmipc::NodeUSB::USBEvent::Start, m_pEventArgument );
                }
                break;

            default:
                // Ignore other states
                break;
            }
        }
    }

    // Clear the event.
    m_BreakEvent.Clear();

    // Destroy multi-wait
    nn::os::UnlinkMultiWaitHolder( &StateChangeHolder );
    nn::os::FinalizeMultiWaitHolder( &StateChangeHolder );
    nn::os::UnlinkMultiWaitHolder( &BreakHolder );
    nn::os::FinalizeMultiWaitHolder( &BreakHolder );
    nn::os::FinalizeMultiWait( &MultiWait );

TMA_POWER_TEST_PRINT( "[%s] !!! Finish.\n", _BestFunctionName_ );
    DEJA_TRACE( "USBInterface::IndicationPollingThread", "IndicationPollingThread Stopped" );
    return 0;
}

// Sets the thread priority.
void USBInterface::SetThreadPriority(s32 ThreadPriority)
{
    m_ThreadPriority = ThreadPriority;
}

// ============================================================================
// Gets the thread priority.
s32 USBInterface::GetThreadPriority()
{
    return m_ThreadPriority;
}

// ============================================================================
// Initialize the USB interface.
tmipc::Result USBInterface::Initialize( s32 ThreadPriority )
{
    nn::Result NnResult;
    tmipc::Result Result = { tmipc::TMIPC_RESULT_OK };

TMA_POWER_TEST_PRINT( "[%s] !!! Start.\n", _BestFunctionName_ );
    DEJA_TRACE( "USBInterface::Initialize", "USBInterface Create the USB interface" );

    // Create the Send Mutex
    m_SendMutex.Create();

    SetThreadPriority(ThreadPriority);

    // Initialize the underlying USB device.
    if( Result == tmipc::TMIPC_RESULT_OK )
    {
        // add serial number
        {
            // Print serial number to sring table
            // Get the hardware serial number.
            nn::settings::factory::SerialNumber SerialNumber;
            SerialNumber.string[0] = '\0';

            // The Jetson-TK2 will print warnings about the GetSerialNumber call, and
            // will "become blocking" if a USB connection to the TK2 is reconnected
            // (after having been initially connected).
            const nn::Result NnResult = nn::settings::factory::GetSerialNumber(&SerialNumber);

            if( NnResult.IsFailure() || (SerialNumber.string[0] == '\0') )
            {
                strcpy(SerialNumber.string, "Corrupted S/N");
            }

            uint16_t *pDst  = &m_SerialNumberStringDescriptor.wData[0];
            uint8_t *pSrc   = reinterpret_cast<uint8_t*>(SerialNumber.string);
            uint8_t count = 0;

            while (*pSrc)
            {
                *pDst++ = static_cast<uint16_t>(*pSrc++);
                count++;
            }

            m_SerialNumberStringDescriptor.bLength = 2 + (count * 2);
        }

        NnResult = m_DsClient.Initialize(nn::usb::ComplexId_Tegra21x);

        if (NnResult.IsSuccess())
        {
            // Clear all descriptor data from device in case someone has used
            // it before.
            // m_DsClient.ClearDeviceData();

            // Add string descriptors to device, since we know the order we
            // added them, we have already primed the device descriptors with
            // the correct string indicies.
            uint8_t stringIndex;

            m_DsClient.AddUsbStringDescriptor(&stringIndex, &m_LanguageStringDescriptor);
            m_DsClient.AddUsbStringDescriptor(&stringIndex, &m_ManufacturerStringDescriptor);
            m_DsClient.AddUsbStringDescriptor(&stringIndex, &m_ProductStringFullSpeedDescriptor);
            m_DsClient.AddUsbStringDescriptor(&stringIndex, &m_SerialNumberStringDescriptor);
            m_DsClient.AddUsbStringDescriptor(&stringIndex, &m_InterfaceStringDescriptor);

            // Add device descriptors, one for each bus speed.
            m_DsClient.SetUsbDeviceDescriptor(&m_UsbDeviceDescriptorFs, nn::usb::UsbDeviceSpeed_Full);
            m_DsClient.SetUsbDeviceDescriptor(&m_UsbDeviceDescriptorHs, nn::usb::UsbDeviceSpeed_High);
            m_DsClient.SetUsbDeviceDescriptor(&m_UsbDeviceDescriptorSs, nn::usb::UsbDeviceSpeed_Super);

            // Binary Object Store descriptor is required for 3.0 host.
            m_DsClient.SetBinaryObjectStore(m_BinaryObjectStore, sizeof(m_BinaryObjectStore));
        }

        Result = NnResult.IsSuccess() ? tmipc::TMIPC_RESULT_OK : tmipc::TMIPC_RESULT_CONNECT_FAILED;
    }

    // Initialize the underlying USB interface.
    if( Result == tmipc::TMIPC_RESULT_OK )
    {
        NnResult = m_DsInterface.Initialize(&m_DsClient, m_UsbInterfaceDescriptor.bInterfaceNumber);

        if (NnResult.IsSuccess())
        {
            // For each speed, append descriptor in the order to appear in the
            // configuration descriptor stream, this allows any descriptors to appear
            // in the stream as needed :)
            m_DsInterface.AppendConfigurationData(nn::usb::UsbDeviceSpeed_Full, &m_UsbInterfaceDescriptor, sizeof(nn::usb::UsbInterfaceDescriptor));
            m_DsInterface.AppendConfigurationData(nn::usb::UsbDeviceSpeed_Full, &m_UsbEndpointDescriptorsFs[0], sizeof(nn::usb::UsbEndpointDescriptor));
            m_DsInterface.AppendConfigurationData(nn::usb::UsbDeviceSpeed_Full, &m_UsbEndpointDescriptorsFs[1], sizeof(nn::usb::UsbEndpointDescriptor));

            m_DsInterface.AppendConfigurationData(nn::usb::UsbDeviceSpeed_High, &m_UsbInterfaceDescriptor, sizeof(nn::usb::UsbInterfaceDescriptor));
            m_DsInterface.AppendConfigurationData(nn::usb::UsbDeviceSpeed_High, &m_UsbEndpointDescriptorsHs[0], sizeof(nn::usb::UsbEndpointDescriptor));
            m_DsInterface.AppendConfigurationData(nn::usb::UsbDeviceSpeed_High, &m_UsbEndpointDescriptorsHs[1], sizeof(nn::usb::UsbEndpointDescriptor));

            m_DsInterface.AppendConfigurationData(nn::usb::UsbDeviceSpeed_Super, &m_UsbInterfaceDescriptor, sizeof(nn::usb::UsbInterfaceDescriptor));
            m_DsInterface.AppendConfigurationData(nn::usb::UsbDeviceSpeed_Super, &m_UsbEndpointDescriptorsSs[0], sizeof(nn::usb::UsbEndpointDescriptor));
            m_DsInterface.AppendConfigurationData(nn::usb::UsbDeviceSpeed_Super, &m_UsbEndpointCompanionDescriptor, sizeof(nn::usb::UsbEndpointCompanionDescriptor));
            m_DsInterface.AppendConfigurationData(nn::usb::UsbDeviceSpeed_Super, &m_UsbEndpointDescriptorsSs[1], sizeof(nn::usb::UsbEndpointDescriptor));
            m_DsInterface.AppendConfigurationData(nn::usb::UsbDeviceSpeed_Super, &m_UsbEndpointCompanionDescriptor, sizeof(nn::usb::UsbEndpointCompanionDescriptor));
        }

        Result = NnResult.IsSuccess() ? tmipc::TMIPC_RESULT_OK : tmipc::TMIPC_RESULT_CONNECT_FAILED;
    }

    // Initialize the USB "in" endpoint (aka: the TargetManager's write endpoint).
    if( Result == tmipc::TMIPC_RESULT_OK )
    {
        NnResult = m_DsEndpoints[USB_WRITE_ENDPOINT_INDEX].Initialize( &m_DsInterface, 0x81 );
        Result = NnResult.IsSuccess() ? tmipc::TMIPC_RESULT_OK : tmipc::TMIPC_RESULT_CONNECT_FAILED;
    }

    // Initialize the Target's USB "out" endpoint (aka: the TargetManager's read endpoint).
    if( Result == tmipc::TMIPC_RESULT_OK )
    {
        NnResult = m_DsEndpoints[USB_READ_ENDPOINT_INDEX].Initialize( &m_DsInterface, 0x01 );
        Result = NnResult.IsSuccess() ? tmipc::TMIPC_RESULT_OK : tmipc::TMIPC_RESULT_CONNECT_FAILED;
    }

    // Start the indication polling thread.
    if( Result == tmipc::TMIPC_RESULT_OK )
    {
        Result = m_IndicationPollingThread.Start( IndicationPollingThread, nullptr, m_IndicationPollingThreadStack, m_IndicationPollingThreadStackSize, NN_SYSTEM_THREAD_PRIORITY(tma, TmaUsbIndication), NN_SYSTEM_THREAD_NAME(tma, TmaUsbIndication) );
        if( Result == tmipc::TMIPC_RESULT_OK )
        {
            // Register the thread.
            tma::ThreadTracker::RegisterThread( m_IndicationPollingThread.GetThreadType(), tma::ThreadTracker::ThreadId::TmipcListen );
        }
    }

    // Enable the USB interface.
    if( Result == tmipc::TMIPC_RESULT_OK )
    {
        NnResult = m_DsInterface.Enable();
        Result = NnResult.IsSuccess() ? tmipc::TMIPC_RESULT_OK : tmipc::TMIPC_RESULT_CONNECT_FAILED;
    }

    // Enable the USB device.
    if( Result == tmipc::TMIPC_RESULT_OK )
    {
        NnResult = m_DsClient.EnableDevice();
        Result = NnResult.IsSuccess() ? tmipc::TMIPC_RESULT_OK : tmipc::TMIPC_RESULT_CONNECT_FAILED;
    }

    // Should this be an ASSERT - or a flag that says the USB interface was
    // not initialized property?
    ASSERT( Result == tmipc::TMIPC_RESULT_OK );
    m_bIsInitialized = Result == tmipc::TMIPC_RESULT_OK;
    if( Result != tmipc::TMIPC_RESULT_OK )
    {
        DEJA_TRACE( "USBInterface::Initialize", "Failed to create the USB interface.  tmipc::Result == %d", Result );
        // If the USB interface failed to Initialize, then Finalize it.
        TMIPC_ERROR_LOG( "USB Interface creation failed.\n" );
        Finalize();
    }

TMA_POWER_TEST_PRINT( "[%s] !!! Finish - Result: %d.\n", _BestFunctionName_, Result);
    return Result;
}

// ============================================================================
// Finalize the USB interface.
tmipc::Result USBInterface::Finalize()
{
    DEJA_TRACE( "USBInterface::Finalize", "Finalize the USB interface.");
TMA_POWER_TEST_PRINT( "[%s] !!! Start.\n", _BestFunctionName_ );

    tmipc::Result Result = { tmipc::TMIPC_RESULT_OK };

    ASSERT( m_bIsInitialized );

    // Ask the IndicationPollingThread to finish
    m_BreakEvent.Signal();

    // Wait for the IndicationPollingThread thread to finish.
    VERIFY( m_IndicationPollingThread.Join() == tmipc::Result::TMIPC_RESULT_OK );
    // Unregister the thread.
    tma::ThreadTracker::UnregisterThread( tma::ThreadTracker::ThreadId::TmipcListen );

    // Disable the device.
    if (Result == tmipc::TMIPC_RESULT_OK)
    {
        nn::Result NnResult = m_DsClient.DisableDevice();
        Result = NnResult.IsSuccess() ? tmipc::TMIPC_RESULT_OK : tmipc::TMIPC_RESULT_CONNECT_FAILED;
    }

    // Disable the Interface.
    if (Result == tmipc::TMIPC_RESULT_OK)
    {
        nn::Result NnResult = m_DsInterface.Disable();
        Result = NnResult.IsSuccess() ? tmipc::TMIPC_RESULT_OK : tmipc::TMIPC_RESULT_CONNECT_FAILED;
    }

    // Finalize the Target's USB "out" endpoint (aka: the TargetManager's read endpoint).
    if (Result == tmipc::TMIPC_RESULT_OK)
    {
        nn::Result NnResult = m_DsEndpoints[USB_READ_ENDPOINT_INDEX].Finalize();
        Result = NnResult.IsSuccess() ? tmipc::TMIPC_RESULT_OK : tmipc::TMIPC_RESULT_CONNECT_FAILED;
    }

    // Finalize the USB "in" endpoint (aka: the TargetManager's write endpoint).
    if( Result == tmipc::TMIPC_RESULT_OK )
    {
        nn::Result NnResult = m_DsEndpoints[USB_WRITE_ENDPOINT_INDEX].Finalize();
        Result = NnResult.IsSuccess() ? tmipc::TMIPC_RESULT_OK : tmipc::TMIPC_RESULT_CONNECT_FAILED;
    }

    // Finalize the underlying USB interface.
    if (Result == tmipc::TMIPC_RESULT_OK)
    {
        nn::Result NnResult = m_DsInterface.Finalize();
        Result = NnResult.IsSuccess() ? tmipc::TMIPC_RESULT_OK : tmipc::TMIPC_RESULT_CONNECT_FAILED;
    }

    // Finalize the USB interface.
    if( Result == tmipc::TMIPC_RESULT_OK )
    {
        nn::Result NnResult = m_DsClient.Finalize();
        Result = NnResult.IsSuccess() ? tmipc::TMIPC_RESULT_OK : tmipc::TMIPC_RESULT_CONNECT_FAILED;
    }

    // Destroy the send mutex.
    m_SendMutex.Destroy();
    if( Result != tmipc::TMIPC_RESULT_OK )
    {
        Result = tmipc::TMIPC_RESULT_CONNECT_FAILED;
    }
    m_bIsInitialized = false;

TMA_POWER_TEST_PRINT( "[%s] !!! Finish - Result: %d.\n", _BestFunctionName_, Result);
    return Result;
}

// ============================================================================
// Sets the callback information.
void USBInterface::SetCallbackInformation( EventCallback pfEventCallback, void* Arguments )
{
    // Record the event callback information.
    m_pfEventCallback = pfEventCallback;
    m_pEventArgument  = Arguments;
}

// ============================================================================
// Aborts blocking calls.
void USBInterface::AbortBlockingCalls()
{
    m_DsEndpoints[USB_WRITE_ENDPOINT_INDEX].Cancel();
    m_DsEndpoints[USB_READ_ENDPOINT_INDEX].Cancel();
}

// ============================================================================
// Reads data from the USB connection.
// Returns: TMIPC_RESULT_OK: success; non-TMIPC_RESULT_OK: an error occurred.
tmipc::Result USBInterface::Read( tmipc::Packet* pPacket )
{
    DEJA_CONTEXT( "USBInterface::Read", "USB Read\n" );
    tmipc::Result Result = { tmipc::TMIPC_RESULT_OK };

    if( !m_bIsInitialized )
    {
        Result = tmipc::TMIPC_RESULT_FAILED;
        TMIPC_ERROR_LOG("USBInterface::Read() while not initialized\n");
    }

    // Validate the parameters.
    if( pPacket == nullptr )
    {
        Result = tmipc::TMIPC_RESULT_FAILED;
        TMIPC_ERROR_LOG( "Bad packet data.\n" );
    }

    // SIGLO-67098: A Fatal Error will frequently occur when closing Target Manager with an EDEV connected.
    // This is not really a fix, but it makes the code more robust that it was.  What seems to be happening
    // is that the Packet header is "successfully" retrieved, but the DataLength is way too large to fit in
    // the m_ReceiveHeaderBuffer - which makes the code crash (writing past the end of an array).  One idea
    // was suggested to test the DataLength to make sure it is valid, before calling PostBuffer(...), which
    // works, but that does not completely solve the problem.
    // I have implemented the DataLength check, and also a "retry mechanism" to detect when this occurs and
    // try again.  The code seems to recover from this error now, but I have seen some new/other issues.
    // Problems left to solve:
    // 1) Why did the nn::Result return code get back a "success" from the PostBuffer(), if it has bad data?
    // 2) What happens if the header packet is still garbage but also has a DataLength that fits in the buffer?
    //    a) The system may still end up crashing, or worse - block waiting to read data that will not be sent.
    // 3) I have seen other crashes:
    //    a) The memcpy( pHeader, m_ReceiveHeaderBuffer, sizeof(*pHeader) ); call got an access violation.
    //       This implies that the Packet was deallocated.  A fix needs to be investigated.
    //    b) I saw the following error messages printed in the TTY:
    //       [usb:ds] eror 1:33:42.356| HandleCmplCodeBabbleDetectedErr(@2666)TrbCmplCode_PrimePipeReceived
    //       [usb:ds] info 1:33 : 42.365 | SetEpHalt(@1624) cmpl code error : EpIndex 2
    //       I do not know what these mean - I will have to mention them in the Jira ticket.
    bool bBadDataLengthTryAgain{ false };
    do
    {
        // Assume the Packet's DataLength value is *not* going to fail.
        bBadDataLengthTryAgain = false;

        // Read the packet header.
        u32 BytesRead = 0;
        if( Result == tmipc::TMIPC_RESULT_OK )
        {
            tmipc::Packet::Header* pHeader = reinterpret_cast<tmipc::Packet::Header*>(pPacket->GetBuffer());
            const nn::Result NnResult = m_DsEndpoints[USB_READ_ENDPOINT_INDEX].PostBuffer( &BytesRead, m_ReceiveHeaderBuffer, sizeof(*pHeader) );
            if( NnResult.IsSuccess() )
            {
                memcpy( pHeader, m_ReceiveHeaderBuffer, sizeof(*pHeader) );
            }
            else
            {
                Result = tmipc::TMIPC_RESULT_FAILED;
//                TMIPC_ERROR_LOG( "Packet header read failed.\n" );
            }
        }

        // Make sure a complete Packet header was read (at least byte size wise).
        if( Result == tmipc::TMIPC_RESULT_OK )
        {
            if( BytesRead != sizeof(tmipc::Packet::Header) )
            {
                Result = tmipc::TMIPC_RESULT_FAILED;
                TMIPC_ERROR_LOG( "Packet header size mismatch.\n" );
            }
        }

        // Read the packet data, if there is data to be read.
        if( Result == tmipc::TMIPC_RESULT_OK )
        {
            const s32 DataLength{ pPacket->GetDataLen() };
            if( DataLength > 0)
            {
                // Make sure a buffer overflow will not occur.
                // NOTE: this will *not* protect against an invalid packet, whose data size happens to fit in the buffer.
                if( DataLength < sizeof(m_ReceiveDataBuffer) )
                {
                    const nn::Result NnResult = m_DsEndpoints[USB_READ_ENDPOINT_INDEX].PostBuffer( &BytesRead, m_ReceiveDataBuffer, static_cast<u32>(DataLength) );
                    if( NnResult.IsSuccess() )
                    {
                        u8 *pLocalBuffer = pPacket->GetBuffer() + sizeof(tmipc::Packet::Header);
                        if( BytesRead == DataLength )
                        {
                            memcpy( pLocalBuffer, m_ReceiveDataBuffer, static_cast<u32>(DataLength) );
                        }
                        else
                        {
                            Result = tmipc::TMIPC_RESULT_FAILED;
//                            TMIPC_ERROR_LOG( "Packet data size mismatch.\n" );
//NN_SDK_LOG( "!!!! Additional information: %u != %u bytes !!!\n", BytesRead, DataLength );
                        }
                    }
                    else
                    {
                        TMIPC_ERROR_LOG( "Read packet data failed.\n" );
                        Result = tmipc::TMIPC_RESULT_FAILED;
                    }
                }
                else
                {
                    // See the comment "SIGLO-67098" at the beginning of the [outer] do-while loop.
//                    NN_SDK_LOG( "[tma] USBInterface::Read: this: %p. m_pBuffer: %p.\n", pPacket, pPacket->GetBuffer() );
                    NN_SDK_LOG( "[tma] Invalid packet data length: 0 < %d <= %d.\n", DataLength, sizeof(m_ReceiveDataBuffer) );
                    bBadDataLengthTryAgain = true;
                }
            }
        }
    } while (bBadDataLengthTryAgain);

#if ENABLE_BENCHMARK_TESTING
    // Do *not* attempt to use the Packet, if an abort was issued.
    if( Result == TMIPC_RESULT_OK )
    {
        // Call the "receive callback".
        if( m_pfOnPacketReceivedCallback != nullptr )
        {
            m_pfOnPacketReceivedCallback( *pPacket );
        }
    }
#endif // ENABLE_BENCHMARK_TESTING

    return Result;
} // NOLINT(readability/fn_size)

// ============================================================================
// Writes data to the USB connection.
// Returns: TMIPC_RESULT_OK: success; non-TMIPC_RESULT_OK: an error occurred.
tmipc::Result USBInterface::Write( tmipc::Packet* pPacket )
{
    tmipc::Result Result = { tmipc::TMIPC_RESULT_OK };

    if( !m_bIsInitialized )
    {
        Result = tmipc::TMIPC_RESULT_FAILED;
        TMIPC_ERROR_LOG("USBInterface::Write() while not initialized\n");
    }

    // Validate the parameters.
    if( pPacket == nullptr )
    {
        Result = tmipc::TMIPC_RESULT_FAILED;
        TMIPC_ERROR_LOG( "Bad packet data.\n" );
    }

    // Lock the send mutex.
    m_SendMutex.Lock();

    // Send the whole packet all at once.  This only works for USB when sending
    // from Target to Target Manager.  Target Manager to Target *must* use the
    // double send/receive. :(
    u32 BytesSent = 0;
    if( Result == tmipc::TMIPC_RESULT_OK )
    {
        const u32 SendLength = pPacket->GetSendLen();
        if( SendLength <= m_SendBufferSize )
        {
            const u8  *pBuffer   = pPacket->GetBuffer();
            memcpy( m_SendDataBuffer, pBuffer, SendLength );

#if ENABLE_BENCHMARK_TESTING
            // Call the "send callback".
            if( m_pfOnPacketSendCallback != nullptr )
            {
                m_pfOnPacketSendCallback( *pPacket );
            }
#endif // ENABLE_BENCHMARK_TESTING

            nn::Result NnResult = m_DsEndpoints[USB_WRITE_ENDPOINT_INDEX].PostBuffer( &BytesSent, m_SendDataBuffer, SendLength );
            if( NnResult.IsFailure() )
            {
                Result = tmipc::TMIPC_RESULT_FAILED;
                TMIPC_ERROR_LOG( "Send [whole] packet failed.\n" );
            }
        }
        else
        {
            Result = tmipc::TMIPC_RESULT_FAILED;
            NN_SDK_LOG( "Send [whole] packet is too large for the 'send buffer': %d <= %d.\n", SendLength, m_SendBufferSize );
        }
    }

    // Unlock the send mutex.
    m_SendMutex.Unlock();

    return Result;
} // NOLINT(readability/fn_size)

// ============================================================================
// Sends a Beacon request back to the remote connection.
tmipc::Result USBInterface::SendBeaconPacket( tmipc::Packet* pPacket )
{
    tmipc::Result Result = tmipc::TMIPC_RESULT_OK;

    // Retrieve the Beacon response.
    char YamlString[256] = "";
    if( Result == tmipc::TMIPC_RESULT_OK )
    {
        if( !tma::GenerateBeaconResponse( YamlString, sizeof(YamlString), "USB", false ) )
        {
            TMIPC_ERROR_LOG( "Generate Beacon Response failed.\n" );
            Result = tmipc::TMIPC_RESULT_CONNECT_FAILED;
        }
    }

    // Reset the packet cursor, for the response packet.
    if( Result == tmipc::TMIPC_RESULT_OK )
    {
        pPacket->ResetCursor();
        pPacket->WriteString( YamlString );
        Result = Write( pPacket );
        if( Result != tmipc::TMIPC_RESULT_OK )
        {
            TMIPC_ERROR_LOG( "Send Beacon Packet failed.\n" );
        }
    }

    return Result;
}

//==============================================================================
#if ENABLE_BENCHMARK_TESTING
// Sets/clears the callback invoked when any packet is received.
void USBInterface::SetOnPacketReceivedCallback( OnPacketReceivedCallback pfOnPacketReceivedCallback )
{
    m_pfOnPacketReceivedCallback = pfOnPacketReceivedCallback;
}

// Sets/clears the callback invoked when right before a packet is sent.
void USBInterface::SetOnPacketSendCallback( OnPacketSendCallback pfOnPacketSendCallback )
{
    m_pfOnPacketSendCallback = pfOnPacketSendCallback;
}
#endif // ENABLE_BENCHMARK_TESTING

// ============================================================================
} // namespace tmipc
// ============================================================================

#endif // TMIPC_TARGET_HORIZON
