﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include "tmipc_node.h"

#if defined( TMIPC_TARGET_HORIZON )
#include <nn/usb/usb_Device.h>
#endif // TMIPC_TARGET_HORIZON

//==============================================================================
namespace tmipc {
//==============================================================================

#if defined( TMIPC_TARGET_HORIZON ) && defined( NN_DETAIL_TMA_NX_RELATED_HARDWARE )

// Node connection, using USB.

class NodeUSB : public Node
{
public:

    // USB callback event parameter.
    enum class USBEvent : uint32_t
    {
        Invalid = 0,        // An invalid event state.
        Reset,              // The USB was internally reset.  Stop Read and Write operations.
        Start,              // The USB started up.  Safe for Read and Write operations.
        Quit                // The USB was shut down (Finalize()'d).
    };

private:
    typedef Node BaseClass;

        // The thread stacks send and receive.
static  NN_OS_ALIGNAS_THREAD_STACK u8 m_SendThreadStack[TMIPC_STACK_SIZE];
static  NN_OS_ALIGNAS_THREAD_STACK u8 m_ReceiveThreadStack[TMIPC_STACK_SIZE];
static  Thread      m_SendThread;           // Threads for send and receive.
static  Thread      m_ReceiveThread;        //
        MsgQueue    m_SendQueue;            // The send queue (Packets).

        bool        m_bAbortedBlockingCalls;// Only true when a connection occurs - prevents the send thread from quitting.
        bool        m_bIsConnected;         // Is there a USB connection? TargetManager might not be connected.

private:

        // Send thread.
static  s32     SendThread      ( void* lpThis );

        // Receive thread.
static  s32     ReceiveThread   ( void* lpThis );

        // Starts the Send/Receive threads.
        void    StartThreads    ();

        // Stops the Send/Receive threads.
        void    StopThreads     ();

        // Callback function from the NodeUSBInterface.
static  void    EventCallback   ( USBEvent Event, void* lpThis );

public:

        // Initialize the USB interface.  ThreadPriority will persist until
        // Finalize() is called - only then can ThreadPriority be changed.
static  Result  Initialize      ( s32 ThreadPriority );
        // Finalize the USB interface.
static  Result  Finalize        ();

                NodeUSB         ();
virtual         ~NodeUSB        ();

virtual Result  Disconnect      ();

virtual bool    IsConnected     ();
virtual Result  Send            ( Packet* pPacket );
};

#else // defined( TMIPC_TARGET_HORIZON ) && defined( NN_DETAIL_TMA_NX_RELATED_HARDWARE )

// This is for non-Horizon compiles.  At this point, that only includes the
// Simulation Instance (NN_TMA_SIM_INSTANCE). This "dummy version" was made
// to prevent build errors.
class NodeUSB : public Node
{
virtual bool IsConnected()
{
    return false;
}
virtual Result Disconnect()
{
    return TMIPC_RESULT_FAILED;
}

virtual Result Send( Packet* pPacket )
{
    (void)pPacket;
    return TMIPC_RESULT_FAILED;
}
};

#endif // defined( TMIPC_TARGET_HORIZON ) && defined( NN_DETAIL_TMA_NX_RELATED_HARDWARE )

//==============================================================================
} // namespace tmipc
//==============================================================================
