﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <Plug.h>
#include <ForeignMemBuffer.h>
#include "tmipc_defines.h"
#include "tmipc_packet.h"
#include "tmipc_thread.h"

//==============================================================================
namespace tmipc {
//==============================================================================

class TICSPlug : public tics::Plug
{
    typedef tics::Plug BaseClass;
public:

    typedef void (*OnSessionStartedCallback)( void* lpContext );
    typedef int  (*OnSessionDetachedCallback)( void* lpContext );

#if ENABLE_BENCHMARK_TESTING
    typedef void (*OnPacketReceivedCallback)( const tmipc::Packet& Packet );
    typedef void (*OnPacketSendCallback)( const tmipc::Packet& Packet );
#endif // ENABLE_BENCHMARK_TESTING

private:

    enum
    {
        // The channel/session name length.
        MaximumNameLength = 20,
    };

    char                        m_Name[MaximumNameLength];          // Name of the plug.
    tics::ForeignMemBuffer      m_ReceiveBuffer;                    // Wrapper for receive and send
    tics::ForeignMemBuffer      m_SendBuffer;                       //  buffers.
    tmipc::Event                m_ReceiveEvent;                     // Set when a packet has been
    tmipc::Event                m_SendEvent;                        //  sent or received.
    OnSessionStartedCallback    m_pfOnSessionStartedCallback;       // Callback functions and context
    void*                       m_pOnSessionStartedContext;         //  arguments that are invoked
    OnSessionDetachedCallback   m_pfOnSessionDetachedCallback;      //  when events have occurred.
    void*                       m_pOnSessionDetachedContext;        //
    bool                        m_bReceiveWasCancelled;             // Helps to determine when a
    bool                        m_bSendWasCancelled;                //  Receive and Send were cancelled.
    bool                        m_IsConnected;                      // true: the Plug is "connected".

#if ENABLE_BENCHMARK_TESTING
    static OnPacketReceivedCallback m_pfOnPacketReceivedCallback;   // Called when a packet is received.
    static OnPacketSendCallback     m_pfOnPacketSendCallback;       // Called right before a packet is sent.
#endif // ENABLE_BENCHMARK_TESTING

private:

    // Initializes member data.
    void Initialize();

    // Callback for Receive complete.
    virtual int OnReceiveComplete( tics::Buffer* pBuffer, long Offset, long Length );

    // Callback for send complete.
    virtual int OnSendComplete( tics::Buffer* pBuffer, long Offset, long Length );

    // Received when a session starts.
    virtual void OnSessionStarted( tics::portability::stl::string Type, Endpoint* pConnectedEndPoint );

    // Received when a Plug detaches.
    virtual int OnDetach( void );

    // Received when a remote plug detaches.
    virtual int OnRemoteDetach( tics::DisconnectReason reason );

public:

    // Constructors/destructor.
    TICSPlug();
    explicit TICSPlug( const char* pName );
    virtual ~TICSPlug( void );

    // Allow the user to register a callback when a Session has started.
    void RegisterOnSessionStartedCallback( OnSessionStartedCallback pfOnSessionStartedCallback, void* pContext );

    // Allow the user to register a callback when a Session has detached.
    void RegisterOnSessionDetachedCallback( OnSessionDetachedCallback pfOnSessionDetachedCallback, void* pContext );

    // Receives data.  Blocking call. This function is not thread safe. It is
    // assumed that the caller prevents Threading issues (Mutex, whatever).
    tmipc::Result Receive( tmipc::Packet& Packet );

    // Sends data.  Blocking call. This function is not thread safe. It is
    // assumed that the caller prevents Threading issues (Mutex, whatever).
    tmipc::Result Send( const tmipc::Packet& Packet );

    // Returns: true: a connection exists; false: no connection exists.
    bool IsConnected() const
    {
        return m_IsConnected;
    }

    // Cancels a current pending read and/or write operation.
    void CancelBlockingOperations();

#if ENABLE_BENCHMARK_TESTING
    // Sets/clears the callback invoked when any packet is received.
    static void SetOnPacketReceivedCallback( OnPacketReceivedCallback pfOnPacketReceivedCallback );
    // Sets/clears the callback invoked when right before a packet is sent.
    static void SetOnPacketSendCallback( OnPacketSendCallback pfOnPacketSendCallback );
#endif // ENABLE_BENCHMARK_TESTING
};

//==============================================================================
}
//==============================================================================
