﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include "tmipc_defines.h"
#include "tmipc_node.h"
#include "tmipc_result.h"

#if defined(TMIPC_TARGET_HORIZON)
#include <nn/nn_Common.h>
#include <nn/htclow.h>

#if defined(NN_BUILD_CONFIG_HARDWARE_BDSLIMX6) || defined(NN_BUILD_CONFIG_HARDWARE_JETSONTK1) || defined(NN_BUILD_CONFIG_HARDWARE_NX) || defined(NN_BUILD_CONFIG_HARDWARE_JETSONTK2)
#define NN_DETAIL_TMA_NX_RELATED_HARDWARE
#endif
#endif

namespace tmipc {

#if defined(TMIPC_TARGET_HORIZON) && !defined(NN_DETAIL_TMA_NX_RELATED_HARDWARE)
class NodeHtclow : public Node
{
public:
    static Result Initialize() NN_NOEXCEPT;
    static Result Finalize() NN_NOEXCEPT;

    NodeHtclow() NN_NOEXCEPT;
    virtual ~NodeHtclow() NN_NOEXCEPT;

    virtual Result Listen() NN_NOEXCEPT;
    virtual void StopListening() NN_NOEXCEPT;

    virtual Result Disconnect() NN_NOEXCEPT;
    virtual bool IsConnected() NN_NOEXCEPT;

    virtual Result Send(Packet* pPacket) NN_NOEXCEPT;

private:
    static const nn::htclow::ModuleId HtclowModuleId = nn::htclow::ModuleId::Tm;
    static const nn::htclow::ChannelId HtclowChannelId = 0;

    static const s32 SendQueueSize = 64;

    static s32 ListenThreadEntry(void* pArg) NN_NOEXCEPT;
    static s32 SendThreadEntry(void* pArg) NN_NOEXCEPT;
    static s32 RecvThreadEntry(void* pArg) NN_NOEXCEPT;

    nn::Result ListenThreadBody() NN_NOEXCEPT;

    void ListenThread() NN_NOEXCEPT;
    void SendThread() NN_NOEXCEPT;
    void RecvThread() NN_NOEXCEPT;

    void StartThreads() NN_NOEXCEPT; // Starts the Send/Receive threads.
    void StopThreads() NN_NOEXCEPT; // Stops the Send/Receive threads.

    nn::Result HtclowSend(const void* pBuffer, size_t bufferSize) NN_NOEXCEPT;
    nn::Result HtclowRecv(void* pOutBuffer, size_t bufferSize) NN_NOEXCEPT;

    static Thread m_ListenThread;
    static Thread m_SendThread;
    static Thread m_RecvThread;

    static NN_OS_ALIGNAS_THREAD_STACK u8 m_ListenThreadStack[TMIPC_STACK_SIZE];
    static NN_OS_ALIGNAS_THREAD_STACK u8 m_SendThreadStack[TMIPC_STACK_SIZE];
    static NN_OS_ALIGNAS_THREAD_STACK u8 m_RecvThreadStack[TMIPC_STACK_SIZE];

    Mutex m_DisconnectLock;
    MsgQueue m_SendQueue;
    Event m_ListenEvent; // event for listen thread

    nn::htclow::Channel m_HtclowChannel;
    bool m_IsConnected;
    bool m_IsListening;
    bool m_StopListening;
};

#else // defined(TMIPC_TARGET_HORIZON) && !defined(NN_DETAIL_TMA_NX_RELATED_HARDWARE)

class NodeHtclow : public Node
{
    virtual Result Listen() NN_NOEXCEPT
    {
        return TMIPC_RESULT_FAILED;
    }

    virtual Result Disconnect() NN_NOEXCEPT
    {
        return TMIPC_RESULT_FAILED;
    }

    virtual bool IsConnected() NN_NOEXCEPT
    {
        return false;
    }

    virtual Result Send(Packet* pPacket) NN_NOEXCEPT
    {
        (void)pPacket;
        return TMIPC_RESULT_FAILED;
    }
};

#endif // defined(TMIPC_TARGET_HORIZON) && !defined(NN_DETAIL_TMA_NX_RELATED_HARDWARE)

}
