﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

//==============================================================================
// Define Target and Define common macros
//==============================================================================

#if 0
#define TMA_PRINTF( ... ) NN_SDK_LOG( __VA_ARGS__ )
#define TMA_TRACE( channel, ... ) NN_SDK_LOG( channel " - " ); NN_SDK_LOG( __VA_ARGS__ ); NN_SDK_LOG( "\n" )
#define TMA_RESULT_TRACE( result ) TMA_TRACE( "TMA_RESULT_TRACE", "File %s Line %d Module %d Description %d\n", __FILE__, __LINE__, result.GetModule(), result.GetDescription() );
#else
#define TMA_PRINTF( ... )
#define TMA_TRACE( channel, ... )
#define TMA_RESULT_TRACE( result )
#endif

#define TMA_SPRINTF( Buffer, Format, ... ) sprintf( Buffer, Format, __VA_ARGS__ )
#define TMA_STRNCPY( Dest, Source, Count ) strncpy( Dest, Source, Count )
#define TMA_STRCAT( Dest, DstLen, Source ) strcat( Dest, Source )
#define TMA_STRCPY( Dest, DstLen, Source ) strcpy( Dest, Source )

//==============================================================================
// Platform includes
//==============================================================================

#include <cstring>
#include <cstdlib>
#include <cstdio>
#include <cstdarg>

#include <nn/os.h>
#include <nn/nn_SdkLog.h>
#include <nn/nn_SdkAssert.h>

#define TMA_MACRO_VALUE(a) a

#define TRACK_MEMORY TMA_MACRO_VALUE(0)

#if defined(NN_BUILD_CONFIG_OS_WIN)
#define NN_TMA_SIM_INSTANCE TMA_MACRO_VALUE(1)
#endif

#if !defined(NN_BUILD_CONFIG_OS_HORIZON)
#ifndef NTDDI_WIN7SP1
#define NTDDI_WIN7SP1 TMA_MACRO_VALUE(0x06010100)
#endif
#include <SDKDDKVer.h>
#include <WinSock2.h>
typedef int socklen_t;
#endif

#if defined(NN_BUILD_CONFIG_OS_HORIZON)

#include <nn/dbg/dbg_Api.h>
#include <nn/svc/svc_Base.h>
#include <nn/svc/svc_Dmnt.h>
#include <nn/svc/svc_Server.h>
#include <nn/svc/svc_Thread.h>
#include <nn/svc/svc_Result.h>
#include <nn/svc/svc_MemoryMapSelect.h>
#include <nn/svc/svc_Tcb.h>
#include <nn/sm/sm_UserApi.h>

#include <nn/ldr/ldr_ShellApi.h>
#include <nn/pm/pm_DebugMonitorApi.h>
#include <nn/lr/lr_Service.h>
#include <nn/lr/lr_LocationResolver.h>

extern void* _aligned_malloc( size_t size, size_t alignment );
extern void _aligned_free( void* memblock );

#define ALL_REGISTERS nn::svc::ContextFlag_General | nn::svc::ContextFlag_Control | nn::svc::ContextFlag_Fpu | nn::svc::ContextFlag_FpuControl

#endif
#include <stdint.h>

#if defined(ASSERT)
#undef ASSERT
#endif
#define ASSERT NN_SDK_ASSERT

//==============================================================================
// Deja Insight
//==============================================================================

#ifndef DEJA_DISABLED
#include "DejaInsight.h"
#endif

//==============================================================================
// System wide defines
//==============================================================================

enum
{
    // Socket definitions.
    TMA_BEACON_PORT         = 17101,
    TMA_AGENT_PORT          = 17101,
    TMA_AUTOCONNECT_PORT    = 17111,
    TMA_CHARACTER_LIMIT     = 1024,
    TMA_AUTOCONNECT_VERSION = 1,

    // CPU definitions.
    TMA_NUM_GP_REGISTERS    = 16,
    TMA_NUM_FP_REGISTERS    = 32
};

//==============================================================================
// Types
//==============================================================================

//==============================================================================
namespace tma {
//==============================================================================

typedef int8_t      s8;
typedef uint8_t     u8;
typedef int16_t     s16;
typedef uint16_t    u16;
typedef int32_t     s32;
typedef uint32_t    u32;
typedef int64_t     s64;
typedef uint64_t    u64;

#if TRACK_MEMORY
typedef void* ( *AllocateFunction )( size_t, const char*, int );
extern AllocateFunction s_AllocateCall;
#define s_Allocate(X) s_AllocateCall(X, __FILE__, __LINE__)
#else
typedef void* ( *AllocateFunction )( size_t );
extern AllocateFunction s_Allocate;
#endif

typedef void ( *DeallocateFunction )( void*, size_t );
extern DeallocateFunction s_Deallocate;


//==============================================================================
namespace dbg {
//==============================================================================

enum
{
    ARM64_REGISTER_FP           = 29,
    ARM64_REGISTER_LR           = 30,
    ARM64_REGISTER_SP           = 31,
    ARM64_REGISTER_PC           = 32, // TEMP HACK: Beyond bounds of cpuRegisters[]
    ARM64_REGISTER_CONTROL      = 33, // TEMP HACK: Beyond bounds of cpuRegisters[]
    ARM64_FPU_REGISTER_START    = ((ARM64_REGISTER_CONTROL) + 1),

    ARM32_REGISTER_SP           = 13,
    ARM32_REGISTER_LR           = 14,
    ARM32_REGISTER_PC           = 15,
    ARM32_REGISTER_CONTROL      = 16,// TEMP HACK: Beyond bounds of cpuRegisters[]
    ARM32_FPU_REGISTER_START    = ((ARM32_REGISTER_CONTROL) + 1),

    CF_REG_IP                   = -3, // Call Frame aliases for registers
    CF_REG_FA                   = -2,
    CF_REG_RA                   = -1,
};

//==============================================================================

enum
{
    SDK_BKPT                    = 0xE7FFFFFF,
    SDK_BKPT_MASK               = 0xFFFFFFFF,

    ARM_BKPT                    = 0xE7FFDEFE,   // Invalid instruction in all three modes
    ARM_BKPT_MASK               = 0xFFFFFFFF,

    A64_BRK                     = 0xD4200000,   // 1101 0100 001  Im16    0 0000
    A64_BRK_MASK                = 0xFFE0001F,

    A64_HLT                     = 0xD4400000,   // 1101 0100 010  Im16    0 0000
    A64_HLT_MASK                = 0xFFE0001F,

    A32_BKPT                    = 0xE1200070,   // 1110 0001 0010 Im12 0111  Im4
    A32_BKPT_MASK               = 0xFFF000F0,

    T16_BKPT                    = 0x0000BE00,   // 1011 1110  Im8
    T16_BKPT_MASK               = 0x0000FF00,
};

enum // Replacements for ARM_BKPT when user coded breakpoint encountered
{
    A64_NOP     = 0xB503201F,
    A32_NOP     = 0xE320F000,
    T32_NOP     = 0x1000F3AF,
    T16_NOP     = 0x0000BF00,
};

//==============================================================================

struct thread_context
{
    u64 cpuRegisters[30];
    u64 lr;
    u64 sp;
    u64 pc;
    u64 cpsr;
    u64 fpuRegisters[64];
    u32 fpcr;
    u32 fpsr;
    u64 tpidr;
};

// Writes a formatted memory dump to NN_SDK_LOG
void MemToLog( void* pMemory, int nTotalBytes, const char* pHeader = 0, int nBytesPerWord = 4, int  nWordsPerLine = 4 );

// LEB Encode/Decode routines
void EncodeULEB(  u8*& pWrite, u8* pEnd, u64 Value );
s32  GetSizeULEB( u64  Value );
void EncodeSLEB(  u8*& pWrite, u8* pEnd, s64 Value );
s32  GetSizeSLEB( s64  Value );
void EncodeUTF8(  u8*& pWrite, u8* pEnd, const char* pString );
u64  DecodeULEB(  u8*& pRead );
s64  DecodeSLEB(  u8*& pRead );
void SkipLEB(     u8*& pRead );
s32  DecodeUTF8(  u8*& pRead, u8* pBuffer, s32 BufSize );
u8*  SkipUTF8(    u8*& pRead );

//==============================================================================
} // namespace dbg
//==============================================================================

// Returns the Target's Configuration ID.  This is guaranteed to not be nullptr or "".
const char* GetTargetConfigurationId();

// Returns the Target's Serial Number.  This is guaranteed to not be nullptr, but can be "".
const char* GetTargetSerialNumber();

// Returns if tma should fail to communicate to host or not
const bool GetEnableDisconnectionEmulationFlag();

// Returns the total amount of Heap RAM that was allocated to TMA.
const s32 GetHeapRAMTotal();

// Returns the current amount of RAM that was allocated from TMA's HEAP.
const s32 GetHeapRAMAllocated();

//==============================================================================
} // namespace tma
//==============================================================================
