﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "htcs_task.h"
#include "htcs_service.h"
#include "htcs_opcodes.h"
#include <nn/nn_Abort.h>

//==============================================================================
namespace tma { namespace htcs {
//==============================================================================

//==============================================================================
// The error codes we get back are Winsock error codes.
// These values are copied from winerror.h, and we need to include them
// here for build environments that don't have that header.
#ifndef _WINERROR_
enum
{
    WSABASEERR                      = 10000,
    WSAEINTR                        = 10004L,
    WSAEBADF                        = 10009L,
    WSAEACCES                       = 10013L,
    WSAEFAULT                       = 10014L,
    WSAEINVAL                       = 10022L,
    WSAEMFILE                       = 10024L,
    WSAEWOULDBLOCK                  = 10035L,
    WSAEINPROGRESS                  = 10036L,
    WSAEALREADY                     = 10037L,
    WSAENOTSOCK                     = 10038L,
    WSAEDESTADDRREQ                 = 10039L,
    WSAEMSGSIZE                     = 10040L,
    WSAEPROTOTYPE                   = 10041L,
    WSAENOPROTOOPT                  = 10042L,
    WSAEPROTONOSUPPORT              = 10043L,
    WSAESOCKTNOSUPPORT              = 10044L,
    WSAEOPNOTSUPP                   = 10045L,
    WSAEPFNOSUPPORT                 = 10046L,
    WSAEAFNOSUPPORT                 = 10047L,
    WSAEADDRINUSE                   = 10048L,
    WSAEADDRNOTAVAIL                = 10049L,
    WSAENETDOWN                     = 10050L,
    WSAENETUNREACH                  = 10051L,
    WSAENETRESET                    = 10052L,
    WSAECONNABORTED                 = 10053L,
    WSAECONNRESET                   = 10054L,
    WSAENOBUFS                      = 10055L,
    WSAEISCONN                      = 10056L,
    WSAENOTCONN                     = 10057L,
    WSAESHUTDOWN                    = 10058L,
    WSAETOOMANYREFS                 = 10059L,
    WSAETIMEDOUT                    = 10060L,
    WSAECONNREFUSED                 = 10061L,
    WSAELOOP                        = 10062L,
    WSAENAMETOOLONG                 = 10063L,
    WSAEHOSTDOWN                    = 10064L,
    WSAEHOSTUNREACH                 = 10065L,
    WSAENOTEMPTY                    = 10066L,
    WSAEPROCLIM                     = 10067L,
    WSAEUSERS                       = 10068L,
    WSAEDQUOT                       = 10069L,
    WSAESTALE                       = 10070L,
    WSAEREMOTE                      = 10071L,
    WSASYSNOTREADY                  = 10091L,
    WSAVERNOTSUPPORTED              = 10092L,
    WSANOTINITIALISED               = 10093L,
    WSAEDISCON                      = 10101L,
    WSAENOMORE                      = 10102L,
    WSAECANCELLED                   = 10103L,
    WSAEINVALIDPROCTABLE            = 10104L,
    WSAEINVALIDPROVIDER             = 10105L,
    WSAEPROVIDERFAILEDINIT          = 10106L,
    WSASYSCALLFAILURE               = 10107L,
    WSASERVICE_NOT_FOUND            = 10108L,
    WSATYPE_NOT_FOUND               = 10109L,
    WSA_E_NO_MORE                   = 10110L,
    WSA_E_CANCELLED                 = 10111L,
    WSAEREFUSED                     = 10112L,
    WSAHOST_NOT_FOUND               = 11001L,
    WSATRY_AGAIN                    = 11002L,
    WSANO_RECOVERY                  = 11003L,
    WSANO_DATA                      = 11004L,
    WSA_QOS_RECEIVERS               = 11005L,
    WSA_QOS_SENDERS                 = 11006L,
    WSA_QOS_NO_SENDERS              = 11007L,
    WSA_QOS_NO_RECEIVERS            = 11008L,
    WSA_QOS_REQUEST_CONFIRMED       = 11009L,
    WSA_QOS_ADMISSION_FAILURE       = 11010L,
    WSA_QOS_POLICY_FAILURE          = 11011L,
    WSA_QOS_BAD_STYLE               = 11012L,
    WSA_QOS_BAD_OBJECT              = 11013L,
    WSA_QOS_TRAFFIC_CTRL_ERROR      = 11014L,
    WSA_QOS_GENERIC_ERROR           = 11015L,
    WSA_QOS_ESERVICETYPE            = 11016L,
    WSA_QOS_EFLOWSPEC               = 11017L,
    WSA_QOS_EPROVSPECBUF            = 11018L,
    WSA_QOS_EFILTERSTYLE            = 11019L,
    WSA_QOS_EFILTERTYPE             = 11020L,
    WSA_QOS_EFILTERCOUNT            = 11021L,
    WSA_QOS_EOBJLENGTH              = 11022L,
    WSA_QOS_EFLOWCOUNT              = 11023L,
    WSA_QOS_EUNKOWNPSOBJ            = 11024L,
    WSA_QOS_EPOLICYOBJ              = 11025L,
    WSA_QOS_EFLOWDESC               = 11026L,
    WSA_QOS_EPSFLOWSPEC             = 11027L,
    WSA_QOS_EPSFILTERSPEC           = 11028L,
    WSA_QOS_ESDMODEOBJ              = 11029L,
    WSA_QOS_ESHAPERATEOBJ           = 11030L,
    WSA_QOS_RESERVED_PETYPE         = 11031L,
    WSA_SECURE_HOST_NOT_FOUND       = 11032L,
    WSA_IPSEC_NAME_POLICY_ERROR     = 11033L,
};
#endif//_WINERR OR_

//==============================================================================

nn::htcs::SocketError ConvertToHtcsSocketError( int WinsockErrorCode )
{
    nn::htcs::SocketError HTCSErrorCode = nn::htcs::HTCS_EUNKNOWN;

    // The error codes we get back are Winsock error codes.
    switch ( WinsockErrorCode )
    {
        case 0:                 HTCSErrorCode = nn::htcs::HTCS_ENONE;  break;
        case WSAEACCES:         HTCSErrorCode = nn::htcs::HTCS_EACCES;  break;
        case WSAEADDRINUSE:     HTCSErrorCode = nn::htcs::HTCS_EADDRINUSE;  break;
        case WSAEADDRNOTAVAIL:  HTCSErrorCode = nn::htcs::HTCS_EADDRNOTAVAIL;  break;

        case WSAEWOULDBLOCK:    HTCSErrorCode = nn::htcs::HTCS_EWOULDBLOCK;  break;
        case WSAEALREADY:       HTCSErrorCode = nn::htcs::HTCS_EALREADY;  break;
        case WSAEBADF:          HTCSErrorCode = nn::htcs::HTCS_EBADF;  break;
        case WSAECONNABORTED:   HTCSErrorCode = nn::htcs::HTCS_ECONNABORTED;  break;
        case WSAECONNREFUSED:   HTCSErrorCode = nn::htcs::HTCS_ECONNREFUSED;  break;
        case WSAECONNRESET:     HTCSErrorCode = nn::htcs::HTCS_ECONNRESET;  break;
        case WSAEDESTADDRREQ:   HTCSErrorCode = nn::htcs::HTCS_EDESTADDRREQ;  break;
        case WSAEFAULT:         HTCSErrorCode = nn::htcs::HTCS_EFAULT;  break;
        case WSAEINPROGRESS:    HTCSErrorCode = nn::htcs::HTCS_EINPROGRESS;  break;
        case WSAEINTR:          HTCSErrorCode = nn::htcs::HTCS_EINTR;  break;

        case WSAESHUTDOWN:      HTCSErrorCode = nn::htcs::HTCS_EBUSY;  break;
        case WSAENOTSOCK:       HTCSErrorCode = nn::htcs::HTCS_EINVAL;  break;
        case WSAEINVAL:         HTCSErrorCode = nn::htcs::HTCS_EINVAL;  break;
        case WSAEOPNOTSUPP:     HTCSErrorCode = nn::htcs::HTCS_EINVAL;  break;
        case WSAEISCONN:        HTCSErrorCode = nn::htcs::HTCS_EISCONN;  break;
        case WSAEMFILE:         HTCSErrorCode = nn::htcs::HTCS_EMFILE;  break;
        case WSAEMSGSIZE:       HTCSErrorCode = nn::htcs::HTCS_EMSGSIZE;  break;
        case WSAENETDOWN:       HTCSErrorCode = nn::htcs::HTCS_ENETDOWN;  break;
        case WSAENETRESET:      HTCSErrorCode = nn::htcs::HTCS_ENETRESET;  break;
        case WSAENOBUFS:        HTCSErrorCode = nn::htcs::HTCS_ENOBUFS;  break;
        case WSAENOTCONN:       HTCSErrorCode = nn::htcs::HTCS_ENOTCONN;  break;
        case WSAETIMEDOUT:      HTCSErrorCode = nn::htcs::HTCS_ETIMEDOUT;  break;

        default:                HTCSErrorCode = nn::htcs::HTCS_EUNKNOWN;  break;
    }
    return HTCSErrorCode;
}

//==============================================================================

//==============================================================================
// Base HTCS task class
//==============================================================================

HTCSTask::HTCSTask( int Socket )
{
    m_Socket        = Socket;
    m_Result        = -1;
    m_ErrorCode     = WSAENETDOWN;
}

//==============================================================================

HTCSTask::~HTCSTask()
{
}

//==============================================================================

int HTCSTask::GetSocket()
{
    return m_Socket;
}

//==============================================================================

void HTCSTask::OnInitiate( tmipc::Packet* )
{
}

//==============================================================================

void HTCSTask::OnRecvPacket( tmipc::Packet* )
{
}

//==============================================================================

void HTCSTask::OnSendPacket( tmipc::Packet* )
{
}

//==============================================================================

int32_t HTCSTask::GetResult()
{
    return m_Result;
}

//==============================================================================

int32_t HTCSTask::GetErrorCode()
{
    return (int)ConvertToHtcsSocketError( m_ErrorCode );
}

//==============================================================================

void HTCSTask::OnServiceKilled( )
{
}


//==============================================================================
// Task that signals on completion or delete
HTCSSignalingTask::HTCSSignalingTask( int Socket )
: HTCSTask( Socket )
{
    DEJA_TRACE( "HTCSSignalingTask::HTCSSignalingTask", "HTCSSignalingTask::HTCSSignalingTask(%d)", GetTaskId() );
    nn::Result res = nn::os::CreateSystemEvent( &m_SignalEvent, nn::os::EventClearMode_ManualClear, true );
    if( res.IsSuccess() )
    {
        nn::os::ClearSystemEvent( &m_SignalEvent );
    }
    else
    {
        NN_ABORT("[tma] [HTCSSignalingTask] CreateSystemEvent FAILED:  %x\n", res.GetInnerValueForDebug() );
    }

    //====================================================================================
    // Because this is a longer-running task, we'll mark ourselves as un-owned - this way
    // we will be properly cleaned by the system regardless of when we're finished.
    //====================================================================================
    SetExternallyOwned( false );
}

//==============================================================================

HTCSSignalingTask::~HTCSSignalingTask()
{
    //==========================================================
    // Signal anyone who's listening, pass or fail.
    //==========================================================
//    if( nn::os::TryWaitSystemEvent( &m_SignalEvent ) == false )
    {
       //NN_SDK_LOG("[tma] HTCSSignalingTask %d signalling event on delete\n", GetTaskId() );
        nn::os::SignalSystemEvent( &m_SignalEvent );
    }

    nn::os::DestroySystemEvent( &m_SignalEvent );

    //NN_SDK_LOG("[tma] HTCSSignalingTask COMPLETE %d\n", GetTaskId() );
    AgentHTCSService* pOriginatingService = (AgentHTCSService*)GetService();
    if( pOriginatingService != NULL )
    {
        pOriginatingService->OnSignalEventDestroyed();
    }
}

//==============================================================================

HTCSSocketTask::HTCSSocketTask( int Socket )
    : HTCSTask( Socket )
{
    m_TaskType = tmipc::TaskType_HtcsSocket;
}

//==============================================================================

HTCSSocketTask::~HTCSSocketTask()
{
    if( m_Status == StatusComplete )
    {
        AgentHTCSService* pOriginatingService = (AgentHTCSService*)GetService();
        if( pOriginatingService != NULL )
        {
            pOriginatingService->OnSocketCreated( m_Socket );
        }
    }
}

//==============================================================================

void HTCSSocketTask::OnRecvPacket( tmipc::Packet* pPacket )
{
    DEJA_TRACE( "HTCSSocketTask::OnRecvPacket", "HTCSSocketTask::OnRecvPacket" );

    // Read our socket and result.
    pPacket->ReadS32( m_Socket );
    pPacket->ReadS32( m_Result );
    pPacket->ReadS32( m_ErrorCode );

    // Task complete.
    Complete();
}

//==============================================================================

void HTCSSocketTask::Socket()
{
    DEJA_TRACE( "HTCSTask::Socket", "HTCSTask::Socket" );
    tmipc::Packet* p = AllocSendPacket( true );
    p->WriteS32( m_Socket );
    m_pServicesManager->SubmitTask( this, p );
}

//=============================================================================

HTCSCloseTask::HTCSCloseTask( int Socket )
    : HTCSTask( Socket )
{
    m_TaskType = tmipc::TaskType_HtcsClose;
}

//==============================================================================

HTCSCloseTask::~HTCSCloseTask()
{
    if( m_Status == StatusCanceled )
    {
        AgentHTCSService* pOriginatingService = (AgentHTCSService*)GetService();
        if( pOriginatingService != NULL )
        {
            pOriginatingService->OnSocketCloseFailed( m_Socket );
        }
    }
}

//==============================================================================

void HTCSCloseTask::OnRecvPacket( tmipc::Packet* pPacket )
{
    DEJA_TRACE( "HTCSCloseTask::OnRecvPacket", "HTCSCloseTask::OnRecvPacket" );

    // Read our socket and result.
    pPacket->ReadS32( m_Socket );
    pPacket->ReadS32( m_Result );
    pPacket->ReadS32( m_ErrorCode );

    AgentHTCSService* pOriginatingService = (AgentHTCSService*)GetService();
    if( pOriginatingService != NULL )
    {
        pOriginatingService->OnSocketClosed( m_Socket );
    }

    // Task complete.
    Complete();
}

//==============================================================================

void HTCSCloseTask::Close()
{
    DEJA_TRACE( "HTCSTask::Close", "HTCSTask::Close socket %d", m_Socket );
    tmipc::Packet* p = AllocSendPacket( true );
    p->WriteS32( m_Socket );
    m_pServicesManager->SubmitTask( this, p );
}

//=============================================================================

HTCSBindTask::HTCSBindTask( int Socket )
    : HTCSTask( Socket )
{
    m_TaskType = tmipc::TaskType_HtcsBind;
}

//==============================================================================

HTCSBindTask::~HTCSBindTask()
{
}

//==============================================================================

void HTCSBindTask::OnRecvPacket( tmipc::Packet* pPacket )
{
    DEJA_TRACE( "HTCSBindTask::OnRecvPacket", "HTCSBindTask::OnRecvPacket" );

    // Read our socket and result.
    pPacket->ReadS32( m_Socket );
    pPacket->ReadS32( m_Result );
    pPacket->ReadS32( m_ErrorCode );

    // Task complete.
    Complete();
}

//==============================================================================

void HTCSBindTask::Bind( const nn::htcs::SockAddrHtcs* pAddr )
{
    DEJA_TRACE( "HTCSTask::Bind", "HTCSTask::Bind socket %d to %s:%s", m_Socket, pAddr->peerName.name, pAddr->portName.name );
    tmipc::Packet* p = AllocSendPacket( true );
    p->WriteS32( m_Socket );
    p->WriteData( pAddr, sizeof(nn::htcs::SockAddrHtcs) );
    m_pServicesManager->SubmitTask( this, p );
}

//=============================================================================

HTCSListenTask::HTCSListenTask( int Socket )
    : HTCSTask( Socket )
{
    m_TaskType = tmipc::TaskType_HtcsListen;
}

//==============================================================================

HTCSListenTask::~HTCSListenTask()
{
}

//==============================================================================

void HTCSListenTask::OnRecvPacket( tmipc::Packet* pPacket )
{
    DEJA_TRACE( "HTCSListenTask::OnRecvPacket", "HTCSListenTask::OnRecvPacket" );

    // Read our socket and result.
    pPacket->ReadS32( m_Socket );
    pPacket->ReadS32( m_Result );
    pPacket->ReadS32( m_ErrorCode );

    // Task complete.
    Complete();
}

//==============================================================================

void HTCSListenTask::Listen( int backlogCount )
{
    DEJA_TRACE( "HTCSTask::Listen", "HTCSTask::Listen socket = %d, backlogCount = %d", m_Socket, backlogCount );
    tmipc::Packet* p = AllocSendPacket( true );
    p->WriteS32( m_Socket );
    p->WriteS32( backlogCount );
    m_pServicesManager->SubmitTask( this, p );
}

//=============================================================================

HTCSShutdownTask::HTCSShutdownTask( int Socket )
    : HTCSTask( Socket )
{
    m_TaskType = tmipc::TaskType_HtcsShutdown;
}

//==============================================================================

HTCSShutdownTask::~HTCSShutdownTask()
{
}

//==============================================================================

void HTCSShutdownTask::OnRecvPacket( tmipc::Packet* pPacket )
{
    DEJA_TRACE( "HTCSShutdownTask::OnRecvPacket", "HTCSShutdownTask::OnRecvPacket" );

    // Read our socket and result.
    pPacket->ReadS32( m_Socket );
    pPacket->ReadS32( m_Result );
    pPacket->ReadS32( m_ErrorCode );

    // Task complete.
    Complete();
}

//==============================================================================

void HTCSShutdownTask::Shutdown( int how )
{
    DEJA_TRACE( "HTCSTask::Shutdown", "HTCSTask::Shutdown socket %d, how = %d", m_Socket, how );
    tmipc::Packet* p = AllocSendPacket( true );
    p->WriteS32( m_Socket );
    p->WriteS32( how );
    m_pServicesManager->SubmitTask( this, p );
}

//=============================================================================

HTCSFcntlTask::HTCSFcntlTask( int Socket )
    : HTCSTask( Socket )
{
    m_TaskType = tmipc::TaskType_HtcsFcntl;
}

//==============================================================================

HTCSFcntlTask::~HTCSFcntlTask()
{
}

//==============================================================================

void HTCSFcntlTask::OnRecvPacket( tmipc::Packet* pPacket )
{
    DEJA_TRACE( "HTCSFcntlTask::OnRecvPacket", "HTCSFcntlTask::OnRecvPacket" );

    // Read our socket and result.
    pPacket->ReadS32( m_Socket );
    pPacket->ReadS32( m_Result );
    pPacket->ReadS32( m_ErrorCode );

    // Task complete.
    Complete();
}

//==============================================================================

void HTCSFcntlTask::Fcntl( int command, int value )
{
    DEJA_TRACE( "HTCSTask::Fcntl", "HTCSTask::Fcntl socket %d, command = %d, value = %d", m_Socket, command, value );
    tmipc::Packet* p = AllocSendPacket( true );
    p->WriteS32( m_Socket );
    p->WriteS32( command );
    p->WriteS32( value );
    m_pServicesManager->SubmitTask( this, p );
}

//=============================================================================

HTCSConnectTask::HTCSConnectTask( int Socket )
: HTCSTask( Socket )
{
    DEJA_TRACE( "HTCSConnectTask::HTCSConnectTask", "HTCSConnectTask::HTCSConnectTask, Socket = %d", Socket );
    m_TaskType = tmipc::TaskType_HtcsConnect;
}

//==============================================================================

HTCSConnectTask::~HTCSConnectTask()
{
}

//==============================================================================

void HTCSConnectTask::OnRecvPacket( tmipc::Packet* pPacket )
{
    DEJA_TRACE( "HTCSConnectTask::OnRecvPacket", "HTCSConnectTask::OnRecvPacket" );

    // Read our socket and result.
    pPacket->ReadS32( m_Socket );
    pPacket->ReadS32( m_Result );
    pPacket->ReadS32( m_ErrorCode );

    // Task complete.
    Complete();
}

//==============================================================================

void HTCSConnectTask::Connect( const nn::htcs::SockAddrHtcs* pAddr )
{
    DEJA_TRACE( "HTCSConnectTask::Connect", "HTCSConnectTask::Connect socket %d to %s:%s", m_Socket, pAddr->peerName.name, pAddr->portName.name );
    tmipc::Packet* p = AllocSendPacket( true );
    p->WriteS32( m_Socket );
    p->WriteData( pAddr, sizeof(nn::htcs::SockAddrHtcs) );
    m_pServicesManager->SubmitTask( this, p );
}

//==============================================================================

HTCSAcceptTask::HTCSAcceptTask( int Socket )
: HTCSSignalingTask( Socket )
{
    DEJA_TRACE( "HTCSAcceptTask::HTCSAcceptTask", "HTCSAcceptTask::HTCSAcceptTask (%d)", GetTaskId() );
    memset( &m_AcceptedAddr, 0, sizeof(m_AcceptedAddr) );

    m_TaskType = tmipc::TaskType_HtcsAccept;
}

//==============================================================================

HTCSAcceptTask::~HTCSAcceptTask()
{
    //NN_SDK_LOG("[tma] HTCSAcceptTask (results = %d) COMPLETE %d \n", m_Result, GetTaskId() );
}

//==============================================================================

void HTCSAcceptTask::Start( nn::os::NativeHandle* pWaitHandle )
{
    *pWaitHandle = nn::os::DetachReadableHandleOfSystemEvent( &m_SignalEvent );

    tmipc::Packet* p = AllocSendPacket( true );
    p->WriteS32( m_Socket );

    m_pServicesManager->SubmitTask( this, p );
}

//==============================================================================

void HTCSAcceptTask::OnRecvPacket( tmipc::Packet* pPacket )
{
    DEJA_TRACE( "HTCSAcceptTask::OnRecvPacket", "HTCSAcceptTask::OnRecvPacket" );
    // Read our socket and result.
    pPacket->ReadS32( m_Socket );
    pPacket->ReadS32( m_Result );
    pPacket->ReadS32( m_ErrorCode );

    pPacket->ReadData( &m_AcceptedAddr.family, sizeof(m_AcceptedAddr.family) );
    pPacket->ReadData( &m_AcceptedAddr.peerName, sizeof(m_AcceptedAddr.peerName) );
    pPacket->ReadData( &m_AcceptedAddr.portName, sizeof(m_AcceptedAddr.portName) );

    // Tell our originating service that we're complete.
    AgentHTCSService* pOriginatingService = (AgentHTCSService*)GetService();
    pOriginatingService->OnAcceptComplete( this );

    //========================================================
    // Task is complete.
    //========================================================
    Complete();
}

//==============================================================================

int HTCSAcceptTask::GetResults( nn::htcs::SockAddrHtcs* pAddr, int* pError )
{
    memcpy( pAddr, &m_AcceptedAddr, sizeof(m_AcceptedAddr) );
    *pError = GetErrorCode();
    return m_Result;
}

//==============================================================================
// HTCSRecvTask

HTCSRecvTask::HTCSRecvTask( int Socket )
: HTCSSignalingTask( Socket )
{
    m_Buffer = NULL;
    m_AmountRead = 0;
    m_TaskType = tmipc::TaskType_HtcsRecv;
}

//==============================================================================

HTCSRecvTask::~HTCSRecvTask()
{
    if( m_Buffer != NULL )
    {
        s_Deallocate( m_Buffer, (size_t)m_AmountRead );
    }

    //NN_SDK_LOG("[tma] HTCSRecvBufferedTask %d COMPLETE\n", GetTaskId() );
}

//==============================================================================

void HTCSRecvTask::Start( int64_t bufferByteSize, int flags, nn::os::NativeHandle* pWaitHandle )
{
    ASSERT( bufferByteSize < SIZE_OF_RECV_DATA_BUFFER );

    //NN_SDK_LOG("[tma] HTCSRecvBufferedTask %d START\n", GetTaskId() );
    *pWaitHandle = nn::os::DetachReadableHandleOfSystemEvent( &m_SignalEvent );
    m_Result                = 0;
    m_ErrorCode             = 0;
    tmipc::Packet* p = AllocSendPacket( true );
    p->WriteS32( m_Socket );
    p->WriteS64( bufferByteSize );

    //======================================================================
    // Translate our native flags to WinSock.
    //======================================================================
    enum
    {
        WINSOCK_PEEK    = 0x2,
        WINSOCK_WAITALL = 0x8
    };

    int TranslatedFlags = 0;
    if( (flags & nn::htcs::HTCS_MSG_PEEK) )
    {
        TranslatedFlags |= WINSOCK_PEEK;
    }
    if( (flags & nn::htcs::HTCS_MSG_WAITALL) )
    {
        TranslatedFlags |= WINSOCK_WAITALL;
    }

    //Did we get ANY flags we know?
    if ( TranslatedFlags == 0 )
    {
        // Don't recognize these flags, so just pass on what we received.
        TranslatedFlags = flags;
    }

    p->WriteS32( TranslatedFlags );

    m_pServicesManager->SubmitTask( this, p );
}

//==============================================================================

void HTCSRecvTask::OnRecvPacket( tmipc::Packet* pPacket )
{
    DEJA_TRACE( "HTCSRecvBufferedTask::HTCSRecvBufferedTask", "HTCSRecvBufferedTask::OnRecvPacket" );

    // Get our data
    s32 AmountRead = 0;
    s32 RecvResult = 0;
    u8  IsLastPacket = 0;
    pPacket->ReadS32( AmountRead );
    pPacket->ReadS32( RecvResult );
    pPacket->ReadS32( m_ErrorCode );
    pPacket->ReadU8 ( IsLastPacket );

    DEJA_TRACE( "HTCSRecvBufferedTask::OnRecvPacket", "HTCSRecvBufferedTask %d received %d", GetTaskId(), AmountRead );
    //TMA_PRINTF( "HTCSRecvTask %d received %d\n", GetTaskId(), AmountRead );

    if( RecvResult >= 0)
    {
        ASSERT( (pPacket->GetDataLen() - SIZE_OF_RECV_BUFFER_HEADER) == AmountRead); //( sizeof( AmountRead ) + sizeof( RecvResult ) + sizeof( m_ErrorCode ) + sizeof( IsLastPacket ));
        m_Buffer = s_Allocate( AmountRead );

        //Read our data
        pPacket->ReadData( m_Buffer, AmountRead );
        //Keep track of how much we've read
        m_Result += AmountRead;
        m_AmountRead += AmountRead;
    }
    else
    {
        ASSERT( (SIZE_OF_RECV_BUFFER_HEADER) == ( sizeof( AmountRead ) + sizeof( RecvResult ) + sizeof( m_ErrorCode ) + sizeof( IsLastPacket ) ) );

        //If there was some problem, remember the error number
        m_Result = RecvResult;
    }

    // If we're through with our packets, or there was a problem...
    if( RecvResult < 0 || IsLastPacket != 0 )
    {
        // Tell our originating service that we're complete.
        AgentHTCSService* pOriginatingService = (AgentHTCSService*)GetService();
        pOriginatingService->OnRecvComplete( this );
        nn::os::SignalSystemEvent( &m_SignalEvent );
        Complete();
    }
}

//==============================================================================

int HTCSRecvTask::GetResults( void** pBuffer, int64_t* pAmountRead, int* pError )
{
    *pBuffer = m_Buffer;
    *pAmountRead = m_AmountRead;
    *pError = GetErrorCode();

    // We've handed off our buffer, so mark these NULL so we won't delete them.
    m_Buffer = NULL;
    m_AmountRead = 0;

    return m_Result;
}

//==============================================================================
// HTCSRecvLargeTask

HTCSRecvLargeTask::HTCSRecvLargeTask( int Socket )
: HTCSSignalingTask( Socket )
{
    m_TaskType              = tmipc::TaskType_HtcsRecv;
    m_Flags                 = 0;
    m_AlignedLeftToRecv     = 0;
    m_pAlignedBuffer        = NULL;
    m_PreLeftToRead         = 0;
    m_PostLeftToRead        = 0;
    m_UnalignedStartSize    = 0;
    m_UnalignedEndSize      = 0;
    m_UnalignedRead         = 0;
    m_UnalignedDataSize     = 0;
    m_pUnalignedData        = NULL;
    m_MemoryMapped          = false;
}

//==============================================================================

HTCSRecvLargeTask::~HTCSRecvLargeTask()
{
    if( m_MemoryMapped )
    {
        nn::os::UnmapTransferMemory( &m_Dest );
        nn::os::DestroyTransferMemory( &m_Dest );
    }

    if( m_pUnalignedData != NULL )
    {
        s_Deallocate(m_pUnalignedData, m_UnalignedDataSize );
    }
    //NN_SDK_LOG("[tma] HTCSRecvLargeTask COMPLETE task %d\n", GetTaskId() );
}

//==============================================================================

void HTCSRecvLargeTask::OnRecvPacket( tmipc::Packet* pPacket )
{
    // Get our data
    s32 AmountRead = 0;
    s32 RecvResult = 0;
    u8  IsLastPacket = 0;
    pPacket->ReadS32( AmountRead );
    pPacket->ReadS32( RecvResult );
    pPacket->ReadS32( m_ErrorCode );
    pPacket->ReadU8 ( IsLastPacket );

    DEJA_TRACE( "HTCSRecvLargeTask::OnRecv", "HTCSRecvLargeTask %d received %d", GetTaskId(), AmountRead );
    //NN_SDK_LOG("[tma] HTCSRecvLargeTask::OnRecvPacket received %d bytes task %d\n", AmountRead, GetTaskId() );
    if( RecvResult >= 0)
    {
        ASSERT( (pPacket->GetDataLen() - SIZE_OF_RECV_BUFFER_HEADER) == AmountRead); //( sizeof( AmountRead ) + sizeof( RecvResult ) + sizeof( m_ErrorCode ) + sizeof( IsLastPacket ));

        //===============================================================================
        // We have three different addresses to write to, so
        // figure out where we're writing, and how much.
        //===============================================================================
        s32 AmountToConsume = AmountRead;

        if( ( AmountToConsume > 0 ) && ( m_PreLeftToRead > 0 ) )
        {
            s32 Consume = (s32)(m_PreLeftToRead > AmountToConsume ? AmountToConsume : m_PreLeftToRead );
            void* pWriteSource = (void*)((int64_t)m_pUnalignedData + m_UnalignedRead );
            pPacket->ReadData( pWriteSource, Consume );
            m_UnalignedRead += Consume;
            AmountToConsume -= Consume;
            m_PreLeftToRead -= Consume;
        }

        if( ( AmountToConsume > 0 ) && ( m_AlignedLeftToRecv > 0 ) )
        {
            s32 Consume = (s32)( m_AlignedLeftToRecv > (int64_t)AmountToConsume ? AmountToConsume : m_AlignedLeftToRecv );
            pPacket->ReadData( m_pAlignedBuffer, Consume );
            m_pAlignedBuffer += Consume;
            AmountToConsume -= Consume;
            m_AlignedLeftToRecv -= Consume;
        }

        if( ( AmountToConsume > 0 ) && ( m_PostLeftToRead > 0 ) )
        {
            s32 Consume = (s32)( m_PostLeftToRead > AmountToConsume ? AmountToConsume : m_PostLeftToRead );
            void* pWriteSource = (void*)((int64_t)m_pUnalignedData + m_UnalignedRead );
            pPacket->ReadData( pWriteSource, Consume );
            m_UnalignedRead += Consume;
            AmountToConsume -= Consume;
            m_PostLeftToRead -= Consume;
        }

        //Keep track of how much we've read
        m_Result += AmountRead;
        m_AmountRead += AmountRead;
    }
    else
    {
        ASSERT( (SIZE_OF_RECV_BUFFER_HEADER) == ( sizeof( AmountRead ) + sizeof( RecvResult ) + sizeof( m_ErrorCode ) + sizeof( IsLastPacket ) ) );

        //If there was some problem, remember the error number
        m_Result = RecvResult;
    }

    // If we're through with our packets, or there was a problem...
    if( RecvResult < 0 || IsLastPacket != 0 )
    {
        // Done.  Unmap our memory so the recieving app can use it
        nn::os::UnmapTransferMemory( &m_Dest );
        nn::os::DestroyTransferMemory( &m_Dest );
        m_MemoryMapped = false;

        // Share the good news.
        AgentHTCSService* pOriginatingService = (AgentHTCSService*)GetService();
        pOriginatingService->OnRecvComplete( this );

        nn::os::SignalSystemEvent( &m_SignalEvent );

        Complete();
    }
}

//==============================================================================

void HTCSRecvLargeTask::Start( int32_t unalignedStartSize,  int32_t unalignedEndSize, int64_t alignedSize, nn::os::NativeHandle alignedMemoryHandle,
    int flags, nn::os::NativeHandle* pWaitHandle )
{
    //NN_SDK_LOG("[tma] HTCSRecvLargeTask read %lld bytes START task %d\n", unalignedStartSize + unalignedEndSize + alignedSize, GetTaskId() );

    //================================================================
    // Make sure we can allocate this buffer.
    //================================================================
    m_UnalignedDataSize     = unalignedStartSize + unalignedEndSize;
    if( m_UnalignedDataSize > 0 )
    {
        m_pUnalignedData        = (char*)s_Allocate( m_UnalignedDataSize );
        if( m_pUnalignedData == NULL )
        {
            NN_SDK_LOG("[tma] HTCSRecvLargeTask %d FAILED allocating %d bytes!\n", GetTaskId(), m_UnalignedDataSize );
            Complete();
            return;
        }
    }

    //================================================================
    // Init these to known values.
    //================================================================
    *pWaitHandle = nn::os::DetachReadableHandleOfSystemEvent( &m_SignalEvent );
    m_Flags                 = flags;
    m_AlignedLeftToRecv     = alignedSize;
    m_PreLeftToRead         = unalignedStartSize;
    m_PostLeftToRead        = unalignedEndSize;
    m_UnalignedStartSize    = unalignedStartSize;
    m_UnalignedEndSize      = unalignedEndSize;
    m_Result                = 0;
    m_ErrorCode             = 0;
    m_AmountRead            = 0;
    m_UnalignedRead         = 0;

    nn::os::AttachTransferMemory( &m_Dest, (size_t)alignedSize, alignedMemoryHandle, true );
    nn::Result result = nn::os::MapTransferMemory( (void**)(&m_pAlignedBuffer), &m_Dest, nn::os::MemoryPermission_None );
    if (!result.IsSuccess())
    {
        NN_SDK_LOG("[tma] HTCSRecvLargeTask FAILED task %d calling MapTransferMemory - 0x%x\n", GetTaskId(), result.GetInnerValueForDebug() );
    }
    else
    {
        m_MemoryMapped = true;
    }

    //======================================================================
    // Now we're ready.  Send the packet to start the process.
    //======================================================================
    tmipc::Packet* p = AllocSendPacket( true );
    p->WriteS32( m_Socket );

    int64_t totalRecvSize = unalignedStartSize + unalignedEndSize + alignedSize;
    p->WriteS64( totalRecvSize );

    //======================================================================
    // Translate our native flags to WinSock.
    //======================================================================
    enum
    {
        WINSOCK_PEEK    = 0x2,
        WINSOCK_WAITALL = 0x8
    };

    int TranslatedFlags = 0;
    if( (flags & nn::htcs::HTCS_MSG_PEEK) )
    {
        TranslatedFlags |= WINSOCK_PEEK;
    }
    if( (flags & nn::htcs::HTCS_MSG_WAITALL) )
    {
        TranslatedFlags |= WINSOCK_WAITALL;
    }

    //Did we get ANY flags we know?
    if ( TranslatedFlags == 0 )
    {
        // Don't recognize these flags, so just pass on what we received.
        TranslatedFlags = flags;
    }

    p->WriteS32( TranslatedFlags );

    m_pServicesManager->SubmitTask( this, p );
}

//==============================================================================

int HTCSRecvLargeTask::GetResults( void**pBuffer, int32_t* pUnalignedStartSize, int32_t* pUnalignedEndSize, int64_t* pAmountRead,int* pError )
{
    *pBuffer                = m_pUnalignedData;
    //=====================================================================
    // If this is not a WAIT_ALL recv call, we might not have received all
    // the data we expected.  Trim our received data sizes accordingly.
    //=====================================================================
    *pUnalignedStartSize    = m_UnalignedStartSize - m_PreLeftToRead;
    *pUnalignedEndSize      = m_UnalignedEndSize - m_PostLeftToRead;
    *pAmountRead            = m_AmountRead;
    *pError                 = GetErrorCode();

    // We've handed off our buffer, so mark these NULL so we won't delete them.
    m_pUnalignedData        = NULL;
    m_UnalignedDataSize     = 0;

    return m_Result;
}

//==============================================================================
// HTCSSendTask

HTCSSendTask::HTCSSendTask( int Socket )
: HTCSSignalingTask( Socket )
{
    DEJA_TRACE( "HTCSSendTask::HTCSSendTask", "HTCSSendTask::HTCSSendTask(%d)", GetTaskId() );
    m_TaskType = tmipc::TaskType_HtcsSend;
}

//==============================================================================

HTCSSendTask::~HTCSSendTask()
{
    ///NN_SDK_LOG("[tma] HTCSSendTask COMPLETE %d\n", GetTaskId() );
}

//==============================================================================

void HTCSSendTask::Start( const void* pBuffer, int32_t bufferByteSize, int flags, nn::os::NativeHandle* pWaitHandle )
{
    ASSERT( bufferByteSize < SIZE_OF_SEND_DATA_BUFFER );

    //NN_SDK_LOG("[tma] HTCSSendTask sending %d, START %d\n", bufferByteSize, GetTaskId() );
    *pWaitHandle = nn::os::DetachReadableHandleOfSystemEvent( &m_SignalEvent );

    tmipc::Packet* pPacket = AllocSendPacket( true );
    pPacket->WriteS32( m_Socket );

    pPacket->WriteS32( flags );
    pPacket->WriteS32( bufferByteSize );

    //Last bite?
    u8 LastPacket = true;

    pPacket->WriteU8( LastPacket );

    //Pack out the rest of the buffer.
    pPacket->WriteData( pBuffer, bufferByteSize );

    m_pServicesManager->SubmitTask( this, pPacket );
}

//==============================================================================

void HTCSSendTask::OnRecvPacket( tmipc::Packet* pPacket )
{
    DEJA_TRACE( "HTCSSendTask::HTCSSendTask", "HTCSSendTask::OnRecvPacket" );
    s32 SendResult = 0;
    pPacket->ReadS32( m_Result );   //Here's how many bytes we've sent.
    pPacket->ReadS32( m_ErrorCode );

    pPacket->ReadS32( SendResult ); //This is the result of the send operation

    if( SendResult < 0 )
    {
        m_Result = SendResult;
    }

    DEJA_TRACE( "HTCSSendTask::OnRecvPacket", "Send task %d complete, received result: %d", GetTaskId(), m_Result );
    //NN_SDK_LOG("[tma] HTCSSendTask OnRecvPacket Result %d, Error %d, task %d\n", m_Result, m_ErrorCode, GetTaskId() );

    AgentHTCSService* pOriginatingService = (AgentHTCSService*)GetService();
    pOriginatingService->OnSendComplete( this );
    nn::os::SignalSystemEvent( &m_SignalEvent );

    // Done.
    Complete();
}

//==============================================================================

int HTCSSendTask::GetResults( int* pError )
{
    *pError = GetErrorCode();
    return m_Result;
}

//==============================================================================
// HTCSSendLargeTask

HTCSSendLargeTask::HTCSSendLargeTask( int Socket )
: HTCSSendTask( Socket )
{
    DEJA_TRACE( "HTCSSendLargeTask::HTCSSendLargeTask", "HTCSSendLargeTask::HTCSSendLargeTask(%d)", GetTaskId() );
    m_Flags                 = 0;
    m_pAlignedBuffer        = NULL;
    m_pUnalignedData        = NULL;
    m_AlignedLeftToWrite    = 0;
    m_UnalignedDataSize     = 0;
    m_PostWritten           = 0;
    m_PostLeftToWrite       = 0;
    m_NumberOfPacketsSent   = 0;
    m_MemoryMapped          = false;
}

//==============================================================================

HTCSSendLargeTask::~HTCSSendLargeTask()
{
    if( m_MemoryMapped )
    {
        nn::os::UnmapTransferMemory( &m_Src );
        nn::os::DestroyTransferMemory( &m_Src );
    }

    if( m_pUnalignedData != NULL )
    {
        s_Deallocate( m_pUnalignedData, m_UnalignedDataSize );
    }

    //NN_SDK_LOG("[tma] HTCSSendLargeTask COMPLETE %d\n", GetTaskId() );
}

//==============================================================================

void HTCSSendLargeTask::OnSendPacket( tmipc::Packet* pPacket )
{
    pPacket->WriteS32( m_Socket );
    if( PopulatePacket( pPacket, 0, NULL ) == false )
    {
        // Done, so no more packets needed.
        SetNeedPackets( false );
    }
}

//==============================================================================

void HTCSSendLargeTask::Start( const void* pUnalignedStart, int32_t unalignedStartSize, const void* pUnalignedEnd, int32_t unalignedEndSize, nn::os::NativeHandle alignedDataHandle,
        int64_t alignedSendSize, int flags, nn::os::NativeHandle* pWaitHandle )
{
    //Init these.
    m_Flags                 = flags;
    m_AlignedLeftToWrite    = alignedSendSize;
    m_PostLeftToWrite       = unalignedEndSize;
    m_UnalignedDataSize     = unalignedEndSize;
    m_PostWritten           = 0;
    m_NumberOfPacketsSent   = 0;
    m_Result                = 0;
    m_ErrorCode             = 0;

    //NN_SDK_LOG("[tma] HTCSSendLargeTask sending %d, START %d\n", unalignedStartSize + unalignedEndSize + alignedSendSize, GetTaskId() );
    *pWaitHandle = nn::os::DetachReadableHandleOfSystemEvent( &m_SignalEvent );

    //?? m_AlignedDataHandle = alignedDataHandle;
    nn::os::AttachTransferMemory( &m_Src, (size_t)alignedSendSize, alignedDataHandle, true );//alignedDataHandle.IsManaged() );
    nn::Result result = nn::os::MapTransferMemory( (void**)(&m_pAlignedBuffer), &m_Src, nn::os::MemoryPermission_None );
    if (!result.IsSuccess())
    {
        NN_SDK_LOG("[tma] HTCSSendLargeTask FAILED task %d calling MapTransferMemory - 0x%x\n", GetTaskId(), result.GetInnerValueForDebug() );
    }
    else
    {
        m_MemoryMapped = true;
    }

    // Save the stuff at the end that's not aligned - we have to write it last.
    if( m_PostLeftToWrite > 0 )
    {
        m_pUnalignedData        = (char*)s_Allocate( m_PostLeftToWrite );
        if( m_pUnalignedData == NULL )
        {
            NN_SDK_LOG("[tma] HTCSSendTask %d FAILED allocating %d bytes!\n", GetTaskId(), m_PostLeftToWrite );
            Complete();
            return;
        }
        memcpy( m_pUnalignedData, pUnalignedEnd, (size_t)unalignedEndSize );
    }

    //Submit the packet
    tmipc::Packet* p = AllocSendPacket( true );
    p->WriteS32( m_Socket );

    if( PopulatePacket( p, unalignedStartSize, pUnalignedStart ) )
    {
        SetNeedPackets();
    }

    m_pServicesManager->SubmitTask( this, p );
}

//==============================================================================

bool HTCSSendLargeTask::PopulatePacket( tmipc::Packet* pPacket, int32_t PreWriteAmount, const void* pSrc )
{
    //===============================================================================
    // We have three different addresses to write from, so
    // figure out from whence we're writing, and how much.
    //===============================================================================
    s32 WriteThisPass = PreWriteAmount;
    s32 AlignedWriteAmount = 0;
    s32 PostWriteAmount = 0;

    s32 SpaceLeft = SIZE_OF_SEND_DATA_BUFFER - WriteThisPass;

    // If we have space left, write out all the aligned data we can
    if( SpaceLeft > 0 )
    {
        AlignedWriteAmount = (s32)(m_AlignedLeftToWrite > SpaceLeft ? SpaceLeft : m_AlignedLeftToWrite );
        WriteThisPass += AlignedWriteAmount;
        m_AlignedLeftToWrite -= AlignedWriteAmount;
        SpaceLeft = SIZE_OF_SEND_DATA_BUFFER - WriteThisPass;
    }

    // If we have space left after the aligned data, write out all the un-aligned data we can
    if( SpaceLeft > 0 )
    {
        PostWriteAmount = (s32)(m_PostLeftToWrite > SpaceLeft ? SpaceLeft : m_PostLeftToWrite );
        WriteThisPass += PostWriteAmount;
        m_PostLeftToWrite -= PostWriteAmount;
        SpaceLeft = SIZE_OF_SEND_DATA_BUFFER - WriteThisPass;
    }

    pPacket->WriteS32( m_Flags );
    pPacket->WriteS32( WriteThisPass );

    //===============================================================================
    // Last bite?
    int64_t AmountLeft = m_PostLeftToWrite + m_AlignedLeftToWrite;
    u8 LastPacket = ( AmountLeft <= 0 );
    pPacket->WriteU8( LastPacket );

    //===============================================================================
    // Now write the data itself
    if( PreWriteAmount > 0  )
    {
        //NN_SDK_LOG("[tma] HTCSSendLargeTask::PopulatePacket writing pre unalign buffer (%d bytes), task %d\n", PreWriteAmount, GetTaskId() );
        pPacket->WriteData( pSrc, PreWriteAmount );
    }

    if( AlignedWriteAmount > 0 )
    {
        //NN_SDK_LOG("[tma] HTCSSendLargeTask::PopulatePacket writing align buffer (%d bytes), task %d\n", AlignedWriteAmount, GetTaskId() );
        pPacket->WriteData( m_pAlignedBuffer, AlignedWriteAmount );
        m_pAlignedBuffer += AlignedWriteAmount;
    }

    if( PostWriteAmount > 0 )
    {
        //NN_SDK_LOG("[tma] HTCSSendLargeTask::PopulatePacket writing post buffer (%d bytes, %d left), task %d\n", PostWriteAmount, m_PostLeftToWrite, GetTaskId() );
        void* pWriteSource = (void*)((int64_t)m_pUnalignedData + m_PostWritten );
        pPacket->WriteData( pWriteSource, PostWriteAmount );
        m_PostWritten += PostWriteAmount;
    }

    m_NumberOfPacketsSent += 1;
    DEJA_TRACE( "HTCSSendLargeTask::PopulatePacket", "HTCSSendLargeTask %d:  LastPacket = %d, Write this pass = %d, Amount Left = %d, sent %d packets", GetTaskId(), LastPacket, WriteThisPass, m_PostLeftToWrite, m_NumberOfPacketsSent );

    return LastPacket == false;
}

//==============================================================================

int HTCSSendLargeTask::GetResults( int* pError )
{
    // Done.  Unmap our memory so the recieving app can use it
    if( m_MemoryMapped )
    {
        nn::os::UnmapTransferMemory( &m_Src );
        nn::os::DestroyTransferMemory( &m_Src );
        m_MemoryMapped = false;
    }
    return HTCSSendTask::GetResults( pError );
}

//==============================================================================
// Deprecated tasks
//==============================================================================


//==============================================================================
DeprecatedAcceptTask::DeprecatedAcceptTask( int Socket )
: HTCSTask( Socket )
{
    m_WriteAddr = NULL;
    m_TaskType = tmipc::TaskType_HtcsAccept;
}

//==============================================================================

DeprecatedAcceptTask::~DeprecatedAcceptTask()
{
}

//==============================================================================

void DeprecatedAcceptTask::Accept( nn::htcs::SockAddrHtcs* pAddr )
{
    tmipc::Packet* p = AllocSendPacket( true );
    p->WriteS32( m_Socket );
    m_WriteAddr = pAddr;
    m_pServicesManager->SubmitTask( this, p );
}

//==============================================================================

void DeprecatedAcceptTask::OnRecvPacket( tmipc::Packet* pPacket )
{
    // Read our socket and result.
    pPacket->ReadS32( m_Socket );
    pPacket->ReadS32( m_Result );
    pPacket->ReadS32( m_ErrorCode );

    if( m_WriteAddr != NULL )
    {
        pPacket->ReadData( &m_WriteAddr->family, sizeof(m_WriteAddr->family) );
        pPacket->ReadData( &m_WriteAddr->peerName, sizeof(m_WriteAddr->peerName) );
        pPacket->ReadData( &m_WriteAddr->portName, sizeof(m_WriteAddr->portName) );
    }
//    TMA_PRINTF( "HTCSAcceptTask::Accept socket %d complete, %s.%s\n", m_Socket, m_WriteAddr->peerName.name, m_WriteAddr->portName.name );

    // Task complete.
    Complete();
}

//==============================================================================

DeprecatedRecvTask::DeprecatedRecvTask( int Socket )
: HTCSTask( Socket )
{
    m_WriteAddr = NULL;
    m_TaskType = tmipc::TaskType_HtcsRecv;
}

//==============================================================================

DeprecatedRecvTask::~DeprecatedRecvTask()
{
}

//==============================================================================

void DeprecatedRecvTask::OnRecvPacket( tmipc::Packet* pPacket )
{
    // Get our data
    s32 AmountRead = 0;
    s32 RecvResult = 0;
    u8  IsLastPacket = 0;
    pPacket->ReadS32( AmountRead );
    pPacket->ReadS32( RecvResult );
    pPacket->ReadS32( m_ErrorCode );
    pPacket->ReadU8 ( IsLastPacket );

    DEJA_TRACE( "HTCSRecvTask::OnRecv", "HTCSRecvTask %d received %d", GetTaskId(), AmountRead );
    //TMA_PRINTF( "HTCSRecvTask %d received %d\n", GetTaskId(), AmountRead );

    if( RecvResult >= 0)
    {
        ASSERT( (pPacket->GetDataLen() - SIZE_OF_RECV_BUFFER_HEADER) == AmountRead); //( sizeof( AmountRead ) + sizeof( RecvResult ) + sizeof( m_ErrorCode ) + sizeof( IsLastPacket ));

        //Read our data
        pPacket->ReadData( m_WriteAddr, AmountRead );

        //Update where we write to.
        m_WriteAddr = (void*)( ((u64)m_WriteAddr) + AmountRead );

        //Keep track of how much we've read
        m_Result += AmountRead;
    }
    else
    {
        ASSERT( (SIZE_OF_RECV_BUFFER_HEADER) == ( sizeof( AmountRead ) + sizeof( RecvResult ) + sizeof( m_ErrorCode ) + sizeof( IsLastPacket ) ) );

        //If there was some problem, remember the error number
        m_Result = RecvResult;
    }

    // If we're through with our packets, or there was a problem...
    if( RecvResult < 0 || IsLastPacket != 0 )
    {
        // Done.
        Complete();
    }
}

//==============================================================================

void DeprecatedRecvTask::Recv( void* pBuffer, size_t bufferByteSize, int flags )
{
    DEJA_TRACE( "HTCSRecvTask::Recv", "HTCSRecvTask::Recv %d bytes on socket %d, flags = %d", bufferByteSize, m_Socket, flags );

    //Init these to known values.
    m_WriteAddr                 = pBuffer;
    m_Result                    = 0;
    m_ErrorCode                 = 0;
    tmipc::Packet* p = AllocSendPacket( true );
    p->WriteS32( m_Socket );
    p->WriteS64( bufferByteSize );

    //======================================================================
    // Translate our native flags to WinSock.
    //======================================================================
    enum
    {
        WINSOCK_PEEK    = 0x2,
        WINSOCK_WAITALL = 0x8
    };

    int TranslatedFlags = 0;
    if( (flags & nn::htcs::HTCS_MSG_PEEK) )
    {
        TranslatedFlags |= WINSOCK_PEEK;
    }
    if( (flags & nn::htcs::HTCS_MSG_WAITALL) )
    {
        TranslatedFlags |= WINSOCK_WAITALL;
    }

    //Did we get ANY flags we know?
    if ( TranslatedFlags == 0 )
    {
        // Don't recognize these flags, so just pass on what we received.
        TranslatedFlags = flags;
    }

    p->WriteS32( TranslatedFlags );

    m_pServicesManager->SubmitTask( this, p );
}

//==============================================================================

DeprecatedSendTask::DeprecatedSendTask(  int Socket )
: HTCSTask( Socket )
{
    m_Flags                 = 0;
    m_pReadFromBuffer       = NULL;
    m_AmountLeftToWrite     = 0;
    m_NumberOfPacketsSent   = 0;
    m_TaskType              = tmipc::TaskType_HtcsSend;
}

//==============================================================================

DeprecatedSendTask::~DeprecatedSendTask()
{
}

//==============================================================================

void DeprecatedSendTask::OnRecvPacket( tmipc::Packet* pPacket )
{
    //Read the result
    s32 SendResult = 0;
    pPacket->ReadS32( m_Result );   //Here's how many bytes we've sent.
    pPacket->ReadS32( m_ErrorCode );

    pPacket->ReadS32( SendResult ); //This is the result of the send operation

    if( SendResult < 0 )
    {
        m_Result = SendResult;
    }

    DEJA_TRACE( "HTCSSendTask::OnRecvPacket", "Send task %d complete, received result: %d", GetTaskId(), m_Result );

    // Done.
    Complete();
}

//==============================================================================

void DeprecatedSendTask::OnSendPacket( tmipc::Packet* pPacket )
{
    pPacket->WriteS32( m_Socket );
    PopulatePacket( pPacket );

    if( m_AmountLeftToWrite <= 0 )
    {
        // Done, so no more packets needed.
        SetNeedPackets( false );
    }
}

//==============================================================================

void DeprecatedSendTask::Send( const void* pBuffer, size_t bufferByteSize, int flags )
{
    DEJA_TRACE( "HTCSSendTask::Send", "HTCSSendTask::Send %d bytes on socket %d, flags = %d", bufferByteSize, m_Socket, flags );

    //Init these.
    m_Flags             = flags;
    m_pReadFromBuffer   = (char*)pBuffer;
    m_AmountLeftToWrite = bufferByteSize;
    m_NumberOfPacketsSent = 0;
    m_Result            = 0;
    m_ErrorCode         = 0;

    //Submit the packet
    tmipc::Packet* p = AllocSendPacket( true );
    p->WriteS32( m_Socket );
    PopulatePacket( p );

    //Do we need to write more?
    if( m_AmountLeftToWrite > 0 )
    {
        SetNeedPackets();
    }

    m_pServicesManager->SubmitTask( this, p );

}

//==============================================================================

void DeprecatedSendTask::PopulatePacket( tmipc::Packet* pPacket )
{
    //const int SIZE_OF_SEND_BUFFER_HEADER = ( (SIZE_OF_HTCS_HEADER) + sizeof(s32)  + sizeof(s32) + sizeof(u8) );
    //const int SIZE_OF_SEND_DATA_BUFFER = ( (tmipc::MAX_PACKET_DATA) - (SIZE_OF_SEND_BUFFER_HEADER) );

    s32 WriteThisPass = (s32)(m_AmountLeftToWrite > SIZE_OF_SEND_DATA_BUFFER ? SIZE_OF_SEND_DATA_BUFFER : m_AmountLeftToWrite);

    pPacket->WriteS32( m_Flags );
    pPacket->WriteS32( WriteThisPass );

    //Last bite?
    m_AmountLeftToWrite -= WriteThisPass;
    u8 LastPacket = ( m_AmountLeftToWrite <= 0 );

    pPacket->WriteU8( LastPacket );

    //Pack out the rest of the buffer.
    pPacket->WriteData( m_pReadFromBuffer, WriteThisPass );
    m_NumberOfPacketsSent += 1;
    DEJA_TRACE( "HTCSSendTask::PopulatePacket", "HTCSSendTask %d:  LastPacket = %d, Write this pass = %d, Amount Left = %d, sent %d packets", GetTaskId(), LastPacket, WriteThisPass, m_AmountLeftToWrite, m_NumberOfPacketsSent );

    //Get ready for next time.
    m_pReadFromBuffer     += WriteThisPass;
}

//==============================================================================
//==============================================================================
} }
//==============================================================================
