﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "hio_directory_task.h"
#include "hio_directory_service.h"
#include "hio_opcodes.h"

//==============================================================================
namespace tma { namespace hio {
//==============================================================================

HostDirectoryIOTask::HostDirectoryIOTask( TMA_HOSTIO_DIRECTORY_HANDLE Handle )
{
    DEJA_TRACE( "HostDirectoryIOTask::HostDirectoryIOTask", "HostDirectoryIOTask::HostDirectoryIOTask, Handle = %lld", Handle );
    m_Handle            = Handle;
    m_OperationValue    = 0;
    m_Result            = 0;
}

//==============================================================================

HostDirectoryIOTask::~HostDirectoryIOTask()
{
}

//==============================================================================

HostDirectoryIOReadTask::HostDirectoryIOReadTask( TMA_HOSTIO_DIRECTORY_HANDLE Handle )
: HostDirectoryIOTask( Handle )
{
    DEJA_TRACE( "HostDirectoryIOReadTask::HostDirectoryIOReadTask", "HostDirectoryIOReadTask::HostDirectoryIOReadTask, Handle = %lld", Handle );
    m_pBytesRead32          = NULL;
    m_pBytesRead            = NULL;
    m_NumberOfEntriesToRead = 0;
    m_pReadToAddr           = NULL;
    m_FSType                = false;
}

//==============================================================================

HostDirectoryIOReadTask::~HostDirectoryIOReadTask()
{
}

//==============================================================================

tmipc::Packet* HostDirectoryIOTask::PreparePacket( s32 TaskType )
{
    DEJA_TRACE( "HostDirectoryIOTask::PreparePacket", "PreparePacket for task: %d", TaskType );
    m_TaskType = TaskType;
    tmipc::Packet* p = AllocSendPacket( true );
    p->WriteU64( m_Handle );
    return p;
}

//==============================================================================

enum
{
    DIRECTORY_READ_ATTRIBUTE_ISDIRECTORY_FLAG   = (1 << 0),
    DIRECTORY_READ_ATTRIBUTE_ISHIDDEN_FLAG      = (1 << 1),
    DIRECTORY_READ_ATTRIBUTE_ISARCHIVE_FLAG     = (1 << 2),
    DIRECTORY_READ_ATTRIBUTE_ISREADONLY_FLAG    = (1 << 3),
    DIRECTORY_IO_ERROR = -1,
    DIRECTORY_IO_HAVE_ENTRY = 1,
    DIRECTORY_IO_NO_MORE_FILES = 0
};

#define DIRECTORY_READ_ATTRIBUTE_FLAG_SET(Flag, Value)  (((Value) & (Flag)) == (Flag))

//==============================================================================

void HostDirectoryIOReadTask::HandleFSPacketEntry( tmipc::Packet* pPacket )
{
    //Read this entry
    nn::fs::DirectoryEntry* pDirEntry = (nn::fs::DirectoryEntry*) m_pReadToAddr;

    //First, decode the attributes
    u8 Attributes = 0;
    pPacket->ReadU8 ( Attributes );

    bool IsDirectory = DIRECTORY_READ_ATTRIBUTE_FLAG_SET(DIRECTORY_READ_ATTRIBUTE_ISDIRECTORY_FLAG, Attributes);
    if( IsDirectory )
    {
        pDirEntry->directoryEntryType = nn::fs::DirectoryEntryType_Directory;
    }
    else
    {
        pDirEntry->directoryEntryType = nn::fs::DirectoryEntryType_File;
    }

    //Now the entry name.
    pPacket->ReadString( pDirEntry->name, sizeof(pDirEntry->name) );

    //Size
    pPacket->ReadS64( pDirEntry->fileSize );

    //Keep track of where we are.
    if( pDirEntry->name[0] == '\0' )
    {
        m_OperationValue -= 1;
    }
    else
    {
        m_pReadToAddr           += sizeof(nn::fs::DirectoryEntry);
    }

    *m_pBytesRead = m_OperationValue;
    m_NumberOfEntriesToRead -= 1;
}

//==============================================================================

void HostDirectoryIOReadTask::OnRecvPacket( tmipc::Packet* pPacket )
{
    //Read the result
    pPacket->ReadS32 ( m_Result );

    DEJA_TRACE( "HostDirectoryIOReadTask::OnRecv", "Received result: %d", m_Result );

    m_OperationValue += 1;

    // Successful read?
    if( m_Result <= TMA_IO_RESULT_OK )
    {
        if( m_FSType == true )
        {
            HandleFSPacketEntry( pPacket );
        }
    }

    // If we're through with our packets, or there was a problem...
    if( m_NumberOfEntriesToRead == 0 || m_Result >= TMA_IO_RESULT_OK )
    {
        // Done.
        Complete();
    }
}

//==============================================================================

void HostDirectoryIOReadTask::OnInitiate( tmipc::Packet* )
{
    Complete();
}

//==============================================================================

void HostDirectoryIOReadTask::OnSendPacket( tmipc::Packet* )
{
    ASSERT( 0 && "Unreachable." );
}

//==============================================================================
// fs signature
void HostDirectoryIOReadTask::Read( int64_t* pNumEntriesRead, nn::fs::DirectoryEntry pEntries[], int64_t numEntries )
{
    m_pBytesRead            = pNumEntriesRead;
    m_NumberOfEntriesToRead = numEntries;
    m_pReadToAddr           = (char*)pEntries;
    m_FSType                = true;

    //Submit the packet
    tmipc::Packet* p = PreparePacket( tmipc::TaskType_HioDirectoryRead );
    p->WriteS32( (s32)numEntries );

    m_pServicesManager->SubmitTask( this, p );
}

//==============================================================================

s32 HostDirectoryIOTask::GetResult()
{
    return m_Result;
}

//==============================================================================

s32 HostDirectoryIOTask::GetPriorityResult()
{
    return (s32)m_OperationValue;
}

//==============================================================================

bool HostDirectoryIOTask::GetExistsResult()
{
    return (m_OperationValue != 0);
}

//==============================================================================

s64 HostDirectoryIOTask::GetCountResult()
{
    return m_OperationValue;
}

//==============================================================================

TMA_HOSTIO_DIRECTORY_HANDLE HostDirectoryIOTask::GetHandle()
{
    return m_Handle;
}

//==============================================================================

HostDirectoryIOExistsTask::HostDirectoryIOExistsTask( TMA_HOSTIO_DIRECTORY_HANDLE Handle )
    : HostDirectoryIOTask( Handle )
{
};

//==============================================================================

void HostDirectoryIOExistsTask::OnRecvPacket( tmipc::Packet* pPacket )
{
    DEJA_TRACE( "HostDirectoryIOExistsTask::OnRecvPacket", "Received command: %d", pPacket->GetTaskType() );

    // Read our code and handle.
    pPacket->ReadU64( m_Handle );

    //Get the result
    pPacket->ReadS32( m_Result );
    pPacket->ReadS64( m_OperationValue );

    // Task complete.
    Complete();
}

//==============================================================================

void HostDirectoryIOExistsTask::OnInitiate( tmipc::Packet* )
{
    Complete();
}

//==============================================================================

void HostDirectoryIOExistsTask::OnSendPacket( tmipc::Packet* )
{
    ASSERT( 0 && "Unreachable." );
}

//==============================================================================

void HostDirectoryIOExistsTask::Exists( const char* pFullPath )
{
    DEJA_TRACE( "HostDirectoryIOExistsTask::Exists", "Exists %s", pFullPath );
    tmipc::Packet* p = PreparePacket( tmipc::TaskType_HioDirectoryExists );
    p->WriteString( pFullPath );
    m_pServicesManager->SubmitTask( this, p );
}

//==============================================================================

HostDirectoryIOOpenTask::HostDirectoryIOOpenTask( TMA_HOSTIO_DIRECTORY_HANDLE Handle )
    : HostDirectoryIOTask( Handle )
{
};

//==============================================================================

void HostDirectoryIOOpenTask::OnRecvPacket( tmipc::Packet* pPacket )
{
    DEJA_TRACE( "HostDirectoryIOOpenTask::OnRecvPacket", "Received command: %d", pPacket->GetTaskType() );

    // Read our code and handle.
    pPacket->ReadU64( m_Handle );

    //Get the result
    pPacket->ReadS32( m_Result );
    pPacket->ReadS64( m_OperationValue );

    u64 Addr = 0;
    pPacket->ReadU64( Addr );
    *((u64*)Addr) = m_Handle;

    // Task complete.
    Complete();
}

//==============================================================================

void HostDirectoryIOOpenTask::OnInitiate( tmipc::Packet* )
{
    Complete();
}

//==============================================================================

void HostDirectoryIOOpenTask::OnSendPacket( tmipc::Packet* )
{
    ASSERT( 0 && "Unreachable." );
}

//==============================================================================

void HostDirectoryIOOpenTask::Open( TMA_HOSTIO_DIRECTORY_HANDLE* pHandle, const char* pFullPath, int32_t OpenMode )
{
    DEJA_TRACE( "HostDirectoryIOOpenTask::Open", "Open %s", pFullPath );
    tmipc::Packet* p = PreparePacket( tmipc::TaskType_HioDirectoryOpen );
    p->WriteString( pFullPath );
    p->WriteU64( (u64)pHandle );
    p->WriteS32( OpenMode );
    m_pServicesManager->SubmitTask( this, p );
}

//==============================================================================

HostDirectoryIOGetEntryCountTask::HostDirectoryIOGetEntryCountTask( TMA_HOSTIO_DIRECTORY_HANDLE Handle )
    : HostDirectoryIOTask( Handle )
{
};

//==============================================================================

void HostDirectoryIOGetEntryCountTask::OnRecvPacket( tmipc::Packet* pPacket )
{
    DEJA_TRACE( "HostDirectoryIOGetEntryCountTask::OnRecvPacket", "Received command: %d", pPacket->GetTaskType() );

    // Read our code and handle.
    pPacket->ReadU64( m_Handle );

    //Get the result
    pPacket->ReadS32( m_Result );
    pPacket->ReadS64( m_OperationValue );

    // Task complete.
    Complete();
}

//==============================================================================

void HostDirectoryIOGetEntryCountTask::OnInitiate( tmipc::Packet* )
{
    Complete();
}

//==============================================================================

void HostDirectoryIOGetEntryCountTask::OnSendPacket( tmipc::Packet* )
{
    ASSERT( 0 && "Unreachable." );
}

//==============================================================================

void HostDirectoryIOGetEntryCountTask::GetEntryCount()
{
    DEJA_TRACE( "HostDirectoryIOGetEntryCountTask::GetEntryCount", "GetEntryCount" );
    tmipc::Packet* p = PreparePacket( tmipc::TaskType_HioDirectoryGetEntryCount );
    m_pServicesManager->SubmitTask( this, p );
}

//==============================================================================

HostDirectoryIOCloseTask::HostDirectoryIOCloseTask( TMA_HOSTIO_DIRECTORY_HANDLE Handle )
    : HostDirectoryIOTask( Handle )
{
};

//==============================================================================

void HostDirectoryIOCloseTask::OnRecvPacket( tmipc::Packet* pPacket )
{
    DEJA_TRACE( "HostDirectoryIOCloseTask::OnRecvPacket", "Received command: %d", pPacket->GetTaskType() );

    // Read our code and handle.
    pPacket->ReadU64( m_Handle );

    //Get the result
    pPacket->ReadS32( m_Result );
    pPacket->ReadS64( m_OperationValue );

    // Task complete.
    Complete();
}

//==============================================================================

void HostDirectoryIOCloseTask::OnInitiate( tmipc::Packet* )
{
    Complete();
}

//==============================================================================

void HostDirectoryIOCloseTask::OnSendPacket( tmipc::Packet* )
{
    ASSERT( 0 && "Unreachable." );
}

//==============================================================================

void HostDirectoryIOCloseTask::Close()
{
    DEJA_TRACE( "HostDirectoryIOCloseTask::Close", "Close" );
    tmipc::Packet* p = PreparePacket( tmipc::TaskType_HioDirectoryClose );
    m_pServicesManager->SubmitTask( this, p );
}

//==============================================================================

HostDirectoryIOSetPriorityTask::HostDirectoryIOSetPriorityTask( TMA_HOSTIO_DIRECTORY_HANDLE Handle )
    : HostDirectoryIOTask( Handle )
{
};

//==============================================================================

void HostDirectoryIOSetPriorityTask::OnRecvPacket( tmipc::Packet* pPacket )
{
    DEJA_TRACE( "HostDirectoryIOSetPriorityTask::OnRecvPacket", "Received command: %d", pPacket->GetTaskType() );

    // Read our code and handle.
    pPacket->ReadU64( m_Handle );

    //Get the result
    pPacket->ReadS32( m_Result );
    pPacket->ReadS64( m_OperationValue );

    // Task complete.
    Complete();
}

//==============================================================================

void HostDirectoryIOSetPriorityTask::OnInitiate( tmipc::Packet* )
{
    Complete();
}

//==============================================================================

void HostDirectoryIOSetPriorityTask::OnSendPacket( tmipc::Packet* )
{
    ASSERT( 0 && "Unreachable." );
}

//==============================================================================

void HostDirectoryIOSetPriorityTask::SetPriority( s32 Priority )
{
    DEJA_TRACE( "HostDirectoryIOSetPriorityTask::SetPriority", "SetPriority to %d", Priority );
    tmipc::Packet* p = PreparePacket( tmipc::TaskType_HioDirectorySetPriority );
    p->WriteS32( Priority );
    m_pServicesManager->SubmitTask( this, p );
}

//==============================================================================

HostDirectoryIOGetPriorityTask::HostDirectoryIOGetPriorityTask( TMA_HOSTIO_DIRECTORY_HANDLE Handle )
    : HostDirectoryIOTask( Handle )
{
};

//==============================================================================

void HostDirectoryIOGetPriorityTask::OnRecvPacket( tmipc::Packet* pPacket )
{
    DEJA_TRACE( "HostDirectoryIOGetPriorityTask::OnRecvPacket", "Received command: %d", pPacket->GetTaskType() );

    // Read our code and handle.
    pPacket->ReadU64( m_Handle );

    //Get the result
    pPacket->ReadS32( m_Result );
    pPacket->ReadS64( m_OperationValue );

    // Task complete.
    Complete();
}

//==============================================================================

void HostDirectoryIOGetPriorityTask::OnInitiate( tmipc::Packet* )
{
    Complete();
}

//==============================================================================

void HostDirectoryIOGetPriorityTask::OnSendPacket( tmipc::Packet* )
{
    ASSERT( 0 && "Unreachable." );
}

//==============================================================================

void HostDirectoryIOGetPriorityTask::GetPriority()
{
    DEJA_TRACE( "HostDirectoryIOGetPriorityTask::GetPriority", "GetPriority" );
    tmipc::Packet* p = PreparePacket( tmipc::TaskType_HioDirectoryGetPriority );
    m_pServicesManager->SubmitTask( this, p );
}

//==============================================================================

HostDirectoryIOCreateTask::HostDirectoryIOCreateTask( TMA_HOSTIO_DIRECTORY_HANDLE Handle )
    : HostDirectoryIOTask( Handle )
{
};

//==============================================================================

void HostDirectoryIOCreateTask::OnRecvPacket( tmipc::Packet* pPacket )
{
    DEJA_TRACE( "HostDirectoryIOCreateTask::OnRecvPacket", "Received command: %d", pPacket->GetTaskType() );

    // Read our code and handle.
    pPacket->ReadU64( m_Handle );

    //Get the result
    pPacket->ReadS32( m_Result );
    pPacket->ReadS64( m_OperationValue );

    // Task complete.
    Complete();
}

//==============================================================================

void HostDirectoryIOCreateTask::OnInitiate( tmipc::Packet* )
{
    Complete();
}

//==============================================================================

void HostDirectoryIOCreateTask::OnSendPacket( tmipc::Packet* )
{
    ASSERT( 0 && "Unreachable." );
}

//==============================================================================

void HostDirectoryIOCreateTask::Create( const char* pFullPath )
{
    DEJA_TRACE( "HostDirectoryIOCreateTask::Create", "Create %s", pFullPath );
    tmipc::Packet* p = PreparePacket( tmipc::TaskType_HioDirectoryCreate );
    p->WriteString( pFullPath );
    m_pServicesManager->SubmitTask( this, p );
}

//==============================================================================

HostDirectoryIODeleteTask::HostDirectoryIODeleteTask( TMA_HOSTIO_DIRECTORY_HANDLE Handle )
    : HostDirectoryIOTask( Handle )
{
};

//==============================================================================

void HostDirectoryIODeleteTask::OnRecvPacket( tmipc::Packet* pPacket )
{
    DEJA_TRACE( "HostDirectoryIODeleteTask::OnRecvPacket", "Received command: %d", pPacket->GetTaskType() );

    // Read our code and handle.
    pPacket->ReadU64( m_Handle );

    //Get the result
    pPacket->ReadS32( m_Result );
    pPacket->ReadS64( m_OperationValue );

    // Task complete.
    Complete();
}

//==============================================================================

void HostDirectoryIODeleteTask::OnInitiate( tmipc::Packet* )
{
    Complete();
}

//==============================================================================

void HostDirectoryIODeleteTask::OnSendPacket( tmipc::Packet* )
{
    ASSERT( 0 && "Unreachable." );
}

//==============================================================================

void HostDirectoryIODeleteTask::Delete( const char* pFullPath, u8 Recursively )
{
    DEJA_TRACE( "HostDirectoryIODeleteTask::Delete", "Delete %s", pFullPath );
    tmipc::Packet* p = PreparePacket( tmipc::TaskType_HioDirectoryDelete );
    p->WriteString( pFullPath );
    p->WriteU8( Recursively );
    m_pServicesManager->SubmitTask( this, p );
}

//==============================================================================

HostDirectoryIORenameTask::HostDirectoryIORenameTask( TMA_HOSTIO_DIRECTORY_HANDLE Handle )
    : HostDirectoryIOTask( Handle )
{
};

//==============================================================================

void HostDirectoryIORenameTask::OnRecvPacket( tmipc::Packet* pPacket )
{
    DEJA_TRACE( "HostDirectoryIORenameTask::OnRecvPacket", "Received command: %d", pPacket->GetTaskType() );

    // Read our code and handle.
    pPacket->ReadU64( m_Handle );

    //Get the result
    pPacket->ReadS32( m_Result );
    pPacket->ReadS64( m_OperationValue );

    // Task complete.
    Complete();
}

//==============================================================================

void HostDirectoryIORenameTask::OnInitiate( tmipc::Packet* )
{
    Complete();
}

//==============================================================================

void HostDirectoryIORenameTask::OnSendPacket( tmipc::Packet* )
{
    ASSERT( 0 && "Unreachable." );
}

//==============================================================================

void HostDirectoryIORenameTask::Rename( const char* pFrom, const char* pTo )
{
    DEJA_TRACE( "HostDirectoryIORenameTask::Rename", "Rename %s to %s", pFrom, pTo );
    tmipc::Packet* p = PreparePacket( tmipc::TaskType_HioDirectoryRename );
    p->WriteString( pFrom );
    p->WriteString( pTo );
    m_pServicesManager->SubmitTask( this, p );
}

//==============================================================================
} }
//==============================================================================
