﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "hio_api.h"
#include "..\tmagent.h"
#include "hio_directory_service.h"
#include "hio_service.h"
#include "hio_opcodes.h"
#include <nn/util/util_FormatString.h>
#include <nn/nn_SdkAssert.h>
#include <nn/fs/fs_Result.h>

//==============================================================================
namespace nn { namespace tma {
//==============================================================================

//==============================================================================
namespace file_io {
//==============================================================================

//==============================================================================

template <class T> static s32 CompleteTask( T* pTask  )
{
    pTask->WaitComplete( TMIPC_INFINITE );
    s32 ResultCode = pTask->GetResult();
    pTask->~T();
    ::tma::s_Deallocate( pTask, sizeof( T ) );
    return ResultCode;
}

//==============================================================================

nn::Result OpenFile( TMA_FILE_HANDLE* pHandle, const char* pPath, uint32_t openMode )
{
    if (::tma::GetEnableDisconnectionEmulationFlag())
    {
        return nn::fs::ResultTargetNotFound();
    }

    ::tma::hio::AgentHostIOService& HostService = ::tma::hio::GetHostIOService();
    ::tma::hio::HostIOOpenTask* pTask = HostService.OpenFile( openMode, pPath );
    pTask->WaitComplete( TMIPC_INFINITE );
    *pHandle = pTask->GetHandle();
    s32 ResultCode = pTask->GetResult();
    pTask->~HostIOOpenTask();
    ::tma::s_Deallocate( pTask, sizeof( ::tma::hio::HostIOOpenTask ) );

    nn::Result Res;

    switch( ResultCode )
    {
    case ::tma::hio::TMA_IO_RESULT_OK:              Res = nn::ResultSuccess();                      break;
    case ::tma::hio::TMA_IO_RESULT_PATH_NOT_FOUND:  Res = nn::fs::ResultPathNotFound();             break;
    case ::tma::hio::TMA_IO_RESULT_ENTRY_CORRUPTED: Res = nn::fs::ResultHostEntryCorrupted();       break;
    case ::tma::hio::TMA_IO_RESULT_TARGET_LOCKED:   Res = nn::fs::ResultTargetLocked();             break;
    case ::tma::hio::TMA_IO_RESULT_ALLOC_FAILED:    Res = nn::fs::ResultAllocationMemoryFailed();   break;
    case ::tma::hio::TMA_IO_RESULT_DISCONNECTED:    Res = nn::fs::ResultTargetNotFound();           break;
    default:                                        Res = nn::fs::ResultUnexpected();               break;
    }

    if( Res.IsFailure() )
        pHandle = NULL;

    return( Res );
}

//==============================================================================

nn::Result ReadFile( int32_t* pOut, const TMA_FILE_HANDLE handle, int64_t offset, void* pBuffer, int64_t size )
{
    if( handle != 0 )
    {
        if (::tma::GetEnableDisconnectionEmulationFlag())
        {
            return nn::fs::ResultTargetNotFound();
        }

        ::tma::hio::AgentHostIOService& HostService = ::tma::hio::GetHostIOService();
        s32 ResultCode = CompleteTask( HostService.ReadFile( pOut, handle, offset, pBuffer, size ) );

        nn::Result Res;

        switch( ResultCode )
        {
        case ::tma::hio::TMA_IO_RESULT_MORE_ENTRIES:
        case ::tma::hio::TMA_IO_RESULT_OK:                  Res = nn::ResultSuccess();                   break;
        case ::tma::hio::TMA_IO_RESULT_FILE_DATA_CORRUPTED: Res = nn::fs::ResultHostFileDataCorrupted(); break;
        case ::tma::hio::TMA_IO_RESULT_FILE_CORRUPTED:      Res = nn::fs::ResultHostFileCorrupted();     break;
        case ::tma::hio::TMA_IO_RESULT_OUT_OF_RANGE:        Res = nn::fs::ResultOutOfRange();            break;
        case ::tma::hio::TMA_IO_RESULT_DISCONNECTED:        Res = nn::fs::ResultTargetNotFound();        break;
        default:                                            Res = nn::fs::ResultUnexpected();            break;
        }

        return( Res );
    }
    return nn::fs::ResultInvalidHostHandle();
}

//==============================================================================

nn::Result ReadFile( size_t* pOut, const TMA_FILE_HANDLE handle, int64_t offset, void* pBuffer, size_t size, const nn::fs::ReadOption& option )
{
    (void)option;
    if( handle != 0 )
    {
        if (::tma::GetEnableDisconnectionEmulationFlag())
        {
            return nn::fs::ResultTargetNotFound();
        }

        ::tma::hio::AgentHostIOService& HostService = ::tma::hio::GetHostIOService();
        int32_t Out = 0;
        s32 ResultCode = CompleteTask( HostService.ReadFile( &Out, handle, offset, pBuffer, size ) );
        *pOut = Out;

        nn::Result Res;

        switch( ResultCode )
        {
        case ::tma::hio::TMA_IO_RESULT_MORE_ENTRIES:
        case ::tma::hio::TMA_IO_RESULT_OK:                  Res = nn::ResultSuccess();                   break;
        case ::tma::hio::TMA_IO_RESULT_FILE_DATA_CORRUPTED: Res = nn::fs::ResultHostFileDataCorrupted(); break;
        case ::tma::hio::TMA_IO_RESULT_FILE_CORRUPTED:      Res = nn::fs::ResultHostFileCorrupted();     break;
        case ::tma::hio::TMA_IO_RESULT_OUT_OF_RANGE:        Res = nn::fs::ResultOutOfRange();            break;
        case ::tma::hio::TMA_IO_RESULT_DISCONNECTED:        Res = nn::fs::ResultTargetNotFound();        break;
        default:                                            Res = nn::fs::ResultUnexpected();            break;
        }

        return( Res );
    }
    return nn::fs::ResultInvalidHostHandle();
}

//==============================================================================

nn::Result WriteFile( int32_t* pOut, const TMA_FILE_HANDLE handle, int64_t offset, const void* pBuffer, int64_t size, bool flush )
{
    (void)flush;
    if( handle != 0 )
    {
        if (::tma::GetEnableDisconnectionEmulationFlag())
        {
            return nn::fs::ResultTargetNotFound();
        }

        ::tma::hio::AgentHostIOService& HostService = ::tma::hio::GetHostIOService();
        s32 ResultCode = CompleteTask( HostService.WriteFile( pOut, handle, offset, (void*)pBuffer, size ) );

        nn::Result Res;

        switch( ResultCode )
        {
        case ::tma::hio::TMA_IO_RESULT_OK:                  Res = nn::ResultSuccess();                  break;
        case ::tma::hio::TMA_IO_RESULT_FILE_CORRUPTED:      Res = nn::fs::ResultHostFileCorrupted();    break;
        case ::tma::hio::TMA_IO_RESULT_OUT_OF_RANGE:        Res = nn::fs::ResultOutOfRange();           break;
        case ::tma::hio::TMA_IO_RESULT_INSUFFICIENT_SPACE:  Res = nn::fs::ResultUsableSpaceNotEnough(); break;
        case ::tma::hio::TMA_IO_RESULT_DISCONNECTED:        Res = nn::fs::ResultTargetNotFound();       break;
        default:                                            Res = nn::fs::ResultUnexpected();           break;
        }

        return( Res );
    }
    return nn::fs::ResultInvalidHostHandle();
}

//==============================================================================

nn::Result WriteFile( const TMA_FILE_HANDLE handle, int64_t offset, const void* pBuffer, int64_t size, const nn::fs::WriteOption& option)
{
    (void)option;
    if( handle != 0 )
    {
        if (::tma::GetEnableDisconnectionEmulationFlag())
        {
            return nn::fs::ResultTargetNotFound();
        }

        ::tma::hio::AgentHostIOService& HostService = ::tma::hio::GetHostIOService();
        int32_t Out = 0;
        s32 ResultCode = CompleteTask( HostService.WriteFile( &Out, handle, offset, (void*)pBuffer, size ) );

        nn::Result Res;

        switch( ResultCode )
        {
        case ::tma::hio::TMA_IO_RESULT_OK:                  Res = nn::ResultSuccess();                  break;
        case ::tma::hio::TMA_IO_RESULT_FILE_CORRUPTED:      Res = nn::fs::ResultHostFileCorrupted();    break;
        case ::tma::hio::TMA_IO_RESULT_OUT_OF_RANGE:        Res = nn::fs::ResultOutOfRange();           break;
        case ::tma::hio::TMA_IO_RESULT_INSUFFICIENT_SPACE:  Res = nn::fs::ResultUsableSpaceNotEnough(); break;
        case ::tma::hio::TMA_IO_RESULT_DISCONNECTED:        Res = nn::fs::ResultTargetNotFound();       break;
        default:                                            Res = nn::fs::ResultUnexpected();           break;
        }

        return( Res );
    }
    return nn::fs::ResultInvalidHostHandle();
}

//==============================================================================

nn::Result GetFileSize( int64_t* pOut, const TMA_FILE_HANDLE handle )
{
    if( handle != 0 )
    {
        if (::tma::GetEnableDisconnectionEmulationFlag())
        {
            return nn::fs::ResultTargetNotFound();
        }

        ::tma::hio::AgentHostIOService& HostService = ::tma::hio::GetHostIOService();
        ::tma::hio::HostIOGetSizeTask* pTask = HostService.GetFileSize( handle);
        pTask->WaitComplete( TMIPC_INFINITE );
        *pOut = pTask->GetSizeResult();
        s32 ResultCode = pTask->GetResult();
        pTask->~HostIOGetSizeTask();
        ::tma::s_Deallocate( pTask, sizeof( ::tma::hio::HostIOGetSizeTask ) );

        nn::Result Res;

        switch( ResultCode )
        {
        case ::tma::hio::TMA_IO_RESULT_OK:              Res = nn::ResultSuccess();               break;
        case ::tma::hio::TMA_IO_RESULT_FILE_CORRUPTED:  Res = nn::fs::ResultHostFileCorrupted(); break;
        case ::tma::hio::TMA_IO_RESULT_DISCONNECTED:    Res = nn::fs::ResultTargetNotFound();    break;
        default:                                        Res = nn::fs::ResultUnexpected();        break;
        }

        return( Res );
    }

    return nn::fs::ResultInvalidHostHandle();
}

//==============================================================================

nn::Result SetFileSize( const TMA_FILE_HANDLE handle, int64_t size )
{
    if( handle != 0 )
    {
        if (::tma::GetEnableDisconnectionEmulationFlag())
        {
            return nn::fs::ResultTargetNotFound();
        }

        ::tma::hio::AgentHostIOService& HostService = ::tma::hio::GetHostIOService();
        s32 ResultCode = CompleteTask( HostService.SetFileSize( handle, size ) );

        nn::Result Res;

        switch( ResultCode )
        {
        case ::tma::hio::TMA_IO_RESULT_OK:                  Res = nn::ResultSuccess();                  break;
        case ::tma::hio::TMA_IO_RESULT_OUT_OF_RANGE:        Res = nn::fs::ResultOutOfRange();           break;
        case ::tma::hio::TMA_IO_RESULT_FILE_CORRUPTED:      Res = nn::fs::ResultHostFileCorrupted();    break;
        case ::tma::hio::TMA_IO_RESULT_INSUFFICIENT_SPACE:  Res = nn::fs::ResultUsableSpaceNotEnough(); break;
        case ::tma::hio::TMA_IO_RESULT_DISCONNECTED:        Res = nn::fs::ResultTargetNotFound();       break;
        default:                                            Res = nn::fs::ResultUnexpected();           break;
        }

        return( Res );
    }

    return nn::fs::ResultInvalidHostHandle();
}

//==============================================================================

nn::Result GetFileTimeStamp( uint64_t* pCreateTime, uint64_t* pAccessTime, uint64_t* pModifyTime, const char* pPath )
{
    if (::tma::GetEnableDisconnectionEmulationFlag())
    {
        return nn::fs::ResultTargetNotFound();
    }

    ::tma::hio::AgentHostIOService& HostService = ::tma::hio::GetHostIOService();
    ::tma::hio::HostIOGetFileTimeStampTask* pTask = HostService.GetFileTimeStamp( pPath );
    pTask->WaitComplete( TMIPC_INFINITE );
    pTask->GetTimeStampResults( pCreateTime, pAccessTime, pModifyTime );
    s32 ResultCode = pTask->GetResult();
    delete pTask;

    nn::Result Res;

    switch( ResultCode )
    {
    case ::tma::hio::TMA_IO_RESULT_OK:              Res = nn::ResultSuccess();              break;
    case ::tma::hio::TMA_IO_RESULT_PATH_NOT_FOUND:  Res = nn::fs::ResultPathNotFound();     break;
    case ::tma::hio::TMA_IO_RESULT_DISCONNECTED:    Res = nn::fs::ResultTargetNotFound();   break;
    default:                                        Res = nn::fs::ResultUnexpected();       break;
    }

    return( Res );
}

//==============================================================================

nn::Result FlushFile( const TMA_FILE_HANDLE handle )
{
    if( handle != 0 )
    {
        if (::tma::GetEnableDisconnectionEmulationFlag())
        {
            return nn::fs::ResultTargetNotFound();
        }

        ::tma::hio::AgentHostIOService& HostService = ::tma::hio::GetHostIOService();
        s32 ResultCode = CompleteTask( HostService.FlushFile( handle ) );

        nn::Result Res;

        switch( ResultCode )
        {
        case ::tma::hio::TMA_IO_RESULT_OK:                  Res = nn::ResultSuccess();                  break;
        case ::tma::hio::TMA_IO_RESULT_FILE_CORRUPTED:      Res = nn::fs::ResultHostFileCorrupted();    break;
        case ::tma::hio::TMA_IO_RESULT_INSUFFICIENT_SPACE:  Res = nn::fs::ResultUsableSpaceNotEnough(); break;
        case ::tma::hio::TMA_IO_RESULT_DISCONNECTED:        Res = nn::fs::ResultTargetNotFound();       break;
        default:                                            Res = nn::fs::ResultUnexpected();           break;
        }

        return( Res );
    }

    return nn::fs::ResultInvalidHostHandle();
}

//==============================================================================

nn::Result SetPriorityForFile( const TMA_FILE_HANDLE handle, int32_t priority )
{
    if( handle != 0 )
    {
        if (::tma::GetEnableDisconnectionEmulationFlag())
        {
            return nn::fs::ResultTargetNotFound();
        }

        ::tma::hio::AgentHostIOService& HostService = ::tma::hio::GetHostIOService();
        s32 ResultCode = CompleteTask( HostService.SetFilePriority( handle, priority ) );

        nn::Result Res;

        switch( ResultCode )
        {
        case ::tma::hio::TMA_IO_RESULT_OK:              Res = nn::ResultSuccess();              break;
        case ::tma::hio::TMA_IO_RESULT_PATH_NOT_FOUND:  Res = nn::fs::ResultPathNotFound();     break;
        case ::tma::hio::TMA_IO_RESULT_DISCONNECTED:    Res = nn::fs::ResultTargetNotFound();   break;
        default:                                        Res = nn::fs::ResultUnexpected();       break;
        }

        return( Res );
    }

    return nn::fs::ResultPathNotFound();
}

//==============================================================================

nn::Result GetPriorityForFile( int32_t* pOut, const TMA_FILE_HANDLE handle )
{
    if( handle != 0 )
    {
        if (::tma::GetEnableDisconnectionEmulationFlag())
        {
            return nn::fs::ResultTargetNotFound();
        }

        ::tma::hio::AgentHostIOService& HostService = ::tma::hio::GetHostIOService();
        ::tma::hio::HostIOGetPriorityTask* pTask = HostService.GetFilePriority( handle);
        pTask->WaitComplete( TMIPC_INFINITE );
        *pOut = pTask->GetPriorityResult();
        s32 ResultCode = pTask->GetResult();
        pTask->~HostIOGetPriorityTask();
        ::tma::s_Deallocate( pTask, sizeof( ::tma::hio::HostIOGetPriorityTask ) );

        nn::Result Res;

        switch( ResultCode )
        {
        case ::tma::hio::TMA_IO_RESULT_OK:              Res = nn::ResultSuccess();              break;
        case ::tma::hio::TMA_IO_RESULT_PATH_NOT_FOUND:  Res = nn::fs::ResultPathNotFound();     break;
        case ::tma::hio::TMA_IO_RESULT_DISCONNECTED:    Res = nn::fs::ResultTargetNotFound();   break;
        default:                                        Res = nn::fs::ResultUnexpected();       break;
        }

        return( Res );
    }

    return nn::fs::ResultPathNotFound();
}

//==============================================================================

void CloseFile( const TMA_FILE_HANDLE handle )
{
    if( handle != 0 )
    {
        if (::tma::GetEnableDisconnectionEmulationFlag())
        {
            return;
        }

        ::tma::hio::AgentHostIOService& HostService = ::tma::hio::GetHostIOService();
        s32 ResultCode = CompleteTask( HostService.CloseFile( handle ) );
        (void) ResultCode;
    }
}

//==============================================================================

nn::Result FileExists( bool* pExists, const char* pPath )
{
    if (::tma::GetEnableDisconnectionEmulationFlag())
    {
        return nn::fs::ResultTargetNotFound();
    }

    ::tma::hio::AgentHostIOService& HostService = ::tma::hio::GetHostIOService();
    ::tma::hio::HostIOExistsTask* pTask = HostService.FileExists( pPath );
    pTask->WaitComplete( TMIPC_INFINITE );
    *pExists = pTask->GetExistsResult();
    s32 ResultCode = pTask->GetResult();
    pTask->~HostIOExistsTask();
    ::tma::s_Deallocate( pTask, sizeof( ::tma::hio::HostIOExistsTask ) );

    nn::Result Res;

    switch( ResultCode )
    {
    case ::tma::hio::TMA_IO_RESULT_OK:              Res = nn::ResultSuccess();              break;
    case ::tma::hio::TMA_IO_RESULT_DISCONNECTED:    Res = nn::fs::ResultTargetNotFound();   break;
    default:                                        Res = nn::fs::ResultUnexpected();       break;
    }

    return( Res );
}

//==============================================================================

nn::Result CreateFile( const char* pPath, int64_t size )
{
    if (::tma::GetEnableDisconnectionEmulationFlag())
    {
        return nn::fs::ResultTargetNotFound();
    }

    ::tma::hio::AgentHostIOService& HostService = ::tma::hio::GetHostIOService();
    ::tma::hio::HostIOCreateTask* pTask = HostService.CreateFile( pPath, size );
    pTask->WaitComplete( TMIPC_INFINITE );
    s32 ResultCode = pTask->GetResult();
    pTask->~HostIOCreateTask();
    ::tma::s_Deallocate( pTask, sizeof( ::tma::hio::HostIOCreateTask ) );

    nn::Result Res;

    switch( ResultCode )
    {
    case ::tma::hio::TMA_IO_RESULT_OK:                  Res = nn::ResultSuccess();                  break;
    case ::tma::hio::TMA_IO_RESULT_PATH_NOT_FOUND:      Res = nn::fs::ResultPathNotFound();         break;
    case ::tma::hio::TMA_IO_RESULT_PATH_EXISTS:         Res = nn::fs::ResultPathAlreadyExists();    break;
    case ::tma::hio::TMA_IO_RESULT_ENTRY_CORRUPTED:     Res = nn::fs::ResultHostEntryCorrupted();   break;
    case ::tma::hio::TMA_IO_RESULT_OUT_OF_RANGE:        Res = nn::fs::ResultOutOfRange();           break;
    case ::tma::hio::TMA_IO_RESULT_INSUFFICIENT_SPACE:  Res = nn::fs::ResultUsableSpaceNotEnough(); break;
    case ::tma::hio::TMA_IO_RESULT_DISCONNECTED:        Res = nn::fs::ResultTargetNotFound();       break;
    default:                                            Res = nn::fs::ResultUnexpected();           break;
    }

    return Res;
}

//==============================================================================

nn::Result DeleteFile( const char* pPath )
{
    if (::tma::GetEnableDisconnectionEmulationFlag())
    {
        return nn::fs::ResultTargetNotFound();
    }

    ::tma::hio::AgentHostIOService& HostService = ::tma::hio::GetHostIOService();
    s32 ResultCode = CompleteTask( HostService.DeleteFile( pPath ) );

    nn::Result Res;

    switch( ResultCode )
    {
    case ::tma::hio::TMA_IO_RESULT_OK:              Res = nn::ResultSuccess();                break;
    case ::tma::hio::TMA_IO_RESULT_PATH_NOT_FOUND:  Res = nn::fs::ResultPathNotFound();       break;
    case ::tma::hio::TMA_IO_RESULT_ENTRY_CORRUPTED: Res = nn::fs::ResultHostEntryCorrupted(); break;
    case ::tma::hio::TMA_IO_RESULT_TARGET_LOCKED:   Res = nn::fs::ResultTargetLocked();       break;
    case ::tma::hio::TMA_IO_RESULT_DISCONNECTED:    Res = nn::fs::ResultTargetNotFound();     break;
    default:                                        Res = nn::fs::ResultUnexpected();         break;
    }

    return( Res );
}

//==============================================================================

nn::Result RenameFile( const char* pFromName, const char* pToName )
{
    if (::tma::GetEnableDisconnectionEmulationFlag())
    {
        return nn::fs::ResultTargetNotFound();
    }

    ::tma::hio::AgentHostIOService& HostService = ::tma::hio::GetHostIOService();
    s32 ResultCode = CompleteTask( HostService.RenameFile( pFromName, pToName ) );

    nn::Result Res;

    switch( ResultCode )
    {
    case ::tma::hio::TMA_IO_RESULT_OK:              Res = nn::ResultSuccess();                  break;
    case ::tma::hio::TMA_IO_RESULT_PATH_NOT_FOUND:  Res = nn::fs::ResultPathNotFound();         break;
    case ::tma::hio::TMA_IO_RESULT_PATH_EXISTS:     Res = nn::fs::ResultPathAlreadyExists();    break;
    case ::tma::hio::TMA_IO_RESULT_ENTRY_CORRUPTED: Res = nn::fs::ResultHostEntryCorrupted();   break;
    case ::tma::hio::TMA_IO_RESULT_TARGET_LOCKED:   Res = nn::fs::ResultTargetLocked();         break;
    case ::tma::hio::TMA_IO_RESULT_DISCONNECTED:    Res = nn::fs::ResultTargetNotFound();       break;
    default:                                        Res = nn::fs::ResultUnexpected();           break;
    }

    return( Res );
}

//==============================================================================

nn::Result GetIOType( nn::fs::DirectoryEntryType* pType, const char* pPath )
{
    if (::tma::GetEnableDisconnectionEmulationFlag())
    {
        return nn::fs::ResultTargetNotFound();
    }

    ::tma::hio::AgentHostIOService& HostService = ::tma::hio::GetHostIOService();
    ::tma::hio::HostIOGetIOTypeTask* pTask = HostService.GetIOType( pPath );
    pTask->WaitComplete( TMIPC_INFINITE );
    *pType = (nn::fs::DirectoryEntryType)pTask->GetIOTypeResult();
    s32 ResultCode = pTask->GetResult();
    pTask->~HostIOGetIOTypeTask();
    ::tma::s_Deallocate( pTask, sizeof( ::tma::hio::HostIOGetIOTypeTask ) );

    nn::Result Res;

    switch( ResultCode )
    {
    case ::tma::hio::TMA_IO_RESULT_OK:              Res = nn::ResultSuccess();              break;
    case ::tma::hio::TMA_IO_RESULT_PATH_NOT_FOUND:  Res = nn::fs::ResultPathNotFound();     break;
    case ::tma::hio::TMA_IO_RESULT_DISCONNECTED:    Res = nn::fs::ResultTargetNotFound();   break;
    default:                                        Res = nn::fs::ResultUnexpected();       break;
    }

    return( Res );
}

//==============================================================================
} //file_io namespace
//==============================================================================

//==============================================================================
namespace directory_io {
//==============================================================================

//==============================================================================

static s32 CompleteTask( ::tma::hio::HostDirectoryIOTask* pTask  )
{
    pTask->WaitComplete( TMIPC_INFINITE );
    s32 ResultCode = pTask->GetResult();
    pTask->~HostDirectoryIOTask();
    ::tma::s_Deallocate( pTask, sizeof( ::tma::hio::HostDirectoryIOTask ) );
    return ResultCode;
}

//==============================================================================

nn::Result OpenDirectory( TMA_DIR_HANDLE* pHandle, const char* pathName, fs::OpenDirectoryMode OpenMode )
{
    if (::tma::GetEnableDisconnectionEmulationFlag())
    {
        return nn::fs::ResultTargetNotFound();
    }

    ::tma::hio::AgentHostDirectoryIOService& HostService = ::tma::hio::GetHostDirectoryIOService();
    s32 ResultCode = CompleteTask( HostService.OpenDirectory( pHandle, pathName, OpenMode ) );

    nn::Result Res;

    switch( ResultCode )
    {
    case ::tma::hio::TMA_IO_RESULT_OK:              Res = nn::ResultSuccess();                      break;
    case ::tma::hio::TMA_IO_RESULT_PATH_NOT_FOUND:  Res = nn::fs::ResultPathNotFound();             break;
    case ::tma::hio::TMA_IO_RESULT_ENTRY_CORRUPTED: Res = nn::fs::ResultHostEntryCorrupted();       break;
    case ::tma::hio::TMA_IO_RESULT_ALLOC_FAILED:    Res = nn::fs::ResultAllocationMemoryFailed();   break;
    case ::tma::hio::TMA_IO_RESULT_DISCONNECTED:    Res = nn::fs::ResultTargetNotFound();           break;
    default:                                        Res = nn::fs::ResultUnexpected();               break;
    }

    return( Res );
}

//==============================================================================

nn::Result GetEntryCount( int64_t* pCount, const TMA_DIR_HANDLE& handle )
{
    if( handle != 0 )
    {
        if (::tma::GetEnableDisconnectionEmulationFlag())
        {
            return nn::fs::ResultTargetNotFound();
        }

        ::tma::hio::AgentHostDirectoryIOService& HostService = ::tma::hio::GetHostDirectoryIOService();
        ::tma::hio::HostDirectoryIOTask* pTask = HostService.GetDirectoryCount( handle );
        pTask->WaitComplete( -1 );
        s32 ResultCode = pTask->GetResult();
        *pCount = pTask->GetCountResult();
        pTask->~HostDirectoryIOTask();
        ::tma::s_Deallocate( pTask, sizeof( ::tma::hio::HostDirectoryIOTask ) );

        nn::Result Res;

        switch( ResultCode )
        {
        case ::tma::hio::TMA_IO_RESULT_OK:                  Res = nn::ResultSuccess();                      break;
        case ::tma::hio::TMA_IO_RESULT_ENTRY_CORRUPTED:     Res = nn::fs::ResultHostEntryCorrupted();       break;
        case ::tma::hio::TMA_IO_RESULT_DIR_STATUS_CHANGED:  Res = nn::fs::ResultDirectoryStatusChanged();   break;
        case ::tma::hio::TMA_IO_RESULT_DISCONNECTED:        Res = nn::fs::ResultTargetNotFound();           break;
        default:                                            Res = nn::fs::ResultUnexpected();               break;
        }

        return( Res );
    }
    return nn::fs::ResultInvalidHostHandle();
}

//==============================================================================

nn::Result ReadDirectory( int64_t* pOut, nn::fs::DirectoryEntry pEntries[], const TMA_DIR_HANDLE& handle, int64_t numEntries )
{
    if( handle != 0 )
    {
        if (::tma::GetEnableDisconnectionEmulationFlag())
        {
            return nn::fs::ResultTargetNotFound();
        }

        ::tma::hio::AgentHostDirectoryIOService& HostService = ::tma::hio::GetHostDirectoryIOService();
        s32 ResultCode = CompleteTask( HostService.ReadDirectory( pOut, handle, pEntries, numEntries ) );

        nn::Result Res;

        switch( ResultCode )
        {
        case ::tma::hio::TMA_IO_RESULT_MORE_ENTRIES:
        case ::tma::hio::TMA_IO_RESULT_OK:                  Res = nn::ResultSuccess();                      break;
        case ::tma::hio::TMA_IO_RESULT_ENTRY_CORRUPTED:     Res = nn::fs::ResultHostEntryCorrupted();       break;
        case ::tma::hio::TMA_IO_RESULT_DIR_STATUS_CHANGED:  Res = nn::fs::ResultDirectoryStatusChanged();   break;
        case ::tma::hio::TMA_IO_RESULT_DISCONNECTED:        Res = nn::fs::ResultTargetNotFound();           break;
        default:                                            Res = nn::fs::ResultUnexpected();               break;
        }

        return( Res );
    }
    return nn::fs::ResultInvalidHostHandle();
}

//==============================================================================

nn::Result SetPriorityForDirectory( const TMA_DIR_HANDLE handle, int32_t priority )
{
    if( handle != 0 )
    {
        if (::tma::GetEnableDisconnectionEmulationFlag())
        {
            return nn::fs::ResultTargetNotFound();
        }

        ::tma::hio::AgentHostDirectoryIOService& HostService = ::tma::hio::GetHostDirectoryIOService();
        s32 ResultCode = CompleteTask( HostService.SetDirectoryPriority( handle, priority ) );

        nn::Result Res;

        switch( ResultCode )
        {
        case ::tma::hio::TMA_IO_RESULT_OK:              Res = nn::ResultSuccess();              break;
        case ::tma::hio::TMA_IO_RESULT_PATH_NOT_FOUND:  Res = nn::fs::ResultPathNotFound();     break;
        case ::tma::hio::TMA_IO_RESULT_DISCONNECTED:    Res = nn::fs::ResultTargetNotFound();   break;
        default:                                        Res = nn::fs::ResultUnexpected();       break;
        }

        return( Res );
    }
    return nn::fs::ResultPathNotFound();
}

//==============================================================================

nn::Result GetPriorityForDirectory( int32_t* pOut, const TMA_DIR_HANDLE handle )
{
    if( handle != 0 )
    {
        if (::tma::GetEnableDisconnectionEmulationFlag())
        {
            return nn::fs::ResultTargetNotFound();
        }

        ::tma::hio::AgentHostDirectoryIOService& HostService = ::tma::hio::GetHostDirectoryIOService();
        ::tma::hio::HostDirectoryIOTask* pTask = HostService.GetDirectoryPriority( handle );
        pTask->WaitComplete( -1 );
        s32 ResultCode = pTask->GetResult();
        *pOut = pTask->GetPriorityResult();
        pTask->~HostDirectoryIOTask();
        ::tma::s_Deallocate( pTask, sizeof( ::tma::hio::HostDirectoryIOTask ) );

        nn::Result Res;

        switch( ResultCode )
        {
        case ::tma::hio::TMA_IO_RESULT_OK:              Res = nn::ResultSuccess();              break;
        case ::tma::hio::TMA_IO_RESULT_PATH_NOT_FOUND:  Res = nn::fs::ResultPathNotFound();     break;
        case ::tma::hio::TMA_IO_RESULT_DISCONNECTED:    Res = nn::fs::ResultTargetNotFound();   break;
        default:                                        Res = nn::fs::ResultUnexpected();       break;
        }

        return( Res );
    }
    return nn::fs::ResultPathNotFound();
}

//==============================================================================

void CloseDirectory( const TMA_DIR_HANDLE handle )
{
    if( handle != 0 )
    {
        if (::tma::GetEnableDisconnectionEmulationFlag())
        {
            return;
        }

        ::tma::hio::AgentHostDirectoryIOService& HostService = ::tma::hio::GetHostDirectoryIOService();
        s32 ResultCode = CompleteTask( HostService.CloseDirectory( handle ) );
        (void) ResultCode;
    }
}

//==============================================================================

nn::Result DirectoryExists( bool* pExists, const char* pathName )
{
    if (::tma::GetEnableDisconnectionEmulationFlag())
    {
        return nn::fs::ResultTargetNotFound();
    }

    ::tma::hio::AgentHostDirectoryIOService& HostService = ::tma::hio::GetHostDirectoryIOService();
    ::tma::hio::HostDirectoryIOTask* pTask = HostService.DirectoryExists( pathName );
    pTask->WaitComplete( TMIPC_INFINITE );
    *pExists = pTask->GetExistsResult();
    s32 ResultCode = pTask->GetResult();
    pTask->~HostDirectoryIOTask();
    ::tma::s_Deallocate( pTask, sizeof( ::tma::hio::HostDirectoryIOTask ) );

    nn::Result Res;

    switch( ResultCode )
    {
    case ::tma::hio::TMA_IO_RESULT_OK:              Res = nn::ResultSuccess();              break;
    case ::tma::hio::TMA_IO_RESULT_DISCONNECTED:    Res = nn::fs::ResultTargetNotFound();   break;
    default:                                        Res = nn::fs::ResultUnexpected();       break;
    }

    return( Res );
}

//==============================================================================

nn::Result CreateDirectory( const char* pathName )
{
    if (::tma::GetEnableDisconnectionEmulationFlag())
    {
        return nn::fs::ResultTargetNotFound();
    }

    ::tma::hio::AgentHostDirectoryIOService& HostService = ::tma::hio::GetHostDirectoryIOService();
    s32 ResultCode = CompleteTask( HostService.CreateDirectory( pathName ) );

    nn::Result Res;

    switch( ResultCode )
    {
    case ::tma::hio::TMA_IO_RESULT_OK:                  Res = nn::ResultSuccess();                  break;
    case ::tma::hio::TMA_IO_RESULT_PATH_NOT_FOUND:      Res = nn::fs::ResultPathNotFound();         break;
    case ::tma::hio::TMA_IO_RESULT_PATH_EXISTS:         Res = nn::fs::ResultPathAlreadyExists();    break;
    case ::tma::hio::TMA_IO_RESULT_ENTRY_CORRUPTED:     Res = nn::fs::ResultHostEntryCorrupted();   break;
    case ::tma::hio::TMA_IO_RESULT_INSUFFICIENT_SPACE:  Res = nn::fs::ResultUsableSpaceNotEnough(); break;
    case ::tma::hio::TMA_IO_RESULT_DISCONNECTED:        Res = nn::fs::ResultTargetNotFound();       break;
    default:                                            Res = nn::fs::ResultUnexpected();           break;
    }

    return( Res );
}

//==============================================================================

nn::Result DeleteDirectory( const char* pathName, bool recursively )
{
    if (::tma::GetEnableDisconnectionEmulationFlag())
    {
        return nn::fs::ResultTargetNotFound();
    }

    ::tma::hio::AgentHostDirectoryIOService& HostService = ::tma::hio::GetHostDirectoryIOService();
    s32 ResultCode = CompleteTask( HostService.DeleteDirectory( pathName, recursively ) );

    nn::Result Res;

    switch( ResultCode )
    {
    case ::tma::hio::TMA_IO_RESULT_OK:              Res = nn::ResultSuccess();                  break;
    case ::tma::hio::TMA_IO_RESULT_PATH_NOT_FOUND:  Res = nn::fs::ResultPathNotFound();         break;
    case ::tma::hio::TMA_IO_RESULT_ENTRY_CORRUPTED: Res = nn::fs::ResultHostEntryCorrupted();   break;
    case ::tma::hio::TMA_IO_RESULT_TARGET_LOCKED:   Res = nn::fs::ResultTargetLocked();         break;
    case ::tma::hio::TMA_IO_RESULT_DISCONNECTED:    Res = nn::fs::ResultTargetNotFound();       break;
    case ::tma::hio::TMA_IO_RESULT_DIR_NOT_EMPTY: {
        if( recursively )   // Should never receive this error when deleting recursively.
            Res = nn::fs::ResultUnexpected();
        else
            Res = nn::fs::ResultDirectoryNotEmpty();
    }break;
    default:                                        Res = nn::fs::ResultUnexpected();   break;
    }

    return( Res );
}

//==============================================================================

nn::Result RenameDirectory( const char* fromName, const char* ToName )
{
    if (::tma::GetEnableDisconnectionEmulationFlag())
    {
        return nn::fs::ResultTargetNotFound();
    }

    ::tma::hio::AgentHostDirectoryIOService& HostService = ::tma::hio::GetHostDirectoryIOService();
    s32 ResultCode = CompleteTask( HostService.RenameDirectory( fromName, ToName ) );

    nn::Result Res;

    switch( ResultCode )
    {
    case ::tma::hio::TMA_IO_RESULT_OK:              Res = nn::ResultSuccess();                  break;
    case ::tma::hio::TMA_IO_RESULT_PATH_NOT_FOUND:  Res = nn::fs::ResultPathNotFound();         break;
    case ::tma::hio::TMA_IO_RESULT_PATH_EXISTS:     Res = nn::fs::ResultPathAlreadyExists();    break;
    case ::tma::hio::TMA_IO_RESULT_ENTRY_CORRUPTED: Res = nn::fs::ResultHostEntryCorrupted();   break;
    case ::tma::hio::TMA_IO_RESULT_TARGET_LOCKED:   Res = nn::fs::ResultTargetLocked();         break;
    case ::tma::hio::TMA_IO_RESULT_DISCONNECTED:    Res = nn::fs::ResultTargetNotFound();       break;
    default:                                        Res = nn::fs::ResultUnexpected();           break;
    }

    return( Res );
}

//==============================================================================
} //directory_io namespace
//==============================================================================

//==============================================================================
}} //nn::tma namespace
//==============================================================================

