﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

// このファイルはユーザ定義

#include "hio_FileManagerImpl.h"
#include "hio_FileAccessorImpl.h"
#include "hio_DirectoryAccessorImpl.h"

#include <nn/result/result_HandlingUtility.h>
#include <nn/sf/sf_ExpHeapAllocator.h>
#include <nn/sf/sf_ObjectFactory.h>
#include "hio_api.h"

namespace nn { namespace tma {

namespace {

struct FileAccessorAllocatorTag;
typedef nn::sf::ExpHeapStaticAllocator<4096 * 8, FileAccessorAllocatorTag> FileAccessorAllocator;

class FileAccessorAllocatorInitializer
{
public:
    FileAccessorAllocatorInitializer() NN_NOEXCEPT
    {
        FileAccessorAllocator::Initialize(nn::lmem::CreationOption_ThreadSafe);
    }
} g_FileAccessorAllocatorInitializer;

struct DirectoryAccesorAllocatorTag;
typedef nn::sf::ExpHeapStaticAllocator<4096 * 8, DirectoryAccesorAllocatorTag> DirectoryAccesorAllocator;

class DirectoryAccesorAllocatorInitializer
{
public:
    DirectoryAccesorAllocatorInitializer() NN_NOEXCEPT
    {
        DirectoryAccesorAllocator::Initialize(nn::lmem::CreationOption_ThreadSafe);
    }
} g_DirectoryAccesorAllocatorInitializer;

}

nn::Result FileManagerImpl::OpenFile(nn::sf::Out<nn::sf::SharedPointer<nn::tma::IFileAccessor>> handle, const nn::tma::Path& pPath, std::uint32_t openMode) NN_NOEXCEPT
{
    file_io::TMA_FILE_HANDLE fileHandle;
    NN_RESULT_DO( file_io::OpenFile( &fileHandle, pPath.str, openMode ) );

    auto p = nn::sf::ObjectFactory<FileAccessorAllocator::Policy>::CreateSharedEmplaced<
        IFileAccessor,
        FileAccessorImpl
    >(fileHandle);

    handle.Set(std::move(p));
    return ResultSuccess();
}

nn::Result FileManagerImpl::FileExists(nn::sf::Out<bool> pExists, const nn::tma::Path& pPath) NN_NOEXCEPT
{
    bool exists;
    NN_RESULT_DO( file_io::FileExists( &exists, pPath.str ) );
    pExists.Set( exists );
    NN_RESULT_SUCCESS;
}

nn::Result FileManagerImpl::DeleteFile(const nn::tma::Path& pPath) NN_NOEXCEPT
{
    return file_io::DeleteFile( pPath.str );
}

nn::Result FileManagerImpl::RenameFile(const nn::tma::Path& pFromName, const nn::tma::Path& pToName) NN_NOEXCEPT
{
    return file_io::RenameFile( pFromName.str, pToName.str );
}

nn::Result FileManagerImpl::GetIOType(nn::sf::Out<std::int32_t> pType, const nn::tma::Path& pPath) NN_NOEXCEPT
{
    nn::fs::DirectoryEntryType type;
    NN_RESULT_DO( file_io::GetIOType( &type, pPath.str ) );
    pType.Set( type );
    NN_RESULT_SUCCESS;
}

nn::Result FileManagerImpl::OpenDirectory(nn::sf::Out<nn::sf::SharedPointer<nn::tma::IDirectoryAccessor>> pHandle, const nn::tma::Path& pathName, std::int32_t openMode) NN_NOEXCEPT
{
    directory_io::TMA_DIR_HANDLE directoryHandle;
    NN_RESULT_DO( directory_io::OpenDirectory( &directoryHandle, pathName.str, static_cast<nn::fs::OpenDirectoryMode>(openMode) ) );

    auto p = nn::sf::ObjectFactory<DirectoryAccesorAllocator::Policy>::CreateSharedEmplaced<
        IDirectoryAccessor,
        DirectoryAccessorImpl
    >(directoryHandle);

    pHandle.Set(std::move(p));
    NN_RESULT_SUCCESS;
}

nn::Result FileManagerImpl::DirectoryExists(nn::sf::Out<bool> pExists, const nn::tma::Path& pathName) NN_NOEXCEPT
{
    bool exists;
    NN_RESULT_DO( directory_io::DirectoryExists( &exists, pathName.str ) );
    pExists.Set( exists );
    NN_RESULT_SUCCESS;
}

nn::Result FileManagerImpl::CreateDirectory(const nn::tma::Path& pathName) NN_NOEXCEPT
{
    return directory_io::CreateDirectory( pathName.str );
}

nn::Result FileManagerImpl::DeleteDirectory(const nn::tma::Path& pathName, bool recursively) NN_NOEXCEPT
{
    return directory_io::DeleteDirectory( pathName.str, recursively );
}

nn::Result FileManagerImpl::RenameDirectory(const nn::tma::Path& fromName, const nn::tma::Path& toName) NN_NOEXCEPT
{
    return directory_io::RenameDirectory( fromName.str, toName.str );
}

nn::Result FileManagerImpl::CreateFile(const nn::tma::Path& pPath, std::int64_t size) NN_NOEXCEPT
{
    return file_io::CreateFile( pPath.str, size );
}

nn::Result FileManagerImpl::GetFileTimeStamp( nn::sf::Out<std::uint64_t> pCreateTime, nn::sf::Out<std::uint64_t> pAccessTime, nn::sf::Out<std::uint64_t> pModifyTime, const nn::tma::Path& pathName )
{
    std::uint64_t CreateTime, AccessTime, ModifyTime;
    nn::Result Res = file_io::GetFileTimeStamp( &CreateTime, &AccessTime, &ModifyTime, pathName.str );
    pCreateTime.Set( CreateTime );
    pAccessTime.Set( AccessTime );
    pModifyTime.Set( ModifyTime );
    return Res;
}

}}
