﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_SystemThreadDefinition.h>
#include "dbg_Task.h"
#include "dbg_Service.h"
#include "..\tmipc\tmipc_packet.h"
#include "dbg_ProcessMgr.h"

//==============================================================================
namespace tma { namespace dbg {
//==============================================================================

//==============================================================================

DebugService::DebugService()
:m_pProcessMgr( NULL )
{
    TMA_TRACE( "DebugService", "DebugService" );

    // Let DebugTask initialize.
    DebugTask::Initialize();

    m_ServiceId = HashString( "DebugService" );
    m_DispatchTaskQ.Create( 4 );
    m_DispatchTaskThread.Start( DispatchTaskThread, this, (16 * 1024), NN_SYSTEM_THREAD_PRIORITY(tma, DbgDispatchTask), NN_SYSTEM_THREAD_NAME(tma, DbgDispatchTask) );
}

//==============================================================================

DebugService::~DebugService()
{
    void* pMem = s_Allocate( sizeof(DispatchDebugTask) );
    m_DispatchTaskQ.Send( new (pMem) DispatchDebugTask() );
    m_DispatchTaskThread.Join();
    m_DispatchTaskThread.Destroy();
    while( m_DispatchTaskQ.GetCount() )
    {
        delete (DispatchDebugTask*)m_DispatchTaskQ.Receive();
    }
    m_DispatchTaskQ.Destroy();

    // Let DebugTask shut down.
    DebugTask::Finalize();
}

//==============================================================================

void* DebugService::DispatchTaskThread( void* pArg )
{
    bool Continue = true;
    DebugService* pThis = (DebugService*)pArg;
    while( Continue )
    {
        TMA_TRACE( "DebugService", "DispatchTaskThread - Wait Task" );
        DispatchDebugTask* Recv = (DispatchDebugTask*)pThis->m_DispatchTaskQ.Receive();
        TMA_TRACE( "DebugService", "DispatchTaskThread - Got Task" );
        if( Recv->m_ShouldDispatch )
        {
            tmipc::Task* pTask = NULL;
            switch( Recv->m_CommandOp )
            {
            case dbg::TMAgent_message_type::TMAGENT_MESSAGE_TYPE_INFO:
                {
                    DispatchDebugMessageTask* pMsgData = (DispatchDebugMessageTask*)Recv;
                    pTask = pThis->StartDebugTask( dbg::TMAgent_message_type::TMAGENT_MESSAGE_TYPE_INFO, pMsgData->m_ProcessId, pMsgData->m_Message, (strlen(pMsgData->m_Message) + 1) );
                }
                break;
            case dbg::TMAgent_message_type::TMAGENT_MESSAGE_TYPE_EXCEPTION:
                {
                    DispatchDebugExceptionTask* pExceptionData = (DispatchDebugExceptionTask*)Recv;
                    pTask = pThis->StartDebugTask( dbg::TMAgent_message_type::TMAGENT_MESSAGE_TYPE_EXCEPTION, pExceptionData->m_ProcessId, &pExceptionData->m_exceptionData, sizeof(pExceptionData->m_exceptionData) );
                    if( pExceptionData->m_exceptionData.m_ExceptionId == tma::DEBUG_EXCEPTION_PROCESS_EXIT )
                    {
                        pThis->m_pProcessMgr->ProcessHasExited( pExceptionData->m_ProcessId );
                    }
                }
                break;
            default:
                break;
            }
            if( pTask != NULL )
            {
                pTask->WaitComplete( TMIPC_INFINITE );
                pThis->m_pServicesManager->FreeTask( pTask );
                pTask = nullptr;
            }
        }
        else
        {
            Continue = false;
        }
        delete Recv;
    }

    return NULL;
}

//==============================================================================

void DebugService::Init( ProcessMgr* pProcessMgr )
{
    m_pProcessMgr = pProcessMgr;
    Create();
}

//==============================================================================

void DebugService::Kill()
{
    m_pProcessMgr = NULL;
    Destroy();
}

//==============================================================================

tmipc::Task* DebugService::StartDebugTask( TMAgent_message_type MessageType, process_id ProcessId, void* pData, s32 SizeOfData )
{
    TMA_TRACE( "DebugService", "StartDebugTask" );
    void* pMem = s_Allocate( sizeof(DebugTask) );
    DebugTask* pTask = new (pMem) DebugTask( m_pProcessMgr );
    pTask->SetServicesManager( m_pServicesManager );
    pTask->SetServiceId( m_ServiceId );
    pTask->SetTaskId( m_pServicesManager->AllocTaskId() );
    pTask->Start( MessageType, ProcessId, pData, SizeOfData );
    return pTask;
}

//==============================================================================

tmipc::Task* DebugService::OnNewTask( tmipc::Packet* pPacket )
{
    tmipc::Task* pRet = NULL;
    //NN_SDK_LOG( "[tma] - DebugService::OnNewTask:  Type %d\n", pPacket->GetTaskType() );
    void* pMem = NULL;

    switch( pPacket->GetTaskType() )
    {
        default:
        case tmipc::TaskType_DebugTask:
            pMem = s_Allocate( sizeof(DebugTask) );
            pRet = new (pMem) DebugTask( m_pProcessMgr );
            break;

        case tmipc::TaskType_MiniCoreDumpTask:
            pMem = s_Allocate( sizeof(MiniCoreDump) );
            pRet = new (pMem) MiniCoreDump( m_pProcessMgr );
            break;
    }

    if( pRet )
    {
        pRet->SetServicesManager( m_pServicesManager );
        pRet->SetServiceId( m_ServiceId );
        pRet->SetTaskId( pPacket->GetTaskId() );
        pRet->SetTaskType( (s16)pPacket->GetTaskType() );
        pRet->OnInitiate( pPacket );
    }

    return( pRet );
}

//==============================================================================

void DebugService::NotifyException( dbg::process_id ProcessId, tma::dbg::Exception* pExceptionData )
{
    TMA_TRACE( "DebugService", "NotifyException" );
    TMA_TRACE( "DebugService", "---------------" );
    void* pMem = s_Allocate( sizeof(DispatchDebugExceptionTask) );
    DispatchDebugExceptionTask* pToSend = new (pMem) DispatchDebugExceptionTask( ProcessId, pExceptionData );
    m_DispatchTaskQ.Send( pToSend );
}

//==============================================================================

void DebugService::NotifyMessage( dbg::process_id ProcessId, const char* pMessage )
{
    void* pMem = s_Allocate( sizeof(DispatchDebugMessageTask) );
    DispatchDebugMessageTask* pToSend = new (pMem) DispatchDebugMessageTask( ProcessId, pMessage );
    m_DispatchTaskQ.Send( pToSend );
}

//==============================================================================
}}
//==============================================================================
