﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/dbg/dbg_Api.h>
#include <nn/dmnt/dmnt_Api.h>
#include <nn/dmnt/dmnt_Result.h>

#include "..\tmagent.h"

#include "dbg_opcodes.h"
#include "dbg_Process.h"
#include "dbg_ProcessMgr.h"
#include "dbg_CoreDump.h"
#include "..\tm_result.h"
#include "../dbghlp/dbghlp.h"
#include <nn/ldr/ldr_Result.h>

//==============================================================================
namespace tma { namespace dbg {
//==============================================================================

//==============================================================================
// Process list functionality.
//==============================================================================

ProcessListNode::ProcessListNode( ProcessMgr* pOwner, const char* pSource, const char* pArgs, nn::svc::Handle processHandle )
{
    void* pMem = s_Allocate( sizeof( Process ));
    m_pProcess = new (pMem) Process( pOwner, pSource, pArgs, processHandle );

    m_pNext = NULL;
}


//==============================================================================

ProcessListNode::ProcessListNode( ProcessMgr* pOwner, process_id PID )
{
    void* pMem = s_Allocate( sizeof( Process ));
    m_pProcess = new (pMem) Process( pOwner, PID );

    m_pNext = NULL;
}

ProcessListNode::ProcessListNode( ProcessMgr* pOwner, process_id PID, nn::svc::Handle processHandle )
{
    void* pMem = s_Allocate( sizeof( Process ));
    m_pProcess = new (pMem) Process( pOwner, PID, processHandle );
    m_pNext = NULL;
}

//==============================================================================

ProcessListNode::~ProcessListNode()
{
    delete m_pProcess;
    m_pNext = NULL;
}

//==============================================================================

Process* ProcessListNode::GetProcess()
{
    return m_pProcess;
}

//==============================================================================

void ProcessListNode::AddToList( struct ProcessListNode* pNew, struct ProcessListNode** pAddToList )
{
    struct ProcessListNode* pList = *pAddToList;
    if( pList == NULL )
    {
        *pAddToList = pNew;
    }
    else
    {
        while(pList->m_pNext != NULL )
        {
            pList = pList->m_pNext;
        }
        pList->m_pNext = pNew;
    }
}

//==============================================================================

void ProcessListNode::RemoveFromList( struct ProcessListNode* pRemove, struct ProcessListNode** pFromList )
{
    struct ProcessListNode* pList = *pFromList;
    if( pList == pRemove )
    {
        *pFromList = pRemove->m_pNext;
    }
    else
    {
        while( pList != NULL && pList->m_pNext != pRemove )
        {
            pList = pList->m_pNext;
        }
        if( pList != NULL )
        {
            pList->m_pNext = pRemove->m_pNext;
        }
        else
        {
            //Couldn't find it in the list.
        }
    }
}

//==============================================================================

void ProcessListNode::ClearList( struct ProcessListNode** pDeleteList )
{
    struct ProcessListNode* pList = *pDeleteList;
    while( pList != NULL )
    {
        ProcessListNode* pNext = pList->m_pNext;
        delete pList;
        pList = pNext;
    }
}

//==============================================================================
//Return the element count.
s32 ProcessListNode::GetCountList( const struct ProcessListNode* pList )
{
    s32 Count = 0;

    for ( const ProcessListNode *pCurrent = pList; pCurrent != NULL; pCurrent = pCurrent->m_pNext )
    {
        Count++;
    }

    return Count;
}

//==============================================================================
//Find by index
struct ProcessListNode* ProcessListNode::FindInList( int AtIndex, struct ProcessListNode* pList )
{
    int Index = 0;
    while( pList != NULL )
    {
        if( Index == AtIndex)
        {
            return pList;
        }
        pList = pList->m_pNext;
        Index += 1;
    }

    return NULL;
}

//==============================================================================
// Find by ID (or pointer)
struct ProcessListNode* ProcessListNode::FindInList( Process* pProcess, struct ProcessListNode* pList )
{
    while( pList != NULL )
    {
        if( pList->m_pProcess == pProcess )
        {
            return pList;
        }
        pList = pList->m_pNext;
    }

    return NULL;
}

//==============================================================================
// Find by PID
struct ProcessListNode* ProcessListNode::FindInList( process_id ProcessId, struct ProcessListNode* pList )
{
    while( pList != NULL )
    {
        if( pList->m_pProcess->GetProcessId() == ProcessId )
        {
            return pList;
        }
        pList = pList->m_pNext;
    }

    return NULL;
}

//==============================================================================
//Find by index
struct ProcessListNode* ProcessListNode::FindByHandleInList( s32 ProcessId, struct ProcessListNode* pList )
{
    while( pList != NULL )
    {
        if( pList->m_pProcess->GetProcessId() == ProcessId )
        {
            return pList;
        }
        pList = pList->m_pNext;
    }

    return NULL;
}

//==============================================================================

void ProcessListNode::RemoveStale( int32_t NumProcesses, nn::Bit64* pActiveProcessIds, struct ProcessListNode** pFromList )
{
    if( NumProcesses == 0 )
    {
        //Easy case
        ClearList( pFromList );
    }

    else if( *pFromList != NULL )
    {
        struct ProcessListNode* pSaveList = NULL;
        struct ProcessListNode* pList = *pFromList;
        while( pList != NULL )
        {
            // Save the next one for handling...
            struct ProcessListNode* pNext = pList->m_pNext;

            //Before cutting this one out of the From list.
            pList->m_pNext = NULL;

            //======================================================
            // Is this one in the active list?
            //======================================================
            int32_t FoundAtIndex = 0;
            for( ; FoundAtIndex < NumProcesses; FoundAtIndex += 1 )
            {
                if( pList->m_pProcess->GetProcessId() == pActiveProcessIds[FoundAtIndex])
                {
                    break;
                }
            }

            //Still there?
            if( FoundAtIndex >= NumProcesses )
            {
                delete pList;
            }
            else
            {
                AddToList( pList, &pSaveList );
            }

            pList = pNext;
        }

        //Save these...
        *pFromList = pSaveList;
    }
}

//==============================================================================

ProcessMgr::ProcessMgr()
{
    m_ProcessList = NULL;
}

//==============================================================================

ProcessMgr::~ProcessMgr()
{
}

//==============================================================================

void ProcessMgr::Init()
{
    TMA_TRACE( "ProcessMgr::Init", "Started" );
    m_ProcessList = NULL;
    m_Mutex.Create();

    DEJA_ENUM_VALUE( process_state, PS_BKPT         );
    DEJA_ENUM_VALUE( process_state, PS_HALT         );
    DEJA_ENUM_VALUE( process_state, PS_RESUME       );
    DEJA_ENUM_VALUE( process_state, PS_UNKNOWN      );
    DEJA_ENUM_VALUE( process_state, PS_NOT_STARTED  );
    DEJA_ENUM_VALUE( process_state, PS_CLONE        );

    DEJA_ENUM_VALUE( bp_state,      BP_NONE         );
    DEJA_ENUM_VALUE( bp_state,      BP_STEP         );
    DEJA_ENUM_VALUE( bp_state,      BP_TEMP         );
    DEJA_ENUM_VALUE( bp_state,      BP_NORMAL       );

    DEJA_ENUM_VALUE( step_state,    SS_NONE         );
    DEJA_ENUM_VALUE( step_state,    SS_BREAK        );
    DEJA_ENUM_VALUE( step_state,    SS_OUT          );
    DEJA_ENUM_VALUE( step_state,    SS_INTO         );
    DEJA_ENUM_VALUE( step_state,    SS_OVER         );
    DEJA_ENUM_VALUE( step_state,    SS_CONT         );

    DEJA_ENUM_VALUE( DebugOpcodes, TMA_OP_NOP );
    DEJA_ENUM_VALUE( DebugOpcodes, TMA_OP_LOAD_ELF );
    DEJA_ENUM_VALUE( DebugOpcodes, TMA_OP_START_DEBUG );
    DEJA_ENUM_VALUE( DebugOpcodes, TMA_OP_HALT_DEBUG );
    DEJA_ENUM_VALUE( DebugOpcodes, TMA_OP_STEP_DEBUG );
    DEJA_ENUM_VALUE( DebugOpcodes, TMA_OP_SET_BKPT );
    DEJA_ENUM_VALUE( DebugOpcodes, TMA_OP_CLR_BKPT );
    DEJA_ENUM_VALUE( DebugOpcodes, TMA_OP_READ_MEM );
    DEJA_ENUM_VALUE( DebugOpcodes, TMA_OP_WRITE_MEM );
    DEJA_ENUM_VALUE( DebugOpcodes, TMA_OP_READ_REG_DEFS );
    DEJA_ENUM_VALUE( DebugOpcodes, TMA_OP_READ_REG_DATA );
    DEJA_ENUM_VALUE( DebugOpcodes, TMA_OP_WRITE_REG );
    DEJA_ENUM_VALUE( DebugOpcodes, TMA_OP_CONT_DEBUG );
    DEJA_ENUM_VALUE( DebugOpcodes, TMA_OP_STOP_DEBUG );
    DEJA_ENUM_VALUE( DebugOpcodes, TMA_OP_KILL_PROCESS );
    DEJA_ENUM_VALUE( DebugOpcodes, TMA_OP_DISCONNECT );
    DEJA_ENUM_VALUE( DebugOpcodes, TMA_OP_NUM_MODULES );
    DEJA_ENUM_VALUE( DebugOpcodes, TMA_OP_MODULE_INFO );
    DEJA_ENUM_VALUE( DebugOpcodes, TMA_OP_NUM_PROCESSES );
    DEJA_ENUM_VALUE( DebugOpcodes, TMA_OP_PROCESS_INFO );
    DEJA_ENUM_VALUE( DebugOpcodes, TMA_OP_NUM_THREADS );
    DEJA_ENUM_VALUE( DebugOpcodes, TMA_OP_THREAD_INFO );
    DEJA_ENUM_VALUE( DebugOpcodes, TMA_OP_MAIN_THREAD );
    TMA_TRACE( "ProcessMgr::Init", "Done initializing" );
}

//==============================================================================

void ProcessMgr::Kill()
{
    //Clear our list.
    {
        ScopedLock Lock( m_Mutex );
        ProcessListNode::ClearList( &m_ProcessList );
        m_ProcessList = NULL;
    }
    m_Mutex.Destroy();
}

//==============================================================================

void ProcessMgr::HandleWriteDump( Command* pCommand, bool WriteMiniDump )
{
    WriteDumpCommandArgs* pArgs = (WriteDumpCommandArgs*)pCommand->m_pData;
    //Find this guy.
    ProcessListNode* pNode = ProcessListNode::FindInList( pCommand->m_ProcessId, m_ProcessList );
    if( pNode != NULL )
    {
        CoreDump::Generate( pNode->m_pProcess, pArgs->m_OutputName, WriteMiniDump );
    }
    else
    {
        NN_SDK_LOG("[tma] HandleWriteDump FAILED:  Unable to find process %lld\n", pCommand->m_ProcessId );
    }
}

//==============================================================================

s32 ProcessMgr::InitiateCoreDump( process_id ProcessId, char* pOutputName, bool QuickDump )
{
    s32 Ret = tmapi::result::RESULT_OK;
    ProcessListNode* pNode = ProcessListNode::FindInList( ProcessId, m_ProcessList );
    if( pNode != NULL )
    {
        CoreDump::Initiate( pNode->m_pProcess, pOutputName, QuickDump );
    }
    else
    {
        NN_SDK_LOG("[tma] InitiateCoreDump FAILED:  Unable to find process %lld\n", ProcessId );
        Ret = tmapi::result::RESULT_PROCESS_NOT_RUNNING;
    }
    return Ret;
}

//==============================================================================

s32 ProcessMgr::ContinueCoreDump( process_id ProcessId, s32* pPercentDone )
{
    s32 Ret = tmapi::result::RESULT_OK;
    ProcessListNode* pNode = ProcessListNode::FindInList( ProcessId, m_ProcessList );
    if( pNode != NULL )
    {
        nn::Result res = CoreDump::Continue( pNode->m_pProcess, pPercentDone );
        if( res.IsFailure() )
        {
            Ret = tmapi::result::RESULT_COREDUMP_FAILED;
        }
    }
    else
    {
        NN_SDK_LOG("[tma] InitiateCoreDump FAILED:  Unable to find process %lld\n", ProcessId );
        Ret = tmapi::result::RESULT_PROCESS_NOT_RUNNING;
    }
    return Ret;
}

//==============================================================================

s32 ProcessMgr::AddTTYToCoreDump( process_id ProcessId, char* pData, u32 SizeOfData )
{
    s32 Ret = tmapi::result::RESULT_OK;
    ProcessListNode* pNode = ProcessListNode::FindInList( ProcessId, m_ProcessList );
    if( pNode != NULL )
    {
        nn::Result res = CoreDump::AddTTY( pNode->m_pProcess, pData, SizeOfData );
        if( res.IsFailure() )
        {
            Ret = tmapi::result::RESULT_COREDUMP_FAILED;
        }
    }
    else
    {
        NN_SDK_LOG("[tma] InitiateCoreDump FAILED:  Unable to find process %lld\n", ProcessId );
        Ret = tmapi::result::RESULT_PROCESS_NOT_RUNNING;
    }
    return Ret;
}

//==============================================================================

void ProcessMgr::CloseCoreDump( process_id ProcessId )
{
    ProcessListNode* pNode = ProcessListNode::FindInList( ProcessId, m_ProcessList );
    if( pNode != NULL )
    {
        CoreDump::Close( pNode->m_pProcess );
    }
    else
    {
        NN_SDK_LOG("[tma] CloseCoreDump FAILED:  Unable to find process %lld\n", ProcessId );
    }
}

//==============================================================================
// Launch-to-TMAPI error table.  Used to address SigloNTD-10466
// "make 'Unable to load executable' messages more helpful".
//==============================================================================

struct nn_result_to_tmapi_result
{
    nn::Result      m_NNRes;
    tmapi::result   m_TMAPIRes;
    bool Is( nn::Result res )
    {
        return ( m_NNRes.GetInnerValueForDebug() == res.GetInnerValueForDebug() );
    }
};

static nn_result_to_tmapi_result sAllLaunchResults[] =
{
    // Here's the known load problem results we've seen in our testing
    { nn::dmnt::ResultDebuggingDisabledForProcess(), tmapi::RESULT_PROCESS_DEBUGGING_DISABLED },
    { nn::fs::ResultPathNotFound(), tmapi::RESULT_PROCESS_PACKAGE_INCOMPLETE },
    { nn::ldr::ResultInvalidNso(), tmapi::RESULT_PROCESS_PACKAGE_INCOMPLETE }
};

#define LAUNCH_RESULT_TABLE_COUNT TMA_MACRO_VALUE(sizeof( sAllLaunchResults ) / sizeof(nn_result_to_tmapi_result) )

//==============================================================================

static int32_t nnLaunchResultToTMAPI( nn::Result result )
{
    tmapi::result Ret = tmapi::result::RESULT_OK;
    if( result.IsFailure() )
    {
//        NN_SDK_LOG("[tmagent] nnLaunchResultToTMAPI translating result 0x%x\n", result.GetInnerValueForDebug() );

        //=========================================================
        // Look through our table for this result.
        //=========================================================
        for (int Index = 0; Index < LAUNCH_RESULT_TABLE_COUNT; Index += 1 )
        {
            if( sAllLaunchResults[Index].Is( result ) )
            {
                Ret = sAllLaunchResults[Index].m_TMAPIRes;
                break;
            }
        }
        //=========================================================
        // If we made it through our table but couldn't find this
        // particular result, pass back our generic result.
        //=========================================================
        if( Ret == tmapi::result::RESULT_OK )
        {
            Ret = tmapi::RESULT_PROCESS_CREATE_FAILED;
        }
    }

    return Ret;
}

//==============================================================================

s32 ProcessMgr::HandleLoadElf( const char* pFileName, const char* pArgs, process_id& ProcessId )
{
    TMA_TRACE( "ProcessMgr::LoadElf", "LoadElf( %s, %i )\n", pFileName, strlen( pFileName ) );
    //NN_SDK_LOG("LoadElf:  %s\n", pFileName );
    // Load the process.
    nn::svc::Handle hProcess;
    nn::Result result = dbghlp::LoadImage( &hProcess, pFileName, pArgs );
    if( result.IsFailure() )
    {
        // Fix for "Make load/start debugging errors more helpful"
        return( nnLaunchResultToTMAPI( result ) );
    }

    //NN_SDK_LOG( "Loaded %s -> PID = %p\n", pFileName, hProcess.GetPrintableBits() );
    tmapi::result Ret = tmapi::result::RESULT_OK;

    // Add the process to the list.
    void* pMem = s_Allocate( sizeof( ProcessListNode ));
    ProcessListNode* pAddNode = new (pMem) ProcessListNode( this, pFileName, pArgs, hProcess );
    {
        ScopedLock Lock( m_Mutex );
        ProcessListNode::AddToList( pAddNode, &m_ProcessList );
        ProcessId = pAddNode->GetProcess()->GetProcessId();
        result = pAddNode->GetProcess()->WaitForCreateProcess();
        if( result.IsFailure() )
        {
            Ret = tmapi::result::RESULT_EXECUTABLE_NOT_COMPATIBLE;
        }
    }
    //NN_SDK_LOG("LoadElf:  Process created.\n" );

    return( Ret );
}

//==============================================================================
// EXE-loading functionality that returns an nn::Result.  This helps Siglo-56601,
// "make unable to start debugging errors more helpful".
nn::Result ProcessMgr::LoadExe( const char*  pFileName, const char* pArgs, process_id& ProcessId )
{
    TMA_TRACE( "ProcessMgr::LoadElf", "LoadElf( %s, %i )\n", pFileName, strlen( pFileName ) );
    //NN_SDK_LOG("LoadElf:  %s\n", pFileName );
    // Load the process.
    nn::svc::Handle hProcess;
    nn::Result result = dbghlp::LoadImage( &hProcess, pFileName, pArgs );
    if( result.IsSuccess() )
    {
        // Add the process to the list.
        void* pMem = s_Allocate( sizeof( ProcessListNode ));
        ProcessListNode* pAddNode = new (pMem) ProcessListNode( this, pFileName, pArgs, hProcess );
        {
            ScopedLock Lock( m_Mutex );
            ProcessListNode::AddToList( pAddNode, &m_ProcessList );
            ProcessId = pAddNode->GetProcess()->GetProcessId();
            result = pAddNode->GetProcess()->WaitForCreateProcess();
        }
    //NN_SDK_LOG("LoadElf:  Process created.\n" );
    }

    return( result );
}

//==============================================================================

s32 ProcessMgr::HandleAttachOnLaunch( u64 ProgramId, process_id& ProcessId )
{
    TMA_TRACE( "ProcessMgr::HandleAttachOnLaunch", "HandleAttachOnLaunch for program %llx\n", ProgramId );
//NN_SDK_LOG("[tma] HandleAttachOnLaunch for program %llx\n", ProgramId );

    nn::svc::Handle hProcess;
    nn::Result result = dbghlp::AttachOnLaunch( &hProcess, &ProcessId, ProgramId );

    if( result.IsFailure() )
    {
//NN_SDK_LOG( "[tmagent] ProcessMgr::HandleAttachOnLaunch FAILED for program %llx --> %08x\n", ProgramId, result.GetInnerValueForDebug() );
        return( tmapi::RESULT_PROCESS_CREATE_FAILED );
    }

//NN_SDK_LOG( "[tmagent] ProcessMgr::HandleAttachOnLaunch %llx has launched-> PID = %lld\n", ProgramId, ProcessId );
    // Add the process to the list.
    tmapi::result Ret = tmapi::result::RESULT_OK;

    void* pMem = s_Allocate( sizeof( ProcessListNode ));
    ProcessListNode* pAddNode = new (pMem) ProcessListNode( this, ProcessId, hProcess );
    {
        ScopedLock Lock( m_Mutex );
        ProcessListNode::AddToList( pAddNode, &m_ProcessList );
        result = pAddNode->GetProcess()->WaitForCreateProcess();
        if( result.IsFailure() )
        {
            Ret = tmapi::result::RESULT_EXECUTABLE_NOT_COMPATIBLE;
        }

        //=====================================================================================
        // When we get this process this way, it's not officially started just yet.  However,
        // when we create a process with a handle, it's assumed to be already running (see
        // dbg_Process.cpp, line 152).  Reset it's state so we can start it properly.
        //=====================================================================================
        pAddNode->GetProcess()->SetProcessState( tma::process_state::PS_NOT_STARTED );
    }

//NN_SDK_LOG("[tmagent] ProcessMgr::HandleAttachOnLaunch:  Process created.\n" );

    return( Ret );
}

//==============================================================================

s32 ProcessMgr::HandleAttachByProgramId( u64 ProgramId, process_id& ProcessId )
{
    TMA_TRACE( "ProcessMgr::HandleAttachByProgramId", "HandleAttachByProgramId for program %llx\n", ProgramId );
    nn::svc::Handle hProcess;
    nn::Result result = dbghlp::AttachByProgramId( &hProcess, &ProcessId, ProgramId );
    if( result.IsFailure() )
    {
        NN_SDK_LOG( "[tma] HandleAttachByProgramId:  dbghlp::AttachByProgramId FAILED with error 0x%x\n", result.GetInnerValueForDebug() );
        return( tmapi::RESULT_PROCESS_ATTACH_FAILED );
    }

    //NN_SDK_LOG( "[tma] HandleAttachByProgramId:  Attached %llx -> PID = %lld\n", ProgramId, ProcessId );
    struct ProcessListNode* pProcessNode = ProcessListNode::FindInList( ProcessId, m_ProcessList );
    if( pProcessNode == NULL )
    {
        void* pMem = s_Allocate( sizeof( ProcessListNode ));
        pProcessNode = new (pMem) ProcessListNode( this, ProcessId, hProcess );
        {
            ScopedLock Lock( m_Mutex );
            ProcessListNode::AddToList( pProcessNode, &m_ProcessList );
        }
    }

    tmapi::result Ret = tmapi::result::RESULT_OK;
    bool AttachResult = pProcessNode->GetProcess()->Attach( hProcess );
    if( AttachResult != true )
    {
        NN_SDK_LOG( "[tma] HandleAttachByProgramId:  pProcessNode->GetProcess()->Attach FAILED\n" );
        Ret = tmapi::RESULT_PROCESS_ATTACH_FAILED;
    }

    return( Ret );
}

//==============================================================================

void ProcessMgr::Kill( Process* pTargetProcess )
{
    if( pTargetProcess != NULL )
    {
        //=====================================================================
        // If we don't have a handle to it, we'll need to attach before
        // we can kill it.
        //=====================================================================

        if( pTargetProcess->GetHandle() == nn::svc::INVALID_HANDLE_VALUE )
        {
            pTargetProcess->Attach();
        }

        pTargetProcess->KillProcess();
    }
}

//==============================================================================

void ProcessMgr::HandleKillProcess( Command* pCommand )
{
    Process* pTargetProcess = FindTargetProcess( pCommand );
    Kill( pTargetProcess );
}

//==============================================================================

s32 ProcessMgr::HandleDetachProcess(Command* pCommand)
{
    s32 Ret = tmapi::RESULT_INVALID_PROCESS;
    Process* pTargetProcess = FindTargetProcess( pCommand );
    if (pTargetProcess != NULL)
    {
        Ret = pTargetProcess->Detach();
    }
    else
    {
        NN_SDK_LOG("[tma] ProcessMgr::HandleDetachProcess FAILED to find process %lld\n", pCommand->m_ProcessId);
    }
    return Ret;
}

//==============================================================================

void ProcessMgr::OnProcessKilled( Process* pKilled )
{
    //NN_SDK_LOG("ProcessMgr::OnProcessKilled:  %lld\n", pKilled->GetProcessId() );
    ScopedLock Lock( m_Mutex );
    ProcessListNode* pNode = ProcessListNode::FindInList( pKilled, m_ProcessList );
    if( pNode != NULL )
    {
        ProcessListNode::RemoveFromList( pNode, &m_ProcessList );
        delete pNode;
    }
}

//==============================================================================

void ProcessMgr::ProcessHasExited( process_id ProcessId )
{
//NN_SDK_LOG( "ProcessMgr::ProcessHasExited - process %lld", ProcessId );

    ScopedLock Lock( m_Mutex );
    struct ProcessListNode* ProcessNode = ProcessListNode::FindInList( ProcessId, m_ProcessList );
    if( ProcessNode != NULL )
    {
        ProcessNode->m_pProcess->KillProcess();
    }
}

//==============================================================================

s32 ProcessMgr::HandleGetNumberOfProcesses()
{
    UpdateProcessList();
    s32 nProcesses = ProcessListNode::GetCountList( m_ProcessList );

    return( nProcesses );
}

//==============================================================================

void ProcessMgr::OnSleep()
{
    //============================================================================
    // For sleep, detach all our processes - Any debug sessions will be killed
    // on the host side, we're just syncing with that by releasing debug handles.
    //============================================================================
    while (ProcessListNode::GetCountList(m_ProcessList))
    {
        m_ProcessList->GetProcess()->Detach();
    }
}

//==============================================================================

enum
{
    NUMBER_OF_JIT_PROCESS_IDS   = 16,
};

//==============================================================================

bool ProcessMgr::GetJITProcessList( int32_t* pNumProcesses, nn::os::ProcessId pProcessIds[], int32_t arraySize )
{
    nn::Result res = dbghlp::GetJitDebugProcessList( pNumProcesses, pProcessIds, arraySize );

    if( res.IsSuccess() )
    {
        return true;
    }

    NN_SDK_LOG("GetJitDebugProcessList returned failure -> %lld\n", res.GetDescription() );
    return false;
}

//==============================================================================

s32 ProcessMgr::HandleGetNumberOfJITProcesses()
{
    int32_t NumJITProcesses = 0;
    nn::os::ProcessId ProcessIds[NUMBER_OF_JIT_PROCESS_IDS];

    if( GetJITProcessList( &NumJITProcesses, ProcessIds, NUMBER_OF_JIT_PROCESS_IDS ) == true )
    {
        return NumJITProcesses;
    }
    return 0;
}

//==============================================================================

s32 ProcessMgr::HandleGetJITProcessInfo ( s32 iIndex, ProcessDefinition& ProcessDefinition )
{
    if( iIndex >= 0 )
    {
        int32_t NumJITProcesses = 0;
        nn::os::ProcessId ProcessIds[NUMBER_OF_JIT_PROCESS_IDS];

        if( GetJITProcessList( &NumJITProcesses, ProcessIds, NUMBER_OF_JIT_PROCESS_IDS ) == true )
        {
            if( iIndex < NumJITProcesses )
            {
                //See if we've already got this one in our list:
                ScopedLock Lock( m_Mutex );
                struct ProcessListNode* ProcessNode = ProcessListNode::FindInList( ProcessIds[iIndex].value, m_ProcessList );

                //===================================================================================
                // As per SigloNTD-10212 (http://spdlybra.nintendo.co.jp/jira/browse/SIGLONTD-10212),
                // if we do this, we'll cause an additional JIT notification.  Not really vital
                // that we get a handle now anyway.
                //===================================================================================
#if 0
                if( ProcessNode == NULL )
                {
                    //====================================================================================
                    // If we can attach to it, add it to our list - we'll probably want to attach to it.
                    //====================================================================================
                    nn::svc::Handle debugHandle;
                    nn::Result result = dbghlp::GetProcessHandle( &debugHandle, ProcessIds[iIndex].value );
                    if (result.IsSuccess())
                    {
                        ProcessNode = new ProcessListNode( this, ProcessIds[iIndex].value );
                        ProcessListNode::AddToList( ProcessNode, &m_ProcessList );
                        dbghlp::CloseHandle( debugHandle );
                    }
                    else
                    {
                        NN_SDK_LOG("[tma] ERROR getting debug handle to JIT process %lld\n", ProcessIds[iIndex].value );
                        return tmapi::RESULT_INTERNAL_ERROR;
                    }
                }
#endif
                if( ProcessNode != NULL )
                {
                    int32_t ExceptionId = ProcessNode->m_pProcess->GetJITException();
                    ProcessNode->m_pProcess->GetDefinition( ProcessDefinition );

                    //Insure we pass back the exception
                    ProcessDefinition.m_LastExceptionId = ExceptionId;

                    // By definition, our JIT processes are not running.
                    ProcessDefinition.m_State = 1;
                    return tmapi::RESULT_OK;
                }
                else
                {
                    ProcessDefinition.m_PID = ProcessIds[iIndex].value;
                    ProcessDefinition.m_LastExceptionId = -1;

                    // By definition, our JIT processes are not running.
                    ProcessDefinition.m_State = 1;
                    return tmapi::RESULT_OK;
                }
            }
        }
    }

    return tmapi::RESULT_PROCESS_INVALID_ARGUMENTS;
}

//===================================================================================
// As per SigloNTD-8052 (http://spdlybra.nintendo.co.jp/jira/browse/SIGLONTD-8052),
// tmagent no longer actively searches for processes.  However, we'll leave
// this code here for now for future reference, to be removed once we are
// confident that all the pieces work as they should.
//===================================================================================
#if 0
static nn::Bit64 pProcessIds[1024];
#endif

void ProcessMgr::UpdateProcessList()
{
//===================================================================================
// As per SigloNTD-8052 (http://spdlybra.nintendo.co.jp/jira/browse/SIGLONTD-8052),
// tmagent no longer actively searches for processes.  However, we'll leave
// this code here for now for future reference, to be removed once we are
// confident that all the pieces work as they should.
//===================================================================================
#if 0
    nn::Bit64 tmaProcessId;
    NN_ABORT_UNLESS_RESULT_SUCCESS( nn::dbg::GetCurrentProcessId(&tmaProcessId) );

    nn::Bit64 dmntProcessId;
    NN_ABORT_UNLESS_RESULT_SUCCESS( nn::dmnt::GetDebugMonitorProcessId(&dmntProcessId) );

    nn::Result res = nn::ResultSuccess();
    int32_t NumProcesses = 0;
    {
        //======================================================================
        // Lock down our list while we get the process Id list from the OS.
        ScopedLock Lock( m_Mutex );
        res = dbghlp::GetProcessList( &NumProcesses, pProcessIds, sizeof(pProcessIds) / sizeof(nn::Bit64) );
        if( res.IsSuccess() )
        {
            // ...and remove any of the stale processes.
            ProcessListNode::RemoveStale( NumProcesses, pProcessIds, &m_ProcessList );
        }
    }
    if( res.IsSuccess() )
    {
NN_SDK_LOG( "[tma] ProcessMgr::UpdateProcessList:  Found %d processes running.\n", NumProcesses );

        //=======================================================================
        // Now let's add the new ones.
        //=======================================================================
        for( int ProcessIndex = 0; ProcessIndex < NumProcesses; ProcessIndex += 1 )
        {
            nn::Bit64 ProcessId = pProcessIds[ProcessIndex];
//NN_SDK_LOG( "Found process %lld\n", pProcessIds[ProcessIndex] );

            if( ProcessId == tmaProcessId || ProcessId == dmntProcessId )
            {
NN_SDK_LOG( "[tma] ProcessMgr::UpdateProcessList:  Can't debug process %lld (it's either DMNT or TMA).\n", ProcessId );
                continue;
            }

//NN_SDK_LOG( "Found process %lld\n", ProcessId );
            struct ProcessListNode* ProcessNode = ProcessListNode::FindInList( ProcessId, m_ProcessList );
            if( ProcessNode == NULL )
            {
                nn::svc::Handle debugHandle;
                nn::Result result = dbghlp::GetProcessHandle(&debugHandle, ProcessId );
                if (result.IsSuccess())
                {
                    NN_SDK_LOG("[tma] ProcessMgr::UpdateProcessList:  Found unrecognized debuggable process %lld\n", ProcessId );
                    ProcessListNode* pAddNode = new ProcessListNode( this, ProcessId );
                    {
                        ScopedLock Lock( m_Mutex );
                        ProcessListNode::AddToList( pAddNode, &m_ProcessList );
                    }
                    dbghlp::CloseHandle( debugHandle );
                }
                else
                {
                    NN_SDK_LOG("[tma] ProcessMgr::UpdateProcessList:  Can't debug %lld -> %lld\n", ProcessId, result.GetInnerValueForDebug() ); //GetDescription() );
                }
            }
        }
    }
    else
    {
        NN_SDK_LOG("GetProcessList returned failure -> %lld\n", res.GetDescription() );
    }

#endif
}

//==============================================================================

s32 ProcessMgr::HandleProcessInfo( s32 iIndex, ProcessDefinition& ProcessDefinition )
{
    s32 Result = tmapi::RESULT_OK;
    ScopedLock Lock( m_Mutex );
    const ProcessListNode* ProcessNode = ProcessListNode::FindInList( iIndex, m_ProcessList );
    if( ProcessNode != NULL )
    {
        ProcessNode->m_pProcess->GetDefinition( ProcessDefinition );
    }
    else
    {
        Result = tmapi::RESULT_PROCESS_INVALID_ARGUMENTS;
    }

    return Result;
}

//==============================================================================

Process* ProcessMgr::FindTargetProcess( Command* pCommand )
{
    process_id ProcessId = pCommand->m_ProcessId;
    ScopedLock Lock( m_Mutex );
    struct ProcessListNode* pNode = ProcessListNode::FindInList( ProcessId, m_ProcessList );

    if(pNode != NULL )
    {
        return pNode->m_pProcess;
    }

    return NULL;
}

//==============================================================================

Process* ProcessMgr::FindTargetProcess( process_id ProcessId )
{
    ScopedLock Lock( m_Mutex );
    struct ProcessListNode* pNode = ProcessListNode::FindInList( 0, m_ProcessList );

    if(pNode != NULL )
    {
        return pNode->m_pProcess;
    }

    return NULL;
}

//==============================================================================

Process* ProcessMgr::CreateTargetProcess( process_id ProcessId )
{
    //Make sure we haven't already created him.
    Process* pRet = FindTargetProcess( ProcessId );
    if(pRet == NULL )
    {
        // See if we can attach to this guy.
        nn::svc::Handle processHandle;
        nn::Result result = dbghlp::GetProcessHandle( &processHandle, ProcessId );
        if (result.IsSuccess())
        {
            //====================================================================================
            // We've got a live one, so create him and save him for posterity.
            //====================================================================================

            NN_SDK_LOG("[tma] Creating process from unrecognized debuggable process %lld\n", ProcessId );
            void* pMem = s_Allocate( sizeof( ProcessListNode ));
            ProcessListNode* pAddNode = new (pMem) ProcessListNode( this, ProcessId, processHandle );
            {
                ScopedLock Lock( m_Mutex );
                ProcessListNode::AddToList( pAddNode, &m_ProcessList );
            }

            pRet = pAddNode->GetProcess();

            // We should go ahead and attach.
            pRet->Attach( processHandle );
        }
    }

    return pRet;
}

//==============================================================================

TMAgent_message_type ProcessMgr::HandleCommand( Command* pCommand, s32& ResultCode, void* pReturnBuffer )
{
    s32 Command         = pCommand->m_Command;
    u64 ParameterValue  = pCommand->m_Parameter;

    TMA_TRACE("ProcessMgr::HandleCommand", "Command( %d, %llx, %llx, %d, %p )", pCommand->m_Command, pCommand->m_ProcessId, pCommand->m_Parameter, pCommand->m_DataSize, pCommand->m_pData );

    TMAgent_message_type MessageType = TMAGENT_MESSAGE_TYPE_UNKNOWN;
    switch( Command )
    {
        case TMA_OP_NOP:
            ResultCode = tmapi::RESULT_OK;
            MessageType = TMAGENT_MESSAGE_TYPE_RESULT;
            break;

        case TMA_OP_WRITE_DUMP:
            HandleWriteDump( pCommand );
            ResultCode  = tmapi::RESULT_OK;
            MessageType = TMAGENT_MESSAGE_TYPE_RESULT;
            break;

        case TMA_OP_WRITE_MINI_DUMP:
            HandleWriteDump( pCommand, true );
            ResultCode  = tmapi::RESULT_OK;
            MessageType = TMAGENT_MESSAGE_TYPE_RESULT;
            break;

        case TMA_OP_DISCONNECT:
        case TMA_OP_STOP_DEBUG:
        case TMA_OP_KILL_PROCESS:
            ResultCode = tmapi::RESULT_OK;
            HandleKillProcess( pCommand );
            MessageType = TMAGENT_MESSAGE_TYPE_RESULT;
            break;

        case TMA_OP_NUM_PROCESSES:
            *(s32*)pReturnBuffer = HandleGetNumberOfProcesses();
            MessageType = TMAGENT_MESSAGE_TYPE_NUMERIC_S32;
            ResultCode = tmapi::RESULT_OK;
            break;

        case TMA_OP_PROCESS_INFO:
        {
            const s32 ProcessIndex = static_cast<s32>(ParameterValue);
            ProcessDefinition &LocalProcessDefinition = *reinterpret_cast<ProcessDefinition *>( pReturnBuffer );
            ResultCode = HandleProcessInfo( ProcessIndex, LocalProcessDefinition );
            MessageType = TMAGENT_MESSAGE_TYPE_PROCESS;
            break;
        }

        case TMA_OP_KILL_CURRENT_APPLICATION_ID:
        {
            //===================================================================================
            // As per Siglo-75688 (http://spdlybra.nintendo.co.jp/jira/browse/SIGLO-75688),
            // tmagent no longer kills the current application.  Backwards-compatibility
            // is ignored in this case, as per NTD.
            //===================================================================================
            *(u64*)pReturnBuffer = 0;
            MessageType = TMAGENT_MESSAGE_TYPE_NUMERIC_U64;
            ResultCode = tmapi::RESULT_OK;
            break;
        }

        case TMA_OP_NUM_JIT_PROCESSES:
        {
            *(s32*)pReturnBuffer = HandleGetNumberOfJITProcesses();
            MessageType = TMAGENT_MESSAGE_TYPE_NUMERIC_S32;
            ResultCode = tmapi::RESULT_OK;
            break;
        }

        case TMA_OP_JIT_PROCESS_INFO:
        {
            const s32 ProcessIndex = static_cast<s32>(ParameterValue);
            ProcessDefinition &LocalProcessDefinition = *reinterpret_cast<ProcessDefinition *>( pReturnBuffer );
            ResultCode = HandleGetJITProcessInfo ( ProcessIndex, LocalProcessDefinition );
            MessageType = TMAGENT_MESSAGE_TYPE_PROCESS;
            break;
        }

        case TMA_OP_DETACH:
        {
            ResultCode = HandleDetachProcess( pCommand );
            MessageType = TMAGENT_MESSAGE_TYPE_RESULT;
            break;
        }

        default:
        {
            if( pCommand->m_ProcessId == TMA_INVALID_PROCESS_ID )
            {
                //===============================================================
                // The query was for the unknown process, so if we made it here,
                // we don't understand what this command is.
                //===============================================================
                TMA_TRACE("ProcessMgr::HandleCommand", "HandleCommand:  Rejecting unknown command %d", Command );
                ResultCode = tmapi::RESULT_NOT_IMPLEMENTED;
                MessageType = TMAGENT_MESSAGE_TYPE_RESULT;
            }
            else
            {
                Process* pTargetProcess = FindTargetProcess( pCommand );
                if( pTargetProcess == NULL)
                {
                    //===============================================================
                    // Not the invalid process, so this must be a valid ID, but
                    // we don't know about this process.  Can we create it?
                    //===============================================================
                    pTargetProcess = CreateTargetProcess( pCommand->m_ProcessId );
                }

                if( pTargetProcess != NULL)
                {
                    //===================================================================
                    // Is this a process we didn't start?
                    //===================================================================
                    if( pTargetProcess->GetHandle() == nn::svc::INVALID_HANDLE_VALUE )
                    {
                        pTargetProcess->Attach();
                    }

                    MessageType = pTargetProcess->HandleCommand( pCommand, ResultCode, pReturnBuffer );
                    if( ResultCode != tmapi::RESULT_OK )
                    {
                        if( ResultCode == tmapi::RESULT_PROCESS_NOT_RUNNING )
                        {
                            TMA_TRACE("ProcessMgr::HandleCommand", "HandleCommand:  Killing dead process" );
                            pTargetProcess->KillProcess();
                        }
                    }
                }
                else
                {

                    TMA_TRACE("ProcessMgr::HandleCommand", "HandleCommand:  Rejecting message to dead process" );
                    ResultCode = tmapi::RESULT_PROCESS_NOT_RUNNING;
                    MessageType = TMAGENT_MESSAGE_TYPE_RESULT;
                }
            }
            break;
        }
    }

    return MessageType;
} // NOLINT(impl/function_size)

//==============================================================================

void  ProcessMgr::NotifyUser(process_id ProcessId, const char* pMessage )
{
    //TMA_TRACE( "ProcessMgr::NotifyUser", "Sending message %s", pMessage );
    DebugService& Svc = GetDebugService();
    Svc.NotifyMessage( ProcessId, pMessage );
}

//==============================================================================
}} // namespace
//==============================================================================
