﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include "..\tmagent.h"
#include "dbg_CommandArgs.h"
#include "dbg_BreakpointMgr.h"
#include "dbg_StepMgr.h"
#include "dbg_Modules.h"
#include "..\tma\tma_Thread.h"
#include "..\tma\tma_MQ.h"
#include "dbg_Threads.h"
#include "dbg_Command.h"
#include "dbg_Message.h"

#include <nn/os.h>

//==============================================================================
namespace tma { namespace dbg {
//==============================================================================

enum enum_STEPKIND
{
    STEP_CONT        = -1, // Added to MS definition
    STEP_INTO        = 0,
    STEP_OVER        = 1,
    STEP_OUT         = 2,
    STEP_BACKWARDS   = 3,
};

enum enum_STEPUNIT
{
    STEP_STATEMENT   = 0,
    STEP_LINE        = 1,
    STEP_INSTRUCTION = 2,
};

typedef u64 process_id;

class ProcessMgr;

enum
{
    TMA_NO_THREAD_ID            = 0xFFFFFFFF,
    TMA_INVALID_PROCESS_ID      = 0xFFFFFFFFFFFFFFFF,
    TMA_MAX_COMMAND_LINE_ARGS   = 256,
    PROCESS_NAME_MAX_LENGTH     = 261 + 1,  //Need enough space for the terminating 0
    NO_EXCEPTION_ID             = 0xFF,
};

//==============================================================================
//Currently only used for querying data.  This structure must match:
// ProcessDefinition         (dbg_Process.h - main (TargetRef))
// ProcessDefinition         (dbg_Process.h - main (HORIZON))
// ProcessData               (target_data.h - main)
// res_get_process_info      (tm_protocol.h - main)
// process_info              (TMAPI.h - main)
// tmapi::process_info       (tmapi.h - prototype)
// tmapi_mcpp::process_info  (tmapi_mcpp.h - prototype)
// res_get_process_info      (tm_protocol.h - prototype)
struct ProcessDefinition
{
    ProcessDefinition()
    {
        m_PID               = 0;
        m_NumberOfModules   = 0;
        m_NumThreads        = 0;
        m_LastExceptionId   = 0;
        m_64BitFlags        = 0;
        m_State             = 0;
        memset( m_Name, 0, sizeof(m_Name));
    }

    uint64_t        m_PID;                              // The Process's ID.
    int32_t         m_NumberOfModules;                  // Number of Modules loaded.
    uint32_t        m_NumThreads;                       // Number of threads associated with the process.
    uint32_t        m_LastExceptionId;                  // Last exception this process has received
    u8              m_64BitFlags;                       // Bit 0 = 64 bit process, bit 1 = 64 bit addresses
    u8              m_State;                            // State of the process.  0 = running, 1 = stopped
    char            m_Name[PROCESS_NAME_MAX_LENGTH];    // File name (including path).
};

//================================================================================================

class Process
{
protected:
        dbg::BreakpointMgr      m_BreakManager;
        dbg::StepMgr            m_StepManager;
        dbg::Modules            m_Modules;

        process_state           m_ProcessState;

        process_id              m_PID;

        nn::svc::Handle         m_Handle;

        bool                    m_HasAttached;
        bool                    m_HasLoaded;
        bool                    m_ModulesChanged;
        bool                    m_NeedUpdate;
        uint32_t                m_CurrentThreadId = TMA_NO_THREAD_ID;
        uint32_t                m_LastExceptionId;
        uint64_t                m_LastExceptionAddress;
        ThreadDefinitionCollection m_Threads;
        s32                     m_NumActiveThreads;
        tma::Thread             m_DebugEventThread;
        bool                    m_DebugEventThreadKill;
        bool                    m_DebugEventThreadStarted;
        bool                    m_64Bit;
        bool                    m_64BitAddressSpace;
        ProcessMgr*             m_pOwner;
        char                    m_Name[PROCESS_NAME_MAX_LENGTH];
        char                    m_Args[1024];

        u8*                     m_pRegisterDefinitions;
        u32                     m_SizeRegisterDefs;
        nn::svc::DebugInfoCreateProcess m_DebugInfoCreateProcess;
        u64                     m_CoreDumpSessionId;

#define NUM_BREADCRUMBS TMA_MACRO_VALUE(0) // Must be a multiple of 8, disable with 0

#if NUM_BREADCRUMBS

        s32                     m_iBreadcrumb;
        u64                     m_Breadcrumbs[ NUM_BREADCRUMBS ];

#endif

public:
        Process( ProcessMgr* pOwner, const char* pSource, const char* pArgs, nn::svc::Handle processHandle );
        Process( ProcessMgr* pOwner, process_id PID, nn::svc::Handle processHandle = nn::svc::INVALID_HANDLE_VALUE );
        ~Process();

        void                    Init                    ();
        nn::Result              WaitForCreateProcess    ();
        tmapi::result           WaitForAttach           ();
        bool                    WaitForTermination      ();
        void                    Kill                    ();
        bool                    Attach                  ( nn::svc::Handle processHandle = nn::svc::INVALID_HANDLE_VALUE );
        s32                     Detach                  ();
        s32                     GetJITException         ();
        TMAgent_message_type    HandleCommand           ( Command* pCommandPacket, s32& ResultCode, void* pReplyBuffer );
        s32                     StartDebugProcess       ( u64 StopAtAddress );
        s32                     HaltDebugProcess        ();
        s32                     Step                    ( Command* pCommandPacket );
        s32                     SetBreakpoint           ( break_point& BP );
        s32                     ReadMemory              ( u64 Address, void* pBuffer, u32 nBytes );
        s32                     WriteMemory             ( u64 Address, void* pBuffer, u32 nBytes );
        s32                     WriteMemory             ( Command* pCommandPacket );
        s32                     ReadRegisterDefinitions ( u32& BufferSize, void* pReplyBuffer );
        s32                     ReadRegisterData        ( Command* pCommandPacket, void* pReplyBuffer );
        s32                     WriteRegister           ( s32 ThreadId, s32 RegisterId, u64 Value );
        s32                     WriteRegister           ( Command* pCommandPacket );
        s32                     Continue                ( s32 ThreadId );
        s32                     StopDebugProcess        ();
        s32                     KillProcess             ();
        s32                     GetNumberOfModules      ( s32& NumberOfModules );
        s32                     GetModuleInfo           ( s32 Index, void* Buffer );
        s32                     GetNumberOfThreads      ();
        s32                     GetThreadDetails        ( s32 AtIndex, ThreadDefinition** ppThreadDetails );
        process_id              GetProcessId            ();
        u64                     GetProgramId            ();
        void                    GetName                 ( char* pBuffer, s32 BufferSize );
        void                    GetArgs                 ( char* pBuffer, s32 BufferSize );
        uint32_t                GetLastExceptionId      ();
        uint64_t                GetLastExceptionAddress ();
        void                    PurgeEventsAndContinue  ();
        process_state           GetProcessState         ();
        void                    SetProcessState         ( process_state State );
        void                    GetDefinition           ( ProcessDefinition& ProcessDefinition );
        bool                    Is64Bit                 ();
        bool                    Is64BitAddressSpace     ();
        dbg::BreakpointMgr*     GetBreakpointManager    ();
        dbg::Modules*           GetModules              ();
        dbg::StepMgr*           GetStepManager          ();
        nn::svc::Handle         GetHandle               ();
        void                    NotifyUser              ( const char* Message );
        void                    NotifyHalt              ( TMAgent_message_type HaltType, uint32_t HaltId, s32 ThreadId, u64 ExceptionAddress );
        void                    SetCurrentThreadId      ( uint32_t ThreadId );
        uint32_t                GetCurrentThreadId      ();
        nn::Result              ContinueDebugProcess    ( u64 ThreadId );
        void                    OnProcessCreated        ( nn::svc::DebugEventInfo* pEventInfo );
        void                    OnThreadCreated         ( nn::svc::DebugEventInfo* pEventInfo, bool Continue = true );
        bool                    OnThreadExited          ( u64 ThreadId, bool Continue = true );
        void                    OnProcessExited         ( u64 ThreadId );
        void                    OnModuleLoadStart       ( nn::svc::DebugEventInfo* pEventInfo );
        void                    OnModuleLoadComplete    ( nn::svc::DebugEventInfo* pEventInfo );
        void                    OnModuleUnloadStart     ( nn::svc::DebugEventInfo* pEventInfo );
        void                    OnModuleUnloadComplete  ( nn::svc::DebugEventInfo* pEventInfo );
        void                    HandleException         ( nn::svc::DebugEventInfo& ExceptionInfo );

        bool                    GetPerThreadStepInfo    ( u64 ThreadId, per_thread_step_info& PTSI );
        bool                    SetPerThreadStepInfo    ( u64 ThreadId, per_thread_step_info& PTSI );

        void                    SetCoredumpSessionId    ( u64 SessionId );
        u64                     GetCoredumpSessionId    ();

protected:
        s32                     GetThreadInfo           ( s32 Index, void* pBuffer );
        void                    ResetState              ();
        void                    Update                  ();
        s32                     UpdateThreadList        ();
static  void*                   DebugEventThread        ( void* pArg );
static  void*                   UpdateProcessThread     ( void* pArg );
        bool                    HaltResume              ( process_state State );
        s32                     Close                   ( bool KillProcess );
public:

        // Low overhead logging routines using a circular buffer of u64 values

        void                    ClearBreadcrumbs        ();
        void                    DropBreadcrumb          ( u64 Value );
        void                    ShowBreadcrumbs         ();
};

//==============================================================================
}} // namespace
//==============================================================================
