﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "..\tmagent.h"
#include <nn/dmnt/dmnt_Api.h>
#include "dbg_Modules.h"
#include "../dbghlp/dbghlp.h"

//==============================================================================
namespace tma { namespace dbg {
//==============================================================================

//extern ModuleDefinition* PlatformModules_GetList( nn::svc::Handle processHandle, s32* pNumberOfModules);

//==============================================================================
// Module definition
ModuleDefinition::ModuleDefinition( const char* pName, u64 Address, u64 Size  )
{
    memset( m_Name, 0, sizeof(m_Name) );
    TMA_STRNCPY( m_Name, pName, sizeof(m_Name) );
    m_Name[sizeof(m_Name) - 1] = 0;

    memset( m_ID, 0, sizeof(m_ID) );
    m_Address   = Address;
    m_Size      = Size;
}

ModuleDefinition::ModuleDefinition( nn::dbg::ModuleInfo* pDefined  )
{
    memset( m_Name, 0, sizeof(m_Name) );
    memcpy( m_ID, pDefined->moduleId, sizeof(m_ID) );

    m_Address   = pDefined->address;
    m_Size      = pDefined->size;
}

ModuleDefinition::ModuleDefinition()
{
    memset( m_Name, 0, sizeof(m_Name) );
    memset( m_ID, 0, sizeof(m_ID) );
    m_Address   = 0;
    m_Size      = 0;
}

//==============================================================================

ModuleDefinition::~ModuleDefinition()
{
}

//==============================================================================

bool ModuleDefinition::operator != ( const ModuleDefinition& Definition ) const
{
    // Same load address and size?
    if( m_Address == Definition.m_Address )
    {
        if( m_Size == Definition.m_Size )
        {
            //Made it here, so we passed all tests - We're equal
            if( strcmp( m_Name, Definition.m_Name ) == 0 )
            {
                return false;
            }
        }
    }

    // Not equal
    return true;
}

//==============================================================================

ModuleDefinition& ModuleDefinition::operator=( const ModuleDefinition& Definition )
{
    m_Address    = Definition.m_Address;
    m_Size  = Definition.m_Size;
    memcpy( m_Name, Definition.m_Name, sizeof(m_Name) );
    memcpy( m_ID, Definition.m_ID, sizeof(m_ID) );

    return *this;
}

//==============================================================================

enum
{
    MAX_NUMBER_OF_MODULES_QUERY = nn::dbg::MaxNsoModules + nn::dbg::MaxNroModules
};

nn::dbg::ModuleInfo pModuleBuffer[MAX_NUMBER_OF_MODULES_QUERY];

ModuleDefinition* GetList( u64 PID, nn::svc::Handle Handle, s32* pNumberOfModules )
{
    ModuleDefinition* pModules = NULL;

    nn::os::ProcessId Id;
    Id.value = PID;

    nn::Result Res = dbghlp::GetProcessModuleInfo( pNumberOfModules, pModuleBuffer, MAX_NUMBER_OF_MODULES_QUERY, Id );
    if( Res.IsSuccess() )
    {
        pModules = (ModuleDefinition*)s_Allocate( sizeof( ModuleDefinition ) * *pNumberOfModules );

        nn::svc::PageInfo PageInfo;
        nn::svc::MemoryInfo MemInfo;
        for(int ModuleIndex = 0; ModuleIndex < *pNumberOfModules; ModuleIndex += 1 )
        {
//            NN_SDK_LOG( "Module %d: address = 0x%llx, size = 0x%llx.\n", ModuleIndex, pModuleBuffer[ModuleIndex].address, pModuleBuffer[ModuleIndex].size );
            //========================================================================================
            // The build Id is only 20 characters, double-check that we have don't have any trailing garbage.
            //========================================================================================
            NN_SDK_ASSERT( (pModuleBuffer[ModuleIndex].moduleId[20] == 0) );

            ModuleDefinition defaultDefinition( &pModuleBuffer[ModuleIndex] );
            //Dig into the NSO to pull out the ELF file information -Wl,--nx-debuglink
            if( Handle != nn::svc::INVALID_HANDLE_VALUE )
            {
                //Get EX size to determine the start of the RO section. First entry should be a Pascal string with the .nss path
                nn::Result Result = dbghlp::QueryDebugProcessMemory( &MemInfo, &PageInfo, Handle, defaultDefinition.GetAddress() );

                //========================================================================================
                // This MemInfo.state check removed to address Siglo-66639, "nro does not have nrs path":
                // Users can load DLL code into memory types other than code.
                //========================================================================================
                if( Result.IsSuccess() ) // && nn::svc::MemoryState::MemoryState_Code == MemInfo.state )
                {
                    u64 Buffer = 0;
                    u64 Address = MemInfo.baseAddress + MemInfo.size;
                    Result = dbghlp::ReadDebugProcessMemory( (uintptr_t)&Buffer, Handle, (uintptr_t)Address, sizeof(u64) );
                    u64 Length = Buffer >> 32;
                    Length |= (Buffer << 32);

                    if( Result.IsSuccess() && Length > 0 && Length < MAX_PATH )
                    {
                        char Path[Length + 1];
                        Result = dbghlp::ReadDebugProcessMemory( (uintptr_t)Path, Handle, (uintptr_t)(Address + sizeof(u64)), Length );
                        Path[Length] = '\0';
                        if( Result.IsSuccess() )
                        {
                            defaultDefinition.SetName( Path );
                        }
                    }
                }
            }

            pModules[ ModuleIndex ] = defaultDefinition;
        }
    }

    return pModules;
}

//==============================================================================
// Module information
ModuleList::ModuleList()
{
    m_NumberOfModules = 0;
    m_pModules = NULL;
}
//==============================================================================
//
ModuleList::~ModuleList()
{
    Clear();
}

//==============================================================================

void ModuleList::Clear()
{
    if( m_pModules != NULL )
    {
        s_Deallocate(m_pModules, 0);
        m_pModules = NULL;
    }

    m_NumberOfModules = 0;
}

//==============================================================================

void ModuleList::Populate( char* pProcessName,  nn::svc::Handle processHandle, u64 ProcessId )
{
    Clear();

    m_pModules = GetList( ProcessId, processHandle, &m_NumberOfModules );

    // Make sure this list has valid names for every module.  If not,
    // just give it this process' name.

    //for(int Index = 0; Index < m_NumberOfModules; Index += 1 )
    //{
    //    if( m_pModules[ Index ].IsValid() == false )
    //    {
    //        m_pModules[ Index ].SetName( pProcessName );
    //    }
    //}
}

//==============================================================================

bool ModuleList::GetModuleAtIndex( s32 AtIndex, void* Buffer )
{
    if( (AtIndex >= 0) && (AtIndex < m_NumberOfModules) )
    {
        ModuleDefinition* pDef = &m_pModules[ AtIndex ];
        memcpy( Buffer, pDef, sizeof(ModuleDefinition) );
        return true;
    }

    return false;
}

//==============================================================================

bool ModuleList::operator != ( const ModuleList& List ) const
{
    // Same number of modules?
    if( m_NumberOfModules == List.m_NumberOfModules )
    {
        //OK, are the modules the same?  Keep in mind, these may not have the same index
        for( s32 MyIndex = 0; MyIndex < m_NumberOfModules; MyIndex++ )
        {
            ModuleDefinition* pMyModule = &m_pModules[MyIndex];
            s32 TheirIndex = 0;
            for( ; TheirIndex < m_NumberOfModules; TheirIndex++ )
            {
                if( *pMyModule != List.m_pModules[TheirIndex] )
                {
                    break;
                }
            }
            //Find this one?
            if( TheirIndex == m_NumberOfModules )
            {
                //Nope, so Yes, these are not equal
                return true;
            }
        }

        //Made it here, so we passed all tests - We're equal
        return false;
    }

    // Not equal
    return true;
}

//==============================================================================

ModuleList& ModuleList::operator=( const ModuleList& List )
{
    Clear();

    m_NumberOfModules = List.m_NumberOfModules;

    m_pModules = (ModuleDefinition*)s_Allocate( sizeof( ModuleDefinition ) * m_NumberOfModules );

    for( s32 Index = 0; Index < m_NumberOfModules; Index++ )
    {
        m_pModules[Index] = List.m_pModules[Index];
    }
    return *this;
}

//==============================================================================

Modules::Modules()
{
    m_NeedsUpdate = true;
    m_ProcessId = 0;
}

//==============================================================================

Modules::~Modules()
{
    m_Modules.Clear();
}

//==============================================================================

void Modules::Init( char* pProcessName, nn::svc::Handle Handle, u64 ProcessId )
{
    m_NeedsUpdate = true;
    m_pProcessName = pProcessName;
    m_ProcessHandle = Handle;
    m_ProcessId = ProcessId;
    m_Modules.Clear();
}

//==============================================================================

void Modules::Kill()
{
    m_NeedsUpdate = false;
    m_Modules.Clear();
}

//==============================================================================

void Modules::Attach( nn::svc::Handle Handle )
{
    m_ProcessHandle = Handle;
    m_NeedsUpdate = true;
}

//==============================================================================

s32 Modules::GetModuleAtIndex( s32 Index, void* Buffer )
{
    return m_Modules.GetModuleAtIndex( Index, Buffer );
}

//==============================================================================

s32 Modules::GetCount()
{
    if ( m_NeedsUpdate )
    {
        Update();
        m_NeedsUpdate = false;
    }

    return m_Modules.GetCount();

}

//==============================================================================

bool Modules::Update()
{
    bool Ret = false;

    //See if we just need to make a new list (first time through, or different debug process)
    if( m_Modules.GetCount() == 0 || m_NeedsUpdate )
    {
        //Just call populate on our list.
        m_Modules.Populate( m_pProcessName, m_ProcessHandle, m_ProcessId );

        Ret = true;
    }
    else
    {
        //Did our list change?  Get the current list and compare...
        ModuleList currentList;

        currentList.Populate( m_pProcessName, m_ProcessHandle, m_ProcessId );

        if( currentList != m_Modules )
        {
            m_Modules = currentList;
            Ret = true;
        }
    }

    return Ret;
}

//==============================================================================
}} // namespace
//==============================================================================
