﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

//==============================================================================
//  INCLUDES
//==============================================================================

#include <sys/time.h>
#include <sys/types.h>

//==============================================================================
//  TYPES
//==============================================================================

/*
    32-bit ELF base types.          64-bit ELF base types.
    ----------------------------    ----------------------------
    typedef u32   Elf32_Addr;       typedef u64   Elf64_Addr;
    typedef u16   Elf32_Half;       typedef u16   Elf64_Half;
                                    typedef s16   Elf64_SHalf;
    typedef u32   Elf32_Off;        typedef u64   Elf64_Off;
    typedef s32   Elf32_Sword;      typedef s32   Elf64_Sword;
    typedef u32   Elf32_Word;       typedef u32   Elf64_Word;
                                    typedef u64   Elf64_Xword;
                                    typedef s64   Elf64_Sxword;
*/

typedef int8_t      s8;
typedef uint8_t     u8;
typedef int16_t     s16;
typedef uint16_t    u16;
typedef int32_t     s32;
typedef uint32_t    u32;
typedef int64_t     s64;
typedef uint64_t    u64;

//==============================================================================
//  ARM Processor Specific values for e_flags
//==============================================================================

enum
{
    EF_ARM_ABIMASK        = 0xFF000000,
    EF_ARM_EABI_VER5      = 0x05000000,
    EF_ARM_HASENTRY       = 0x00000002,
    EF_ARM_ABI_FLOAT_HARD = 0x00000400
};

//==============================================================================
//  FILE HEADER and supporting symbolic constants.
//==============================================================================

struct elf_signature
{
    u8      Byte[16];       // File signature.                          {e_ident}
};

//==============================================================================

struct elf32_header         // 32 bit ELF header record.                {elf32_hdr}
{
    u8      Signature[16];  // File signature.                          {e_ident}
    u16     Type;           // Object file type.                        {e_type}
    u16     Machine;        // Required architexture.                   {e_machine}
    u32     Version;        // Object file version.                     {e_version}
    u32     Entry;          // Entry point virtual address.             {e_entry}
    u32     ProgramOffset;  // Offset to program header table.          {e_phoff}
    u32     SectionOffset;  // Offset to section header table.          {e_shoff}
    u32     Flags;          // Processor specific flags.                {e_flags}
    u16     FileHdrSize;    // Size of file header record.              {e_ehsize}
    u16     ProgramHdrSize; // Size of program header record.           {e_phentsize}
    u16     NProgramHdrs;   // Number of program header records.        {e_phnum}
    u16     SectionHdrSize; // Size of section header record.           {e_shentsize}
    u16     NSectionHdrs;   // Number of section header records.        {e_shnum}
    u16     iStrSection;    // Section header for name string table.    {e_shstrndx}
};

//==============================================================================

struct elf64_header         // 64 bit ELF header record.                {elf64_hdr}
{
    u8      Signature[16];  // File signature.                          {e_ident}
    u16     Type;           // Object file type.                        {e_type}
    u16     Machine;        // Required architexture.                   {e_machine}
    u32     Version;        // Object file version.                     {e_version}
    u64     Entry;          // Entry point virtual address.             {e_entry}
    u64     ProgramOffset;  // Offset to program header table.          {e_phoff}
    u64     SectionOffset;  // Offset to section header table.          {e_shoff}
    u32     Flags;          // Processor specific flags.                {e_flags}
    u16     FileHdrSize;    // Size of file header record.              {e_ehsize}
    u16     ProgramHdrSize; // Size of program header record.           {e_phentsize}
    u16     NProgramHdrs;   // Number of program header records.        {e_phnum}
    u16     SectionHdrSize; // Size of section header record.           {e_shentsize}
    u16     NSectionHdrs;   // Number of section header records.        {e_shnum}
    u16     iStrSection;    // Section header for name string table.    {e_shstrndx}
};

//==============================================================================
//  Index into elf_header.Magic array.

enum
{
    EI_MAG0                   = 0,   // 0x7F
    EI_MAG1                   = 1,   // 'E'
    EI_MAG2                   = 2,   // 'L'
    EI_MAG3                   = 3,   // 'F'
    EI_CLASS                  = 4,   //
    EI_DATA                   = 5,   //
    EI_VERSION                = 6,   //
    EI_OSABI                  = 7,   //
    EI_ABIVERSION             = 8,   //
    EI_PAD                    = 9    //
};

//==============================================================================
//  Values for elf_header.Magic[EI_CLASS] field.

enum
{
    ELFCLASSNONE              = 0,   //
    ELFCLASS32                = 1,   // 32 bit ELF file.
    ELFCLASS64                = 2    // 64 bit ELF file.
};

//==============================================================================
//  Values for elf_header.Magic[EI_DATA] field.

enum
{
    ELFDATANONE               = 0,   // Invalid encoding.
    ELFDATA2LSB               = 1,   // 2's compliment, little endian.
    ELFDATA2MSB               = 2    // 2's compliment, big endian.
};

//==============================================================================
//  Values for elf_header.Magic[EI_OSABI] field.

enum
{
    ELFOSABI_NONE           =   0,   // UNIX System V ABI.
    ELFOSABI_HPUX           =   1,   // HP-UX.
    ELFOSABI_NETBSD         =   2,   // NetBSD.
    ELFOSABI_LINUX          =   3,   // GNU/Linux.
    ELFOSABI_HURD           =   4,   // GNU/Hurd.
    ELFOSABI_SOLARIS        =   6,   // Solaris.
    ELFOSABI_AIX            =   7,   // AIX.
    ELFOSABI_IRIX           =   8,   // IRIX.
    ELFOSABI_FREEBSD        =   9,   // FreeBSD.
    ELFOSABI_TRU64          =  10,   // TRU64 UNIX.
    ELFOSABI_MODESTO        =  11,   // Novell Modesto.
    ELFOSABI_OPENBSD        =  12,   // OpenBSD.
    ELFOSABI_OPENVMS        =  13,   // OpenVMS.
    ELFOSABI_NSK            =  14,   // Hewlett-Packard NSK.
    ELFOSABI_AROS           =  15,   // Amiga Research OS.
    ELFOSABI_ARM            =  97,   // ARM.
    ELFOSABI_STANDALONE     = 255    // Standalone.
};

//==============================================================================
//  Values for elf_header.Type field.                                   {e_type}

enum
{
    ET_NONE              =      0,   // No file type
    ET_REL               =      1,   // Relocatable file
    ET_EXEC              =      2,   // Executable file
    ET_DYN               =      3,   // Shared object file.
    ET_CORE              =      4,   // Core file.
    ET_LOOS              = 0xFE00,   // Operating system-specific.
    ET_HIOS              = 0xFEFF,   // Operating system-specific.
    ET_LOPROC            = 0xFF00,   // Processor-specific.
    ET_HIPROC            = 0xFFFF    // Processor-specific.
};

//==============================================================================
//  Values for elf_header.Machine field.                                {e_machine}

enum
{
    EM_NONE                  =  0,   // Invalid machine.
    EM_MIPS                  =  8,   // MIPS.
    EM_PPC                   = 20,   // Power PC.
    EM_PPC64                 = 21,   // Power PC, 64 bit.
    EM_ARM                   = 40    // ARM.
};

//==============================================================================
//  PROGRAM HEADER and supporting symbolic constants.
//==============================================================================

struct elf32_program_header     // 32 bit program header record.        {elf32_phdr}
{
    u32         Type;           // Indicates program segment type.      {p_type}
    u32         Offset;         // Segment file offset.                 {p_offset}
    u32         VirtAddr;       // Segment virtual address.             {p_vaddr}
    u32         PhysAddr;       // Segment physical address.            {p_paddr}
    u32         FileSize;       // Segment size within file.            {p_filesz}
    u32         MemorySize;     // Segment size within memory.          {p_memsz}
    u32         Flags;          // Segment flags.                       {p_flags}
    u32         Alignment;      // Segment alignment.                   {p_align}
};

//==============================================================================

struct elf64_program_header     // 64 bit program header record.        {elf64_phdr}
{
    u32         Type;           // Indicates program segment type.      {p_type}
    u32         Flags;          // Segment flags.                       {p_flags}
    u64         Offset;         // Segment file offset.                 {p_offset}
    u64         VirtAddr;       // Segment virtual address.             {p_vaddr}
    u64         PhysAddr;       // Segment physical address.            {p_paddr}
    u64         FileSize;       // Segment size within file.            {p_filesz}
    u64         MemorySize;     // Segment size within memory.          {p_memsz}
    u64         Alignment;      // Segment alignment.                   {p_align}
};

//==============================================================================
//  Values for elf_program_header.Type field.                           {p_type}

enum
{
    PT_NULL          =          0,   // Unused.
    PT_LOAD          =          1,   // Loadable program segment.
    PT_DYNAMIC       =          2,   // Dynamic linking information.
    PT_INTERP        =          3,   // Program interpreter.
    PT_NOTE          =          4,   // Auxiliary information.
    PT_SHLIB         =          5,   //
    PT_PHDR          =          6,   //
    PT_TLS           =          7,   // Thread local storage.
    PT_LOPROC        = 0x70000000,   //
    PT_HIPROC        = 0x7FFFFFFF    //
};

//==============================================================================
//  Values for elf_program_header.Flags field.                          {p_flags}

enum
{
    PF_R              = (1 <<  2),   // Read.
    PF_W              = (1 <<  1),   // Write.
    PF_X              = (1 <<  0)    // Execute.
};

//==============================================================================
//  CORE file structures
//==============================================================================

enum
{
    NGREG     = 16
};

typedef int         greg_t;
typedef greg_t      gregset_t[NGREG];
typedef gregset_t   elf_regset_t;

struct elf_siginfo
{
    s32                 SigNo;      // Signal number
    s32                 Code;       // Extra code
    s32                 ErrNo;      // Error number
};

struct elf_prstatus
{
    struct elf_siginfo  SigInfo;
    s32                 CurSig;     // Current signal
    u64                 SigPend;    // Set of pending signals
    u64                 SigHold;    // Set of held signals
    pid_t               Pid;
    pid_t               pPid;
    pid_t               pGrp;
    pid_t               Sid;
    struct timeval      UTime;      // User time
    struct timeval      STime;      // System time
    struct timeval      CUTime;     // Cumulative User time
    struct timeval      CSTime;     // CUmulative System time
    elf_regset_t        Reg;        // General purpose registers
    s32                 FPvalid;    // Math co-processor being used
};

enum
{
    ELF_PRARGS    = 80
};

struct elf_prpsinfo
{
    char                State;
    char                SName;
    char                Zomb;
    char                Nice;
    u64                 Flag;
    u32                 Uid;
    u32                 Gid;
    pid_t               Pid;
    pid_t               pPid;
    pid_t               pGrp;
    pid_t               Sid;
    char                FName[16];
    char                PSArgs[ELF_PRARGS];
};

typedef struct elf_prstatus prstatus_t;
typedef struct elf_prpsinfo prpsinfo_t;

// Note types when the ELF header type field is ET_CORE

enum
{
    NT_PRSTATUS   = 1,
    NT_PRFPREG    = 2,
    NT_PRPSINFO   = 3,
    NT_TASKSTRUCT = 4,
    NT_AUXV       = 6
};

struct elf_note                 // Note header
{
    u32                 NameSz; // Name size
    u32                 DescSz; // Description size, if any, else 0
    u32                 Type;   // Content type

    // Name and Desc padded to 4-byte boundaries
    // NameSz and DescSz do not include the padding
};

///==============================================================================
