﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "..\tmipc\tmipc_result.h"
#include "coredump_service.h"
#include "coredump_Task.h"
#include <nn/dmnt/dmnt_Api.h>
#include "..\dbg\dbg_ProcessMgr.h"
#include <nn/nn_SystemThreadDefinition.h>

#if DEBUG
#define COREDUMP_TRACE( channel, ... ) NN_SDK_LOG( "[tma][coredump_task][" ); \
    NN_SDK_LOG( channel ); \
    NN_SDK_LOG( "] - " ); \
    NN_SDK_LOG( __VA_ARGS__ ); \
    NN_SDK_LOG( "\n" )
#define COREDUMP_LOG(...) NN_SDK_LOG( "[tma][coredump_task] - " ); NN_SDK_LOG( __VA_ARGS__ ); NN_SDK_LOG( "\n" )
#else
#define COREDUMP_TRACE(channel, ... )
#define COREDUMP_LOG(...)
#endif

//==============================================================================
namespace tma { namespace coredump {
//==============================================================================

CoredumpTask::CoredumpTask( Service* pOwner )
:   m_Cancel( false )
,   m_Done( false )
,   m_ProcessId( 0 )
,   m_pTTYData( NULL )
,   m_SizeOfTTYData(0)
,   m_pOwner( pOwner )
{
    COREDUMP_LOG("CoredumpTask::CoredumpTask");
    nn::os::InitializeLightEvent( &m_Event, false, nn::os::EventClearMode_AutoClear );

    //====================================================================================
    // We're owned by our service, who is responsible for our care and maintenance.
    // Done this way to prevent circular dependencies in the task system.
    //====================================================================================
    SetExternallyOwned( true );
}

//==============================================================================

CoredumpTask::~CoredumpTask()
{
    COREDUMP_LOG( "CoredumpTask::~CoredumpTask" );
    Finalize();
}

//==============================================================================

void CoredumpTask::Finalize()
{
    COREDUMP_LOG( "CoredumpTask::Finalize" );
    if( m_Status != StatusComplete )
    {
        // Wait for our thread to quit
        m_WorkThread.Join();
        m_WorkThread.Destroy();
        nn::os::FinalizeLightEvent( &m_Event );
    }

    if( m_pTTYData != NULL )
    {
        s_Deallocate( m_pTTYData, m_SizeOfTTYData );
        m_pTTYData = NULL;
        m_SizeOfTTYData = 0;
    }
}

//==============================================================================

void CoredumpTask::OnRecvPacket( tmipc::Packet* pPacket )
{
    s32 NotificationType = 0;
    pPacket->ReadS32( NotificationType );
    COREDUMP_TRACE("CoredumpTask::OnRecvPacket", "OnRecvPacket notification type %d", NotificationType );

    switch( NotificationType )
    {
        case tma::coredump::COREDUMP_CANCEL:
            // Tell ourselves
            m_Cancel = true;
            nn::os::SignalLightEvent( &m_Event );
            COREDUMP_TRACE("CoredumpTask::OnRecvPacket", "Signalled CANCEL" );
            break;

        case tma::coredump::COREDUMP_CLOSE:
            m_Done = true;
            nn::os::SignalLightEvent( &m_Event );
            COREDUMP_TRACE("CoredumpTask::OnRecvPacket", "Complete" );
            break;

        default:
            break;

        case tma::coredump::COREDUMP_ADD_TTY:
            pPacket->ReadU32( m_SizeOfTTYData );
            m_pTTYData = (char*)s_Allocate( m_SizeOfTTYData );
            pPacket->ReadData( m_pTTYData, m_SizeOfTTYData );
            nn::os::SignalLightEvent( &m_Event );
            COREDUMP_TRACE("CoredumpTask::OnRecvPacket", "Signalled TTY" );
            break;

            //todo:
        case tma::coredump::COREDUMP_ADD_IMAGE:
            break;
    }

}

//==============================================================================

void CoredumpTask::OnSendPacket( tmipc::Packet* pPacket )
{
    // TODO - Handle Packet
    (void)pPacket;
//    Complete();
}

//==============================================================================

enum
{
    StackSize = (8192 * 10)
};

//==============================================================================

void* CoredumpTask::WorkThread( void* pArgs )
{
    COREDUMP_TRACE("CoredumpTask::WorkThread", "WorkThread START" );
    CoredumpTask* pThis = (CoredumpTask*)pArgs;
    Service* pOriginatingService = (Service*)pThis->GetService();
    tma::dbg::ProcessMgr* pMgr = pOriginatingService->GetProcessManager();

    s32 PercentDone = 0;
    while( pThis->m_Done == false && pThis->m_Cancel == false && ( pThis->m_Status != StatusCanceled ) )
    {
        if( PercentDone < 100 )
        {
            s32 res = pMgr->ContinueCoreDump( pThis->m_ProcessId, &PercentDone );
            if( res == tmapi::result::RESULT_OK )
            {
                if( PercentDone == 100 )
                {
                    pThis->Notify( COREDUMP_NOTIFICATION_COMPLETE, tmapi::result::RESULT_OK );
                }
                else
                {
                    pThis->Notify( COREDUMP_NOTIFICATION_PROGRESS, PercentDone );
                }
            }
            else
            {
                COREDUMP_TRACE("CoredumpTask::WorkThread", "WorkThread - ContinueCoreDump FAILED with error code %d", res );
                //Set our cancel flag so that we'll send the cancel notification (below).
                pThis->m_Cancel = true;
                break;
            }

            if( pThis->GetExternallyOwned() == false )
            {
                COREDUMP_TRACE( "CoredumpTask::WorkThread", "Setting ourselves to externally owned." );
                pThis->SetExternallyOwned( true );
            }
        }
        else
        {
            nn::os::WaitLightEvent( &pThis->m_Event );
            nn::os::ClearLightEvent( &pThis->m_Event );
            if( pThis->m_Done == false && pThis->m_Cancel == false )
            {
                if( pThis->m_pTTYData != NULL )
                {
                    // Add the TTY
                    pMgr->AddTTYToCoreDump( pThis->m_ProcessId, pThis->m_pTTYData, pThis->m_SizeOfTTYData );

                    //Clean up
                    s_Deallocate( pThis->m_pTTYData, pThis->m_SizeOfTTYData );
                    pThis->m_pTTYData = NULL;
                    pThis->m_SizeOfTTYData = 0;
                }
            }
        }
    }

    COREDUMP_TRACE( "CoredumpTask::WorkThread", "Closing core dump:  m_Done = %d, m_Cancel = %d, m_Status = %d", pThis->m_Done, pThis->m_Cancel, pThis->m_Status);

    // Close up.
    pMgr->CloseCoreDump( pThis->m_ProcessId );

    // If we were cancelled, signal that we're done cancelling.
    if( pThis->m_Cancel == true && ( pThis->m_Status != StatusCanceled ) )
    {
        pThis->Notify( COREDUMP_NOTIFICATION_CANCELLED, tmapi::result::RESULT_COREDUMP_CANCELLED );
    }

    //Mark ourselves complete
    if( pThis->m_Status != StatusCanceled )
    {
        pThis->Complete();
    }

    //Tell our owner
    pThis->m_pOwner->OnComplete( pThis );

    COREDUMP_TRACE("CoredumpTask::WorkThread", "WorkThread END" );
    return NULL;
}

//==============================================================================

void CoredumpTask::OnInitiate( tmipc::Packet* pPacket )
{
    char OutputName[ 1024 ];

    u8  IsMiniDump = 0;

    // Read our data.
    pPacket->ReadU64( m_ProcessId );
    pPacket->ReadU8( IsMiniDump );
    pPacket->ReadString( OutputName, sizeof(OutputName) );

    COREDUMP_TRACE("CoredumpTask::OnInitiate", "OnInitiate - write %s", OutputName );

    tma::dbg::ProcessMgr* pMgr = m_pOwner->GetProcessManager();
    if( pMgr != NULL )
    {
        s32 res = pMgr->InitiateCoreDump( m_ProcessId, OutputName, IsMiniDump );
        if( res == tmapi::result::RESULT_OK )
        {
            // Start our thread
            m_WorkThread.Start( WorkThread, this, StackSize, NN_SYSTEM_THREAD_PRIORITY(tma, DbgWorkThread), "CoreDumpThread" );
        }
        else
        {
            Notify( COREDUMP_NOTIFICATION_ERROR, res );
        }
    }
}

//==============================================================================

void CoredumpTask::Notify( notification Type, int32_t Data )
{
    COREDUMP_TRACE("CoredumpTask::Notify", "Notify %d", Type );
    tmipc::Packet* p = AllocSendPacket();
    p->WriteS32( Type );
    p->WriteS32( Data );
    m_pServicesManager->Send( p );
}

//==============================================================================

//==============================================================================
}} // namespace
//==============================================================================
