﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>
#include <nn/nn_SdkLog.h>
#include <nn/os.h>
#include "connection_Api.h"

namespace
{
    bool g_IsInitialized;
    nn::os::SystemEventType g_ConnectionEvent;
    nn::os::SystemEventType g_DisconnectionEvent;
    nn::os::SystemEventType g_ConnectionEventForSystem;
    nn::os::SystemEventType g_DisconnectionEventForSystem;

    struct StaticMutex
    {
        nn::os::MutexType    m_Mutex;
        void lock() NN_NOEXCEPT
        {
            nn::os::LockMutex(&m_Mutex);
        }
        void unlock() NN_NOEXCEPT
        {
            nn::os::UnlockMutex(&m_Mutex);
        }
    };
    StaticMutex g_Mutex = { NN_OS_MUTEX_INITIALIZER(false) };
}

//==============================================================================
namespace nn { namespace tma { namespace connection {
//==============================================================================
// Call this function at TMA initialization.
void Initialize()
{
    std::lock_guard<StaticMutex> lock(g_Mutex);
    nn::os::CreateSystemEvent(&g_ConnectionEvent, nn::os::EventClearMode_ManualClear, true);
    nn::os::CreateSystemEvent(&g_DisconnectionEvent, nn::os::EventClearMode_ManualClear, true);
    nn::os::CreateSystemEvent(&g_ConnectionEventForSystem, nn::os::EventClearMode_ManualClear, true);
    nn::os::CreateSystemEvent(&g_DisconnectionEventForSystem, nn::os::EventClearMode_ManualClear, true);
    g_IsInitialized = true;
}

//==============================================================================
// Call this function at TMA Finalization.
void Finalize()
{
    std::lock_guard<StaticMutex> lock(g_Mutex);
    g_IsInitialized = false;
    nn::os::DestroySystemEvent(&g_ConnectionEvent);
    nn::os::DestroySystemEvent(&g_DisconnectionEvent);
    nn::os::DestroySystemEvent(&g_ConnectionEventForSystem);
    nn::os::DestroySystemEvent(&g_DisconnectionEventForSystem);
}

//==============================================================================
// Call this function when TMS and TMA are connected.
void OnConnected()
{
    std::lock_guard<StaticMutex> lock(g_Mutex);

    if (g_IsInitialized)
    {
        nn::os::SignalSystemEvent(&g_ConnectionEvent);
        nn::os::SignalSystemEvent(&g_ConnectionEventForSystem);
        nn::os::ClearSystemEvent(&g_DisconnectionEvent);
        nn::os::ClearSystemEvent(&g_DisconnectionEventForSystem);
    }
}

//==============================================================================
// Call this function when TMS and TMA are disconnected.
void OnDisconnected()
{
    if (g_IsInitialized)
    {
        std::lock_guard<StaticMutex> lock(g_Mutex);
        nn::os::ClearSystemEvent(&g_ConnectionEvent);
        nn::os::ClearSystemEvent(&g_ConnectionEventForSystem);
        nn::os::SignalSystemEvent(&g_DisconnectionEvent);
        nn::os::SignalSystemEvent(&g_DisconnectionEventForSystem);
    }
}

//==============================================================================
nn::os::SystemEventType* GetConnectionEvent()
{
    std::lock_guard<StaticMutex> lock(g_Mutex);
    return &g_ConnectionEvent;
}

nn::os::SystemEventType* GetDisconnectionEvent()
{
    std::lock_guard<StaticMutex> lock(g_Mutex);
    return &g_DisconnectionEvent;
}

nn::os::SystemEventType* GetConnectionEventForSystem()
{
    std::lock_guard<StaticMutex> lock(g_Mutex);
    return &g_ConnectionEventForSystem;
}

nn::os::SystemEventType* GetDisconnectionEventForSystem()
{
    std::lock_guard<StaticMutex> lock(g_Mutex);
    return &g_DisconnectionEventForSystem;
}

//==============================================================================
}}} // namespace nn::tma::connection
//==============================================================================
