﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "stdafx.h"
#include "Version.h"
#include "tmagent.h"
#include "tmipc/tmipc_defines.h"
#include <nn/settings/fwdbg/settings_SettingsGetterApi.h>
#include <nn/settings/system/settings_FirmwareVersion.h>

#if defined( NN_TMA_SIM_INSTANCE ) // || defined( TMA_TARGET_WIN )
#define TMA_MACRO_VALUE(a) a
#define TMA_USE_WINSOCK TMA_MACRO_VALUE(1)
#endif

#ifdef TMA_USE_WINSOCK
#include <IPTypes.h>
#include <Iphlpapi.h>
#endif

#if defined(NN_BUILD_CONFIG_HARDWARE_BDSLIMX6) || defined(NN_BUILD_CONFIG_HARDWARE_JETSONTK1) || defined(NN_BUILD_CONFIG_HARDWARE_NX) || defined(NN_BUILD_CONFIG_HARDWARE_JETSONTK2)
#define NN_DETAIL_TMA_NX_RELATED_HARDWARE
#endif

// Anonymous internal data.
namespace
{
    const char* TMAVersionNumber = "6.1.8.24319";

#if defined( TMA_USE_WINSOCK )

void GetMAC( u8* pMAC )
{
    // Read Network Adapter info
    {
        PIP_ADAPTER_ADDRESSES AdapterAddresses = NULL;
        ULONG OutBufferLength = 0;
        ULONG RetVal = NO_ERROR;

        // Loop until we have the adapter table
        for( s32 i=0; i<5; i++ )
        {
            // Get the table
            RetVal = GetAdaptersAddresses( AF_INET, GAA_FLAG_INCLUDE_PREFIX, NULL, AdapterAddresses, &OutBufferLength );
            if( RetVal != ERROR_BUFFER_OVERFLOW )
            {
                break;
            }
            // Free the current buffer, it wasn't big enough
            if( AdapterAddresses != NULL )
            {
                free( AdapterAddresses );
            }

            // Allocate a new buffer
            AdapterAddresses = (PIP_ADAPTER_ADDRESSES)malloc( OutBufferLength );
            if( AdapterAddresses == NULL )
            {
                RetVal = GetLastError();
                break;
            }
        }

        // Did we get the table?
        if( RetVal == NO_ERROR )
        {
            // Get the first adapter & loop
            PIP_ADAPTER_ADDRESSES pAdapter = AdapterAddresses;
            bool bFound = false;
            while( pAdapter && !bFound )
            {
                if( pAdapter->FirstUnicastAddress && pAdapter->FirstPrefix )
                {
                    switch( pAdapter->FirstUnicastAddress->Address.lpSockaddr->sa_family )
                    {
                    case AF_INET:
                        {
                            if( pAdapter->FirstUnicastAddress->Address.iSockaddrLength >= 6 )
                            {
                                // Copy the MAC.
                                memcpy( pMAC, pAdapter->PhysicalAddress, 6 );
                                bFound = true;
                            }
                        }
                        break;
                    default:
                    case AF_INET6:
                        {
                        }
                        break;
                    }
                }

                // Next adapater
                pAdapter = pAdapter->Next;
            }
        }

        // Free the adapter table
        free( AdapterAddresses );
    }
}

#endif
}

//==============================================================================
namespace tma {
//==============================================================================

bool g_HasWokenUp = false;  // Default to false: not coming from sleep state.

//==============================================================================

const char* GetTMAVersionNumber()
{
    return TMAVersionNumber;
}

//==============================================================================
// Generates a beacon response to send to TargetManager.
bool GenerateBeaconResponse(    char*       pBeaconResponse,
                                const s32   BufferLength,
                                const char* pConnectionType,
                                bool        IncludeMacAddress )
{
    bool Success = { false };

    const s32 BeaconResponseLength = { 1024 };
    char BeaconResponse[BeaconResponseLength] = "";
    char WorkString[512];

    // Document start.
    BeaconResponse[0] = '\0';

    char DeviceName[nn::settings::fwdbg::SettingsNameLengthMax];
    memset( DeviceName, '\0', nn::settings::fwdbg::SettingsNameLengthMax );
    const u32 FirmwareVersionSize = { sizeof(static_cast<nn::settings::system::FirmwareVersion*>(nullptr)->displayName)
                                    + sizeof(static_cast<nn::settings::system::FirmwareVersion*>(nullptr)->revision) };
    char FirmwareVersion[FirmwareVersionSize + 1];
    memset( FirmwareVersion, '\0', FirmwareVersionSize + 1 );
#if !defined(NN_TMA_SIM_INSTANCE) && defined(NN_DETAIL_TMA_NX_RELATED_HARDWARE)
    // Retrieve the Firmware Version from the OS.
    nn::settings::system::FirmwareVersion version;
    nn::settings::system::GetFirmwareVersion( &version );
    if ( (version.displayName[0] != '\0') && (version.revision[0] != '\0') )
    {
        const char *pDisplayName = { version.displayName };
        const char TextToRemove[] = { "NintendoSDK Firmware for NX " };
        if( strncmp( version.displayName, TextToRemove, sizeof(TextToRemove) - 1 ) == 0 )
        {
            pDisplayName = version.displayName + sizeof(TextToRemove) - 1;
        }
        // SigloNTD-10240 (now Siglo-56180): Show the minimal hash of Target FW on TMS.
        // The final version number will be in the format: 1.2.3-4 (01234567).
        // Start iterating at 8, for the requested "first 8 characters".
        // Siglo-79949: Increase displayed hash numbers in TM.
        // "We determined to increase revision hash to display from 7 or
        // 8 digits to 10 digits (DevMenu and NNSDK package description).
        // Could you increase the digits to display in TM to 10 also?"
        const s32 Limit = sizeof(version.revision) / sizeof(*version.revision);
        for( s32 Index = 10; Index < Limit; Index++ )
        {
            version.revision[Index] = '\0';
        }
        TMA_SPRINTF( FirmwareVersion, "FW: %s (%s)\n", pDisplayName, version.revision );
    }

    if (version.major > 1)
    {
        nn::settings::fwdbg::GetSettingsItemValue(DeviceName, nn::settings::fwdbg::SettingsNameLengthMax, "target_manager", "device_name");
    }
#endif

    // Spec.
#if defined(NN_BUILD_CONFIG_SPEC_GENERIC)
    const char Spec[] = { "Spec: Generic\n" };
#elif defined(NN_BUILD_CONFIG_SPEC_NX)
    const char Spec[] = { "Spec: NX\n" };
#else
    const char Spec[] = { "Spec: Unknown\n" };
#endif
    TMA_STRCAT( BeaconResponse, BeaconResponseLength, Spec );

    // MAC.
    if( IncludeMacAddress )
    {
        u8 MAC[8] = { 0 };
#if defined(NN_BUILD_CONFIG_OS_WIN)
        GetMAC( &MAC[0] );
#elif defined(NN_BUILD_CONFIG_HARDWARE_JETSONTK1)
       // nn::socket0::GetMacAddress( &MAC[0], sizeof(MAC) );
#elif defined(NN_BUILD_CONFIG_HARDWARE_JETSONTK2)
      //  nn::socket0::GetMacAddress( &MAC[0], sizeof(MAC) );
#else
    #if defined( TMA_USE_WINSOCK )
        GetMAC( &MAC[0] );
    #endif
#endif
        TMA_SPRINTF( WorkString, "MAC: %02X-%02X-%02X-%02X-%02X-%02X\n",
                    MAC[0], MAC[1], MAC[2], MAC[3], MAC[4], MAC[5] );
        TMA_STRCAT( BeaconResponse, BeaconResponseLength, WorkString );
    }

    // TMA Version number.
    TMA_SPRINTF( WorkString, "TMA: %s\n", tma::GetTMAVersionNumber() );
    TMA_STRCAT( BeaconResponse, BeaconResponseLength, WorkString );

    // TMIPC Version number.
    TMA_SPRINTF( WorkString, "TMIPC: %d\n", tmipc::TMIPC_VERSION );
    TMA_STRCAT( BeaconResponse, BeaconResponseLength, WorkString );

    // Connection Type.
    TMA_SPRINTF( WorkString, "Conn: %s\n", pConnectionType );
    TMA_STRCAT( BeaconResponse, BeaconResponseLength, WorkString );

    // Hardware Type.
    TMA_SPRINTF( WorkString, "HW: %s\n", tma::GetTargetConfigurationId() );
    TMA_STRCAT( BeaconResponse, BeaconResponseLength, WorkString );

    // Target Name.
    if( DeviceName[0] != '\0' )
    {
        TMA_SPRINTF( WorkString, "Name: %s\n", DeviceName );
        TMA_STRCAT( BeaconResponse, BeaconResponseLength, WorkString );
    }

    // Get the hardware serial number.
    if (tma::GetTargetSerialNumber()[0] != '\0' )
    {
        TMA_SPRINTF( WorkString, "SN: %s\n", tma::GetTargetSerialNumber() );
        TMA_STRCAT( BeaconResponse, BeaconResponseLength, WorkString );
    }

    // Backwards compatibility ID.
    TMA_SPRINTF( WorkString, "BCID: %d\n", tmipc::ProtocolVersion);
    TMA_STRCAT( BeaconResponse, BeaconResponseLength, WorkString );

    // Previous state.  0 = Booted (not coming from a sleep state); 1 = From sleep.
    TMA_SPRINTF( WorkString, "PS: %s\n", GetHasWokenUp() ? "1" : "0" );
    TMA_STRCAT( BeaconResponse, BeaconResponseLength, WorkString );

    // Firmware Version.
    TMA_STRCAT( BeaconResponse, BeaconResponseLength, FirmwareVersion );

    // Only fill in the given buffer if the buffer is large enough.
    const s32 ActualBeaconResponseLength = static_cast<s32>(strlen( BeaconResponse ) + 1);
    if( ActualBeaconResponseLength <= BufferLength )
    {
        TMA_STRCPY( pBeaconResponse, ActualBeaconResponseLength, BeaconResponse );
        Success = true;
    }

    return Success;
}

//==============================================================================
// Get/set the Target's previous state.  This affects the Beacon Respons's "Previous State" key/value.
void SetHasWokenUp( bool bHasWokenUp )
{
    g_HasWokenUp = bHasWokenUp;
}

bool GetHasWokenUp()
{
    return g_HasWokenUp;
}

//==============================================================================
// Sets the TMSInfo to a default state.
void SetToDefaultState( TMSInfo &TmsInfo )
{
    TmsInfo.BCID        = 0;
    TmsInfo.WasSleeping = false;
}

//==============================================================================
// Decode the given string into the TMSInfo.
void DecodeTMSInfoString( TMSInfo &TmsInfo, const char* pString )
{
    // Possible string tokens.
    static const char Separator[] = { ": " };
    static const char BCIDToken[] = { "BCID" };
    static const char PSToken[]   = { "PS" };
//static const char NewExampleToken[] = { "NewExample" };

    // Set the tmsInfo to a default state.
    SetToDefaultState( TmsInfo );

    // Prevent nullptr dereference exceptions.
    if( pString != nullptr )
    {
        const s32 Length = { static_cast<s32>(strlen( pString )) };
        const char *pStringEnd = { pString + Length };
        // Start parsing the string.
        const char *pNameStart  = pString;
        const char *pNameEnd    = nullptr;
        const char *pValueStart = nullptr;
        for( const char *pCharacter = pString; pCharacter < pStringEnd; pCharacter++ )
        {
            switch( *pCharacter )
            {
            case ':':
            {
                if( pNameEnd == nullptr )
                {
                    // Make sure:
                    // 1) The ':' appears *after* the first character in the string.
                    // 2) The character after the ':' does not exceed the end of the string.
                    if( (pCharacter > pNameStart) && ((pCharacter + 1) <= pStringEnd) )
                    {
                        if( *(pCharacter + 1) == ' ' )
                        {
                            pNameEnd = pCharacter;
                            pValueStart = pCharacter + 2;
                        }
                    }
                }
            }
            break;
            case '\n':
            {
                if( (pNameStart != nullptr) && (pNameEnd != nullptr) && (pValueStart != nullptr) )
                {
                    const s32 NameLength = { static_cast<s32>(pNameEnd - pNameStart) };

                    // Compare the name against the "known token names".
                    bool bProcessed = { false };

                    // Is this the BCID?
                    if( !bProcessed && (NameLength + 1) == static_cast<s32>(sizeof(BCIDToken)) )
                    {
                        const s32 CompareResult = { strncmp( BCIDToken, pNameStart, NameLength ) };
                        if( CompareResult == 0 )
                        {
                            TmsInfo.BCID = atoi( pValueStart );
                            bProcessed = true;
                        }
                    }

                    // Is this the PS?
                    if( !bProcessed && (NameLength + 1) == static_cast<s32>(sizeof(PSToken)) )
                    {
                        const s32 CompareResult = { strncmp(PSToken, pNameStart, NameLength) };
                        if( CompareResult == 0 )
                        {
                            TmsInfo.WasSleeping = (atoi( pValueStart ) != 0) ? true : false;
                            bProcessed = true;
                        }
                    }
// Here is an example on how to add a new token to the "decoding process".
//                    // Is this a new token?
//                    if( !bProcessed && (NameLength + 1) == static_cast<s32>(sizeof(NewExampleToken)) )
//                    {
//                        const s32 CompareResult = { strncmp( NewExampleToken, pNameStart, NameLength ) };
//                        if( CompareResult == 0 )
//                        {
//                            if( (pCharacter - pValueStart) <= static_cast<s32>(sizeof(tmsInfo.NewStringValue)) )
//                            {
//                                const s32 ValueLength{ pCharacter - pValueStart };
//                                for( s32 Index = 0; Index < ValueLength; Index++ )
//                                {
//                                    tmsInfo.NewStringValue[Index] = pValueStart[Index];
//                                }
//                                tmsInfo.NewStringValue[FileNameLength] = '\0';
//                                bProcessed = true;
//                            }
//                        }
//                    }
                }

                // Reset the cursor to the next character.
                pNameStart  = pCharacter + 1;
                pNameEnd    = nullptr;
                pValueStart = nullptr;
            }
            break;
            default:
                break;
            }
        }
    }
}

//==============================================================================
} // namespace tma
//==============================================================================
