﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/timesrv/detail/fs/timesrv_Fs.h>
#include <nn/time/detail/config/time_ConfigDetail.h>
#include <nn/time/time_ResultPrivate.h>

#include <nn/result/result_HandlingUtility.h>
#include <nn/nn_Abort.h>
#include <nn/nn_SdkLog.h>

#if NN_DETAIL_TIME_CONFIG_TIMEZONE_BINARY_MODEL == NN_DETAIL_TIME_CONFIG_TIMEZONE_BINARY_MODEL_SYSTEM_DATA_TITLE
#include <nn/fs.h>
#include <nn/fs/fs_SystemData.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/util/util_FormatString.h>
namespace
{
    const nn::ncm::SystemDataId TimeZoneBinarySystemDataId = { 0x010000000000080E };
    const char* TimeZoneSystemDataMountName = "TimeZoneBinary";
}
#else
#include <siglo/Include/time_TimeZoneResources.h> // 利用はバイナリソースリンク版のみ
#endif


namespace nn { namespace timesrv { namespace detail { namespace fs {

    namespace
    {
        nn::Result g_MountResult = nn::time::ResultNotInitialized();
        nn::Result CheckFsMounted() NN_NOEXCEPT
        {
            return g_MountResult;
        }
    };

    using nn::time::LocationName;

    nn::Result Mount() NN_NOEXCEPT
    {
#if NN_DETAIL_TIME_CONFIG_TIMEZONE_BINARY_MODEL == NN_DETAIL_TIME_CONFIG_TIMEZONE_BINARY_MODEL_SYSTEM_DATA_TITLE
        size_t outCacheSize = 0;
        auto result = nn::fs::QueryMountSystemDataCacheSize(&outCacheSize, TimeZoneBinarySystemDataId);

        if(result.IsFailure())
        {
            NN_DETAIL_TIME_ERROR("nn::fs::QueryMountSystemDataCacheSize(%s) failed. (%08x, %03d-%04d)\n",
                TimeZoneSystemDataMountName,
                result.GetInnerValueForDebug(),
                result.GetModule(), result.GetDescription());
        }
        else
        {
            const size_t SystemDataCacheSize = 1024 * 40;
            NN_FUNCTION_LOCAL_STATIC(Bit8, s_SystemDataCache[SystemDataCacheSize], = {});

            NN_SDK_ASSERT_GREATER_EQUAL(SystemDataCacheSize, outCacheSize);

            result = nn::fs::MountSystemData(TimeZoneSystemDataMountName, TimeZoneBinarySystemDataId, s_SystemDataCache, SystemDataCacheSize);
            if(result.IsFailure())
            {
                NN_DETAIL_TIME_ERROR("nn::fs::MountSystemData(%s) failed. (%08x, %03d-%04d)\n",
                    TimeZoneSystemDataMountName,
                    result.GetInnerValueForDebug(),
                    result.GetModule(), result.GetDescription());
            }
        }

        g_MountResult = result;
        return result;
#else
        g_MountResult = nn::ResultSuccess();
        NN_RESULT_SUCCESS;
#endif
    }

    void GetFilePath(BinaryPath* pOut, const LocationName& name) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(pOut);

        if(CheckFsMounted().IsFailure())
        {
            return;
        }

#if NN_DETAIL_TIME_CONFIG_TIMEZONE_BINARY_MODEL == NN_DETAIL_TIME_CONFIG_TIMEZONE_BINARY_MODEL_SYSTEM_DATA_TITLE
        int ret = nn::util::SNPrintf(pOut->value, BinaryPath::Size, "%s:/zoneinfo/%s", TimeZoneSystemDataMountName, name._value);
        NN_SDK_ASSERT_LESS(ret, static_cast<int>(BinaryPath::Size));
        NN_UNUSED(ret);
#else
        nn::util::Strlcpy(pOut->value, name._value, sizeof(pOut->value)); // ソースリンクではそのまま出力
#endif
    }

    void GetFilePathOfTimeZoneListData(BinaryPath* pOut) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(pOut);

        if(CheckFsMounted().IsFailure())
        {
            return;
        }

#if NN_DETAIL_TIME_CONFIG_TIMEZONE_BINARY_MODEL == NN_DETAIL_TIME_CONFIG_TIMEZONE_BINARY_MODEL_SYSTEM_DATA_TITLE
        int ret = nn::util::SNPrintf(pOut->value, BinaryPath::Size, "%s:/binaryList.txt", TimeZoneSystemDataMountName);
        NN_SDK_ASSERT_LESS(ret, static_cast<int>(BinaryPath::Size));
        NN_UNUSED(ret);
#else
        std::memset(pOut->value, 0, BinaryPath::Size); // ソースリンクではそもそも存在しないファイルなので空
#endif
    }

    void GetFilePathOfTimeZoneVersion(BinaryPath* pOut) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(pOut);

#if NN_DETAIL_TIME_CONFIG_TIMEZONE_BINARY_MODEL == NN_DETAIL_TIME_CONFIG_TIMEZONE_BINARY_MODEL_SYSTEM_DATA_TITLE
        int ret = nn::util::SNPrintf(pOut->value, BinaryPath::Size, "%s:/version.txt", TimeZoneSystemDataMountName);
        NN_SDK_ASSERT_LESS(ret, static_cast<int>(BinaryPath::Size));
        NN_UNUSED(ret);
#else
        std::memset(pOut->value, 0, BinaryPath::Size); // ソースリンクではそもそも存在しないファイルなので空
#endif
    }

    nn::Result GetFileSize(size_t *pOut, const BinaryPath &path) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(pOut);

        NN_RESULT_DO(CheckFsMounted());

#if NN_DETAIL_TIME_CONFIG_TIMEZONE_BINARY_MODEL == NN_DETAIL_TIME_CONFIG_TIMEZONE_BINARY_MODEL_SYSTEM_DATA_TITLE
        nn::fs::FileHandle handle;
        NN_RESULT_DO(nn::fs::OpenFile(&handle, path.value, nn::fs::OpenMode_Read));
        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::CloseFile(handle);
        };

        int64_t fileSize;
        NN_RESULT_DO(nn::fs::GetFileSize(&fileSize, handle));

        *pOut = static_cast<size_t>(fileSize);
        NN_RESULT_SUCCESS;
#else
        nne::tz::BinaryResource* pResource = nne::tz::GetBinaryResourceListPtr();
        for(size_t i = 0 ; i < nne::tz::BinaryResourceCount ; ++i)
        {
            BinaryPath resourcePath;
            GetFilePath(&resourcePath, pResource[i].name);
            if(resourcePath == path)
            {
                *pOut = pResource[i].size;
                NN_RESULT_SUCCESS;
            }
        }

        NN_RESULT_THROW(nn::time::ResultNotFound());
#endif
    }

    nn::Result ReadFile(size_t* pOutSize, char* pOutBuffer, size_t size, const BinaryPath &path) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(pOutSize);
        NN_SDK_ASSERT_NOT_NULL(pOutBuffer);

        NN_RESULT_DO(CheckFsMounted());

#if NN_DETAIL_TIME_CONFIG_TIMEZONE_BINARY_MODEL == NN_DETAIL_TIME_CONFIG_TIMEZONE_BINARY_MODEL_SYSTEM_DATA_TITLE
        nn::fs::FileHandle handle;
        NN_RESULT_DO(nn::fs::OpenFile(&handle, path.value, nn::fs::OpenMode_Read));
        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::CloseFile(handle);
        };

        int64_t fileSize;
        NN_RESULT_DO(nn::fs::GetFileSize(&fileSize, handle));

        NN_SDK_ASSERT_GREATER_EQUAL(size, static_cast<size_t>(fileSize));
        NN_RESULT_THROW_UNLESS(size > static_cast<size_t>(fileSize), nn::time::ResultOutOfMemory());

        NN_RESULT_DO(nn::fs::ReadFile(handle, 0, pOutBuffer, fileSize));
        *pOutSize = fileSize;

        NN_RESULT_SUCCESS;
#else
        NN_UNUSED(size);
        nne::tz::BinaryResource* pResource = nne::tz::GetBinaryResourceListPtr();
        for(size_t i = 0 ; i < nne::tz::BinaryResourceCount ; ++i)
        {
            BinaryPath resourcePath;
            GetFilePath(&resourcePath, pResource[i].name);
            if(resourcePath == path)
            {
                NN_SDK_ASSERT_GREATER_EQUAL(size, pResource[i].size);

                std::memcpy(pOutBuffer, pResource[i].pData, pResource[i].size);
                *pOutSize = pResource[i].size;

                NN_RESULT_SUCCESS;
            }
        }

        NN_RESULT_THROW(nn::time::ResultNotFound());
#endif
    }

}}}} // nn::timesrv::detail::fs

