﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/timesrv/detail/core/timesrv_TimeZoneServiceCore.h>
#include <nn/timesrv/detail/tz/timesrv_TimeZoneDetail.h>
#include <nn/timesrv/detail/fs/timesrv_TimeZoneBinaryManager.h>
#include <nn/timesrv/detail/service/timesrv_ServiceProvider.h>

#include <nn/timesrv/timesrv_TimeZoneRuleInner.h>
#include <nn/time/time_ResultPrivate.h>

NN_STATIC_ASSERT(sizeof(nn::time::TimeZoneRule) >= sizeof(nn::timesrv::TimeZoneRuleInner));

namespace nn
{
namespace timesrv
{
namespace detail
{
namespace core
{

TimeZoneServiceCore::TimeZoneServiceCore(nn::timesrv::detail::service::PosixTimeNotifierToFs* pPosixTimeNotifierToFs) NN_NOEXCEPT:
    m_OwnLocationName(),
    m_OwnTimeZoneRule(),
    m_LocationUpdatedTime{0, nn::util::InvalidUuid},
    m_LocationNameCount(0),
    m_TimeZoneRuleVersion(),
    m_pPosixTimeNotifierToFs(pPosixTimeNotifierToFs)
{
}

void TimeZoneServiceCore::GetDeviceLocationName(nn::time::LocationName* pOutValue) const NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_Lock);

    *pOutValue = m_OwnLocationName;
}

Result TimeZoneServiceCore::SetDeviceLocationName(const nn::time::LocationName& value) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(fs::TimeZoneBinaryManager::IsValidLocationName(value), nn::time::ResultNotFound());

    NN_UTIL_LOCK_GUARD(m_Lock);

    auto result = LoadTimeZoneRule(&m_OwnTimeZoneRule, value);
    if(result.IsFailure())
    {
        LoadTimeZoneRule(&m_OwnTimeZoneRule, m_OwnLocationName); // 失敗時は元に戻す
        return result;
    }

    m_OwnLocationName = value;

    m_pPosixTimeNotifierToFs->Notify();

    NN_RESULT_SUCCESS;
}

void TimeZoneServiceCore::SetTotalLocationNameCount(int32_t count) NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_Lock);
    m_LocationNameCount = count;
}

int32_t TimeZoneServiceCore::GetTotalLocationNameCount() const NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_Lock);
    return m_LocationNameCount;
}

void TimeZoneServiceCore::SetDeviceLocationUpdatedTime(const nn::time::SteadyClockTimePoint& value)
{
    NN_UTIL_LOCK_GUARD(m_Lock);
    m_LocationUpdatedTime = value;
}

nn::time::SteadyClockTimePoint TimeZoneServiceCore::GetDeviceLocationUpdatedTime() const
{
    NN_UTIL_LOCK_GUARD(m_Lock);
    return m_LocationUpdatedTime;
}

Result TimeZoneServiceCore::LoadTimeZoneRule(
    nn::time::TimeZoneRule* pOut, const nn::time::LocationName& locationName) NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_Lock);

    char* pBinary;
    size_t binarySize;
    NN_RESULT_DO(fs::TimeZoneBinaryManager::ReadTimeZoneBinary(&pBinary, &binarySize, locationName));

    NN_RESULT_DO( timesrv::detail::tz::ParseTimeZoneBinary(GetRuleInner(pOut), pBinary, binarySize) );

    NN_RESULT_SUCCESS;
}

Result TimeZoneServiceCore::LoadLocationNameList(
    int32_t* pOutCount,
    nn::time::LocationName* pOutLocationNameList,
    size_t locationNameListCount,
    int32_t offset) NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_Lock);

    NN_RESULT_DO(fs::TimeZoneBinaryManager::LoadLocationNameList(
        pOutCount,
        pOutLocationNameList,
        locationNameListCount,
        offset));

    NN_RESULT_SUCCESS;
}

void TimeZoneServiceCore::SetTimeZoneRuleVersion(const nn::time::TimeZoneRuleVersion& version) NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_Lock);

    m_TimeZoneRuleVersion = version;
}

void TimeZoneServiceCore::GetTimeZoneRuleVersion(nn::time::TimeZoneRuleVersion* pOut) const NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_Lock);

    *pOut = m_TimeZoneRuleVersion;
}

Result TimeZoneServiceCore::ToCalendarTime(
    nn::time::CalendarTime* pOutCalendarTime,
    nn::time::sf::CalendarAdditionalInfo* pOutCalendarAdditionalInfo,
    const nn::time::PosixTime& posixTime,
    const nn::time::TimeZoneRule& rule) NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_Lock);

    NN_RESULT_DO( timesrv::detail::tz::ToCalendarTime(pOutCalendarTime, pOutCalendarAdditionalInfo, posixTime, GetRuleInnerRef(rule)) );
    NN_RESULT_SUCCESS;
}

Result TimeZoneServiceCore::ToCalendarTimeWithMyRule(
    nn::time::CalendarTime* pOutCalendarTime,
    nn::time::sf::CalendarAdditionalInfo* pOutCalendarAdditionalInfo,
    const nn::time::PosixTime& posixTime) NN_NOEXCEPT
{
    return ToCalendarTime(
            pOutCalendarTime,
            pOutCalendarAdditionalInfo,
            posixTime,
            m_OwnTimeZoneRule);
}

Result TimeZoneServiceCore::ToPosixTime(
    int32_t* pOutCount,
    nn::time::PosixTime* pOutPosixTimeList,
    size_t outPosixTimeListCount,
    const nn::time::CalendarTime& calendarTime,
    const nn::time::TimeZoneRule& rule) NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_Lock);

    NN_RESULT_TRY( timesrv::detail::tz::ToPosixTime(
        pOutCount,
        pOutPosixTimeList,
        outPosixTimeListCount,
        calendarTime,
        GetRuleInnerRef(rule),
        timesrv::detail::tz::DaylightSavingTimeType_Unknown // 夏時間有効性は必ず不明とする(公開APIからは指定できない).
        ))
        NN_RESULT_CATCH(nn::time::ResultNotFound)
        {
            *pOutCount = 0; // 成功扱いとする
        }
    NN_RESULT_END_TRY

    if(*pOutCount == 2)
    {
        NN_SDK_ASSERT_NOT_EQUAL(pOutPosixTimeList[0], pOutPosixTimeList[1]);

        if(pOutPosixTimeList[0] > pOutPosixTimeList[1])
        {
            // 事後条件 pOutPosixTimeList[0] < pOutPosixTimeList[1] を満たす
            std::swap(pOutPosixTimeList[0], pOutPosixTimeList[1]);
        }

        NN_SDK_ASSERT_LESS(pOutPosixTimeList[0], pOutPosixTimeList[1]);
    }

    NN_RESULT_SUCCESS;
}

Result TimeZoneServiceCore::ToPosixTimeWithMyRule(
    int32_t* pOutCount,
    nn::time::PosixTime* pOutPosixTimeList,
    size_t outPosixTimeListCount,
    const nn::time::CalendarTime& calendarTime) NN_NOEXCEPT
{
    return ToPosixTime(
       pOutCount,
       pOutPosixTimeList,
       outPosixTimeListCount,
       calendarTime,
       m_OwnTimeZoneRule);
}

}
}
}
}

