﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/timesrv/detail/core/timesrv_SystemClockCore.h>

#include <nn/timesrv/detail/core/timesrv_SteadyClockCore.h>

namespace nn
{
namespace timesrv
{
namespace detail
{
namespace core
{

using ::nn::time::PosixTime;
using ::nn::time::SteadyClockTimePoint;
using ::nn::time::SystemClockContext;


SystemClockCore::SystemClockCore(SteadyClockCore* pSteadyClockCore) NN_NOEXCEPT:
    m_pSystemClockContextUpdateCallback(nullptr),
    m_pSteadyClockCore(pSteadyClockCore),
    m_SystemClockContext{0, {0, nn::util::InvalidUuid}}
{
    NN_SDK_ASSERT_NOT_NULL(m_pSteadyClockCore);
}

SystemClockCore::~SystemClockCore() NN_NOEXCEPT
{
}

Result SystemClockCore::GetCurrentTime(PosixTime* pOutPosixTime) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pOutPosixTime);
    NN_RESULT_THROW_UNLESS(pOutPosixTime != nullptr, nn::time::ResultInvalidPointer());

    SteadyClockTimePoint steadyClockTimePoint;
    SystemClockContext systemClockContext;

    NN_RESULT_DO(GetSteadyClockCore().GetCurrentTimePoint(&steadyClockTimePoint));
    NN_RESULT_DO(GetSystemClockContext(&systemClockContext));

    return CalculatePosixTime(pOutPosixTime, steadyClockTimePoint, systemClockContext);
}

Result SystemClockCore::SetCurrentTime(const PosixTime& posixTime) NN_NOEXCEPT
{
    SteadyClockTimePoint steadyClockTimePoint;
    SystemClockContext systemClockContext;

    NN_RESULT_DO(GetSteadyClockCore().GetCurrentTimePoint(&steadyClockTimePoint));

    CalculateSystemClockContext(&systemClockContext, posixTime, steadyClockTimePoint);

    NN_RESULT_DO(SetSystemClockContext(systemClockContext));

    NN_RESULT_SUCCESS;
}

SteadyClockCore& SystemClockCore::GetSteadyClockCore() NN_NOEXCEPT
{
    return *m_pSteadyClockCore;
}

const SteadyClockCore& SystemClockCore::GetSteadyClockCore() const NN_NOEXCEPT
{
    return *m_pSteadyClockCore;
}

Result SystemClockCore::GetSystemClockContext(SystemClockContext* pOutSystemClockContext) const NN_NOEXCEPT
{
    return GetSystemClockContextImpl(pOutSystemClockContext);
}

Result SystemClockCore::SetSystemClockContext(const SystemClockContext& systemClockContext) NN_NOEXCEPT
{
    NN_RESULT_DO(SetSystemClockContextImpl(systemClockContext));

    if(m_pSystemClockContextUpdateCallback)
    {
        NN_RESULT_DO(m_pSystemClockContextUpdateCallback->Set(systemClockContext));
    }

    NN_RESULT_SUCCESS;
}

bool SystemClockCore::IsAvailable() const NN_NOEXCEPT
{
    // SystemClock補正時の sourceId と、SteadyClock の sourceId が一致していれば補正済であり、時刻が返せる

    SystemClockContext context;
    if(GetSystemClockContextImpl(&context).IsFailure())
    {
        return false;
    }

    nn::time::SteadyClockTimePoint timePoint;
    if(GetSteadyClockCore().GetCurrentTimePoint(&timePoint).IsFailure())
    {
        return false;
    }

    return context.timeStamp.sourceId == timePoint.sourceId;
}

void SystemClockCore::SetSystemClockContextUpdateCallback(SystemClockContextUpdateCallback* pCallback) NN_NOEXCEPT
{
    m_pSystemClockContextUpdateCallback = pCallback;
}

Result SystemClockCore::CalculatePosixTime(PosixTime* pOutPosixTime, const SteadyClockTimePoint& steadyClockTimePoint, const SystemClockContext& systemClockContext) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pOutPosixTime);
    NN_RESULT_THROW_UNLESS(pOutPosixTime != nullptr, nn::time::ResultInvalidPointer());

    NN_RESULT_THROW_UNLESS(steadyClockTimePoint.sourceId == systemClockContext.timeStamp.sourceId, nn::time::ResultOffsetInvalid());

    pOutPosixTime->value = steadyClockTimePoint.value + systemClockContext.offset;

    NN_RESULT_SUCCESS;
}

void SystemClockCore::CalculateSystemClockContext(SystemClockContext* pOutSystemClockContext, const PosixTime& posixTime, const SteadyClockTimePoint& steadyClockTimePoint) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pOutSystemClockContext);

    pOutSystemClockContext->offset = posixTime.value - steadyClockTimePoint.value;
    pOutSystemClockContext->timeStamp = steadyClockTimePoint;
}

Result SystemClockCore::GetSystemClockContextImpl(nn::time::SystemClockContext* pSystemClockContext) const NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pSystemClockContext);
    *pSystemClockContext = m_SystemClockContext;
    NN_RESULT_SUCCESS;
}
Result SystemClockCore::SetSystemClockContextImpl(const nn::time::SystemClockContext& systemClockContext) NN_NOEXCEPT
{
    m_SystemClockContext = systemClockContext;
    NN_RESULT_SUCCESS;
}

}
}
}
}

