﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/timesrv/detail/core/timesrv_StandardSteadyClockCore.h>

#include <nn/os.h>
#include <nn/util/util_UuidApi.h>

#if NN_DETAIL_TIME_CONFIG_EXTERNAL_STEADY_CLOCK_MODEL == NN_DETAIL_TIME_CONFIG_EXTERNAL_STEADY_CLOCK_MODEL_STD_CHRONO
#include <chrono>
#elif NN_DETAIL_TIME_CONFIG_EXTERNAL_STEADY_CLOCK_MODEL == NN_DETAIL_TIME_CONFIG_EXTERNAL_STEADY_CLOCK_MODEL_NX_PMIC
#include <nn/timesrv/detail/core/timesrv_PmicApi-hardware.nx.h>
#elif NN_DETAIL_TIME_CONFIG_EXTERNAL_STEADY_CLOCK_MODEL == NN_DETAIL_TIME_CONFIG_EXTERNAL_STEADY_CLOCK_MODEL_OS_TICK
// #include <nn/os/os_Tick.h>
#endif

#include <algorithm>

namespace nn
{
namespace timesrv
{
namespace detail
{
namespace core
{

using ::nn::time::SteadyClockTimePoint;

StandardSteadyClockCore::StandardSteadyClockCore() NN_NOEXCEPT:
    m_TestOffset(0),
    m_InternalOffset(0),
    m_SourceId(nn::util::InvalidUuid),
    m_BaseTime(0),
    m_LastReturnedClockValue(0)
{
}

StandardSteadyClockCore::~StandardSteadyClockCore() NN_NOEXCEPT
{
}

Result StandardSteadyClockCore::GetCurrentTimePointImpl(SteadyClockTimePoint* pSteadyClockTimePoint) const NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pSteadyClockTimePoint);

    pSteadyClockTimePoint->value = GetCurrentTimePointValueImpl().GetSeconds();
    pSteadyClockTimePoint->sourceId = m_SourceId;

    NN_RESULT_SUCCESS;
}

Result StandardSteadyClockCore::GetRtcSecondsImpl(int64_t* pOutValue) NN_NOEXCEPT
{
#if NN_DETAIL_TIME_CONFIG_EXTERNAL_STEADY_CLOCK_MODEL == NN_DETAIL_TIME_CONFIG_EXTERNAL_STEADY_CLOCK_MODEL_STD_CHRONO
    // Windows環境では system_clock::now を以下の理由により採用する
    // (詳細は http://spdlybra.nintendo.co.jp/jira/browse/SIGLO-19128, http://spdlybra.nintendo.co.jp/jira/browse/SIGLO-18824)
    //
    // 考えられる実装は以下の2つです。
    // 1.std::chrono::system_clock::now() を使う(VC++11.0,12.0(VisualStudio2012,2013)の steady_clock 内部実装は system_clock )
    //   PCを再起動して steady ではあるが、PC時計をいじると steady じゃなくなる. 自動補正で数秒はずれるかも.
    // 2.QueryPerformanceCounter を使う(VC++14.0(VisualStudio2015) の steady_clock 内部実装は QueryPerformanceCounter )
    //   PCを再起動すると steadyじゃなくなる、PC時計をいじっても steady.
    //
    // PC 時計をいじるよりは、PC 再起動のほうが頻繁に行われるし、
    // PC 再起動で必ず巻き戻る症状は致命的なので system_clock を採用します.
    // Initialize 後に PC 時計を操作しても、時刻ライブラリが返す時刻には影響しないので、自動補正がかかった瞬間に時刻ライブラリが返す時刻が巻き戻ることもないです.
    // なお、PC 時計をいじると時刻がずれることはWindows上の環境では仕様としています(time.h を参照)
    auto tp = std::chrono::system_clock::now();
    time_t t = std::chrono::system_clock::to_time_t(tp);
    *pOutValue = static_cast<int64_t>(t);
#elif NN_DETAIL_TIME_CONFIG_EXTERNAL_STEADY_CLOCK_MODEL == NN_DETAIL_TIME_CONFIG_EXTERNAL_STEADY_CLOCK_MODEL_NX_PMIC
    int64_t seconds;
    NN_RESULT_DO( RequestPmicSeconds(&seconds) );
    *pOutValue = seconds;
#elif NN_DETAIL_TIME_CONFIG_EXTERNAL_STEADY_CLOCK_MODEL == NN_DETAIL_TIME_CONFIG_EXTERNAL_STEADY_CLOCK_MODEL_OS_TICK
    // Jetson ボードなどは電源断でリセットされるけどチック値で対応
    nn::os::Tick tick = nn::os::GetSystemTick();
    *pOutValue = tick.ToTimeSpan().GetSeconds();
#else
#error "有効な NN_DETAIL_TIME_CONFIG_EXTERNAL_STEADY_CLOCK_MODEL が設定されていません"
#endif

    NN_RESULT_SUCCESS;
}

nn::TimeSpan StandardSteadyClockCore::GetCurrentTimePointValueImpl() const NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_Lock);

    // m_LastReturnedClockValue から巻き戻るのを防止する
    m_LastReturnedClockValue = std::max(m_BaseTime + nn::os::GetSystemTick().ToTimeSpan(), m_LastReturnedClockValue);

    return m_LastReturnedClockValue;
}

nn::TimeSpan StandardSteadyClockCore::GetTestOffsetImpl() const NN_NOEXCEPT
{
    return m_TestOffset;
}

void StandardSteadyClockCore::SetTestOffsetImpl(const nn::TimeSpan& testOffset) NN_NOEXCEPT
{
    m_TestOffset = testOffset;
}

nn::TimeSpan StandardSteadyClockCore::GetInternalOffsetImpl() const NN_NOEXCEPT
{
    return m_InternalOffset;
}

void StandardSteadyClockCore::SetInternalOffsetImpl(const nn::TimeSpan& internalOffset) NN_NOEXCEPT
{
    m_InternalOffset = internalOffset;
}

void StandardSteadyClockCore::SetupSourceId(nn::time::SourceId* pOutLatestClockSourceId, const nn::time::SourceId& clockSourceId) NN_NOEXCEPT
{
    m_SetupResult = TryAdjustBaseTimeWithRtc(20);

    if (m_SetupResult.IsFailure())
    {
        // セットアップできなかったら強制的に 0 からカウント、 SourceId も作り直し
        this->InitializeBaseTime(0);
        m_SourceId = nn::util::GenerateUuid();
    }
    else if(clockSourceId == nn::util::InvalidUuid)
    {
        // InvalidUuid は初期状態なので SourceId 生成
        m_SourceId = nn::util::GenerateUuid();
    }
    else
    {
        m_SourceId = clockSourceId;
    }

    if(pOutLatestClockSourceId)
    {
        *pOutLatestClockSourceId = m_SourceId;
    }
}

uint32_t StandardSteadyClockCore::GetSetupResultValue() const NN_NOEXCEPT
{
    return m_SetupResult.GetInnerValueForDebug();
}

void StandardSteadyClockCore::InitializeBaseTime(int64_t rtcSeconds) NN_NOEXCEPT
{
    m_BaseTime = nn::TimeSpan::FromSeconds(rtcSeconds) - nn::os::GetSystemTick().ToTimeSpan();
}

void StandardSteadyClockCore::UpdateBaseTimeWithRtc() NN_NOEXCEPT
{
    if (m_SetupResult.IsFailure())
    {
        // セットアップ失敗の場合、ネットワーク時計などが大きくずれる可能性があるので何もしない
        NN_DETAIL_TIME_WARN("Skip StandardSteadyClockCore::UpdateBaseTimeWithRtc. SetupResult(0x%08x)\n", m_SetupResult.GetInnerValueForDebug());
        return;
    }

    auto result = TryAdjustBaseTimeWithRtc(3);
    NN_UNUSED(result);
}

Result StandardSteadyClockCore::TryAdjustBaseTimeWithRtc(int tryCount) NN_NOEXCEPT
{
    NN_SDK_ASSERT_GREATER(tryCount, 0);
    Result result = nn::ResultSuccess();
    for (int i = 0; i < tryCount; ++i)
    {
        result = TryAdjustBaseTimeWithRtcImpl();
        if (result.IsSuccess())
        {
            NN_RESULT_SUCCESS;
        }
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(1)); // busy loop 回避
    }
    NN_DETAIL_TIME_WARN("Faild to StandardSteadyClockCore::TryUpdateBaseTimeWithRtc. (0x%08x)\n", result.GetInnerValueForDebug());
    NN_RESULT_THROW(result);
}

Result StandardSteadyClockCore::TryAdjustBaseTimeWithRtcImpl() NN_NOEXCEPT
{
    const auto elapseBegin = nn::os::GetSystemTick();

    // RTC取得
    int64_t rtcSeconds;
    auto result = this->GetRtcSeconds(&rtcSeconds);
    NN_DETAIL_TIME_ASSERT_IF_RESULT_FAILURE(result);
    NN_RESULT_DO(result);

    // Tick取得
    nn::os::Tick tick = nn::os::GetSystemTick();

    // RTC取得とTick取得の間にスレッド切り替えやスリープが挟まるとベース計算に大きな誤差が生じるので、
    // 想定外の時間経過を観測した場合は除外する
    const auto elapsedMilliSeconds = (tick - elapseBegin).ToTimeSpan().GetMilliSeconds();
    if (elapsedMilliSeconds > 100)
    {
        NN_DETAIL_TIME_INFO("Skip StandardSteadyClockCore::UpdateBaseTimeWithRtc. elapsed %ld [ms]\n", elapsedMilliSeconds);
        NN_RESULT_THROW(nn::time::ResultTimeout());
    }

    nn::TimeSpan newValue = nn::TimeSpan::FromSeconds(rtcSeconds) - tick.ToTimeSpan();
    //NN_DETAIL_TIME_INFO("TryAdjustBaseTimeWithRtcImpl rtc:%lld[sec] m_BaseTime:%lld[ms]->%lld[ms]\n",
    //    rtcSeconds, m_BaseTime.load().GetMilliSeconds(), newValue.GetMilliSeconds());

    NN_UTIL_LOCK_GUARD(m_Lock);
    m_BaseTime = newValue;
    NN_RESULT_SUCCESS;
}

}
}
}
}

