﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/timesrv/detail/core/timesrv_StandardLocalSystemClockCore.h>
#include <nn/timesrv/detail/core/timesrv_SteadyClockCore.h>
#include <nn/timesrv/detail/settings/timesrv_ClockSettings.h>
#include <nn/time/detail/util/time_UtilApi.h>

#if NN_DETAIL_TIME_CONFIG_EXTERNAL_STEADY_CLOCK_MODEL == NN_DETAIL_TIME_CONFIG_EXTERNAL_STEADY_CLOCK_MODEL_STD_CHRONO
#include <chrono>
#include <algorithm>
#endif

namespace nn
{
namespace timesrv
{
namespace detail
{
namespace core
{

// ユーザ時計にセットできる最小値、最大値
const nn::time::PosixTime StandardLocalSystemClockCore::SettablePosixTimeMin = { 946684800LL }; // 2000/01/01 00:00:00
const nn::time::PosixTime StandardLocalSystemClockCore::SettablePosixTimeMax = { 2871763199LL }; // 2060/12/31 23:59:59

namespace
{
    using ::nn::time::SystemClockContext;
}

StandardLocalSystemClockCore::StandardLocalSystemClockCore(SteadyClockCore* pSteadyClockCore) NN_NOEXCEPT
    : SystemClockCore(pSteadyClockCore)
{
}

StandardLocalSystemClockCore::~StandardLocalSystemClockCore() NN_NOEXCEPT
{
}

void StandardLocalSystemClockCore::Initialize(const SystemClockContext& systemClockContext) NN_NOEXCEPT
{
    nn::time::SteadyClockTimePoint steadyClockTimePoint;
    Result result = m_pSteadyClockCore->GetCurrentTimePoint(&steadyClockTimePoint);

    // UserSystemClock はどんなときでも有効な値を返す仕様なので、
    // GetCurrentTimePoint の失敗時、SteadyClockTimePoint の sourceId が食い違ったら時計をデフォルト値でセットしなおす.
    //
    // sourceId が食い違ったら
    // - UserSystemClock の初回起動
    // - NAND破壊
    // - RTC がリセットされた可能性(電池切れで起こり得る)
    // があり得る.
    if( result.IsFailure() || systemClockContext.timeStamp.sourceId != steadyClockTimePoint.sourceId)
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS( this->SetCurrentTime( StandardLocalSystemClockCore::GetInitialPosixTime() ) );// 失敗しない想定

        // SetCurrentTime() で SetSystemClockContextImpl() が呼ばれ
        // m_SystemClockContext.timeStamp.sourceId は steadyClockSourceId と同等の値に上書きされる
        NN_SDK_ASSERT_NOT_EQUAL(m_SystemClockContext.timeStamp.sourceId, nn::util::InvalidUuid);
        NN_SDK_ASSERT_EQUAL(m_SystemClockContext.timeStamp.sourceId, steadyClockTimePoint.sourceId);
    }
    else
    {
        this->SetSystemClockContext(systemClockContext);
        NN_SDK_ASSERT_EQUAL(m_SystemClockContext, systemClockContext);
    }
}

// StandardLocalSystemClockCore が初期状態でカウントし始める起点の取得
nn::time::PosixTime StandardLocalSystemClockCore::GetInitialPosixTime() NN_NOEXCEPT
{
#if NN_DETAIL_TIME_CONFIG_EXTERNAL_STEADY_CLOCK_MODEL == NN_DETAIL_TIME_CONFIG_EXTERNAL_STEADY_CLOCK_MODEL_STD_CHRONO
    // (Win環境など) std::chrono が SteadyClock として代用される場合は、
    // 便宜上ユーザ時計を PC の現在時刻で初期化する
    auto tp = std::chrono::system_clock::now();
    nn::time::PosixTime posixTime = { static_cast<int64_t>( std::chrono::system_clock::to_time_t(tp) ) };

    // 最小値～最大値の範囲に丸めこみ
    posixTime = std::max(SettablePosixTimeMin, posixTime);
    posixTime = std::min(SettablePosixTimeMax, posixTime);
    return posixTime;
#else
    // N年1月1日 00:00:00 (UTC) を初期値とする
    const nn::time::CalendarTime InitialCalendar =
    {
        static_cast<int16_t>(settings::ReadStandardUserSystemClockInitialYear()),
        1,
        1,
        0,
        0,
        0
    };
    return nn::time::detail::util::ToPosixTimeFromUtc(InitialCalendar);
#endif
}

}
}
}
}

