﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/timesrv/detail/core/timesrv_SharedMemoryManager.h>
#include <mutex>

namespace nn { namespace timesrv { namespace detail { namespace core {

SharedMemoryManager::SharedMemoryManager() NN_NOEXCEPT
    : m_SharedMemory(nn::time::detail::SharedMemoryMappingSize, nn::os::MemoryPermission_ReadWrite, nn::os::MemoryPermission_ReadOnly)
{
    void* p = m_SharedMemory.Map(nn::os::MemoryPermission_ReadWrite);
    NN_ABORT_UNLESS_NOT_NULL(p);
    m_pSharedMemoryClockSettings = static_cast<nn::time::detail::SharedMemoryClockSettings*>(p);

    // 初期値の設定

    {
        nn::time::detail::SharedMemoryClockSettings::StandardSteadyClockTimePointType initialData =
        {
            nn::TimeSpan(0),
            nn::util::InvalidUuid
        };
        nn::util::StoreToLockFreeAtomicType(&m_pSharedMemoryClockSettings->standardSteadyClockTimePoint, initialData);
    }

    {
        nn::time::SystemClockContext initialContext = {
            0LL,
            {0LL, nn::util::InvalidUuid}
        };
        nn::util::StoreToLockFreeAtomicType(&m_pSharedMemoryClockSettings->standardLocalSystemClockContext, initialContext);
        nn::util::StoreToLockFreeAtomicType(&m_pSharedMemoryClockSettings->standardNetworkSystemClockContext, initialContext);
    }

    nn::util::StoreToLockFreeAtomicType(&m_pSharedMemoryClockSettings->isStandardUserSystemClockAutomaticCorrectionEnabled, false);
}

void SharedMemoryManager::SetStandardSteadyClockTimePoint(const nn::time::SourceId& sourceId, const nn::TimeSpan& value) NN_NOEXCEPT
{
    nn::time::detail::SharedMemoryClockSettings::StandardSteadyClockTimePointType data =
    {
        // 最新のTick値を足しこめば現在時刻になる値を BaseTime とする
        value - nn::os::GetSystemTick().ToTimeSpan(),
        sourceId
    };

    nn::util::StoreToLockFreeAtomicType(&m_pSharedMemoryClockSettings->standardSteadyClockTimePoint, data);
}

void SharedMemoryManager::SetStandardSteadyClockTimePointValue(const nn::TimeSpan& value) NN_NOEXCEPT
{
    nn::time::detail::SharedMemoryClockSettings::StandardSteadyClockTimePointType data =
        nn::util::LoadFromLockFreeAtomicType(&m_pSharedMemoryClockSettings->standardSteadyClockTimePoint);

    // 最新のTick値を足しこめば現在時刻になる値を BaseTime とする
    data.baseTime = value - nn::os::GetSystemTick().ToTimeSpan();

    nn::util::StoreToLockFreeAtomicType(&m_pSharedMemoryClockSettings->standardSteadyClockTimePoint, data);
}

void SharedMemoryManager::SetStandardLocalSystemClockContext(const nn::time::SystemClockContext& value) NN_NOEXCEPT
{
    nn::util::StoreToLockFreeAtomicType(&m_pSharedMemoryClockSettings->standardLocalSystemClockContext, value);
}

void SharedMemoryManager::SetStandardNetworkSystemClockContext(const nn::time::SystemClockContext& value) NN_NOEXCEPT
{
    nn::util::StoreToLockFreeAtomicType(&m_pSharedMemoryClockSettings->standardNetworkSystemClockContext, value);
}

void SharedMemoryManager::SetStandardUserSystemClockAutomaticCorrectionEnabled(bool value) NN_NOEXCEPT
{
    nn::util::StoreToLockFreeAtomicType(&m_pSharedMemoryClockSettings->isStandardUserSystemClockAutomaticCorrectionEnabled, value);
}

}}}}
