﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/time/time_StandardSteadyClock.h>

#include <nn/time/detail/time_CommonDetail.h>
#include <nn/time/detail/time_SharedMemoryClient.h>
#include <nn/timesrv/detail/service/timesrv_ISteadyClock.sfdl.h>
#include <nn/timesrv/detail/service/timesrv_IStaticService.sfdl.h>

#include <nn/util/util_Optional.h>

namespace nn
{
namespace time
{

extern nn::sf::SharedPointer<nn::timesrv::detail::service::IStaticService> g_pStaticService;
extern nn::util::optional<nn::time::detail::SharedMemoryClient> g_SharedMemoryClient;
extern Result GetStandardSteadyClockService(
    nn::sf::SharedPointer<nn::timesrv::detail::service::ISteadyClock>* pOut) NN_NOEXCEPT;

Result StandardSteadyClock::GetCurrentTimePoint(SteadyClockTimePoint* pOutSteadyClockTimePoint) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(g_SharedMemoryClient == true, ResultNotInitialized());
    NN_SDK_ASSERT_NOT_NULL(pOutSteadyClockTimePoint);

    g_SharedMemoryClient->GetStandardSteadyClockCurrentTimePoint(pOutSteadyClockTimePoint);

    NN_RESULT_SUCCESS;
}

StandardSteadyClock::time_point StandardSteadyClock::now() NN_NOEXCEPT
{
    SteadyClockTimePoint steadyClocktimePoint = {0, nn::util::InvalidUuid};

    Result result = StandardSteadyClock::GetCurrentTimePoint(&steadyClocktimePoint);
    if(result.IsFailure())
    {
        // ResultNotInitialized だけは返る可能性があるのでそのときは
        // 無効なソースIDと 0 を返す
        steadyClocktimePoint.value = 0;
        steadyClocktimePoint.sourceId = nn::util::InvalidUuid;
    }

    return StandardSteadyClock::time_point(StandardSteadyClock::duration(steadyClocktimePoint.value));
}

// 以下、システム向けや DevMenu, DevMenuCommand 向け API

int64_t GetStandardSteadyClockRtcValue() NN_NOEXCEPT
{
    nn::sf::SharedPointer<nn::timesrv::detail::service::ISteadyClock> pClock;
    NN_ABORT_UNLESS_RESULT_SUCCESS(GetStandardSteadyClockService(&pClock));

    int64_t rtcValue;
    NN_ABORT_UNLESS_RESULT_SUCCESS(pClock->GetRtcValue(&rtcValue));
    return rtcValue;
}

bool IsRtcResetDetected() NN_NOEXCEPT
{
    nn::sf::SharedPointer<nn::timesrv::detail::service::ISteadyClock> pClock;
    NN_ABORT_UNLESS_RESULT_SUCCESS(GetStandardSteadyClockService(&pClock));

    bool isDetected;
    NN_ABORT_UNLESS_RESULT_SUCCESS(pClock->IsRtcResetDetected(&isDetected));
    return isDetected;
}

uint32_t GetStandardSteadyClockSetupResultValue() NN_NOEXCEPT
{
    nn::sf::SharedPointer<nn::timesrv::detail::service::ISteadyClock> pClock;
    NN_ABORT_UNLESS_RESULT_SUCCESS(GetStandardSteadyClockService(&pClock));

    uint32_t resultValue;
    NN_ABORT_UNLESS_RESULT_SUCCESS(pClock->GetSetupResutltValue(&resultValue));
    return resultValue;
}

nn::TimeSpan GetStandardSteadyClockInternalOffset() NN_NOEXCEPT
{
    nn::sf::SharedPointer<nn::timesrv::detail::service::ISteadyClock> pClock;
    NN_ABORT_UNLESS_RESULT_SUCCESS(GetStandardSteadyClockService(&pClock));

    nn::TimeSpanType offset;
    NN_ABORT_UNLESS_RESULT_SUCCESS(pClock->GetInternalOffset(&offset));
    return offset;
}

nn::TimeSpan GetStandardSteadyClockTestOffset() NN_NOEXCEPT
{
    nn::sf::SharedPointer<nn::timesrv::detail::service::ISteadyClock> pClock;
    NN_ABORT_UNLESS_RESULT_SUCCESS(GetStandardSteadyClockService(&pClock));

    nn::TimeSpanType offset;
    NN_ABORT_UNLESS_RESULT_SUCCESS(pClock->GetTestOffset(&offset));
    return offset;
}

void SetStandardSteadyClockInternalOffset(const nn::TimeSpan internalOffset) NN_NOEXCEPT
{
    NN_ABORT_UNLESS(g_pStaticService != nullptr, "[TIME] The time library is not initialized.");
    NN_ABORT_UNLESS_RESULT_SUCCESS(g_pStaticService->SetStandardSteadyClockInternalOffset(internalOffset));
}

}
}

