﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/time/time_EphemeralNetworkSystemClock.h>
#include <nn/time/detail/time_CommonDetail.h>
#include <nn/timesrv/detail/service/timesrv_ISystemClock.sfdl.h>

namespace nn
{
namespace time
{

extern Result GetEphemeralNetworkSystemClockService(
    nn::sf::SharedPointer<nn::timesrv::detail::service::ISystemClock>* pOut) NN_NOEXCEPT;

Result EphemeralNetworkSystemClock::GetCurrentTime(PosixTime* pOutPosixTime) NN_NOEXCEPT
{
    nn::sf::SharedPointer<nn::timesrv::detail::service::ISystemClock> pClock;
    NN_RESULT_DO(GetEphemeralNetworkSystemClockService(&pClock));

    NN_RESULT_DO(pClock->GetCurrentTime(pOutPosixTime));

    NN_RESULT_SUCCESS;
}

// テスト用
Result GetEphemeralNetworkSystemClockContext(SystemClockContext* pOutSystemClockContext) NN_NOEXCEPT
{
    nn::sf::SharedPointer<nn::timesrv::detail::service::ISystemClock> pClock;
    NN_RESULT_DO(GetEphemeralNetworkSystemClockService(&pClock));

    NN_RESULT_DO(pClock->GetSystemClockContext(pOutSystemClockContext));
    NN_RESULT_SUCCESS;
}

Result SetEphemeralNetworkSystemClockCurrentTime(const PosixTime& posixTime) NN_NOEXCEPT
{
    nn::sf::SharedPointer<nn::timesrv::detail::service::ISystemClock> pClock;
    NN_RESULT_DO(GetEphemeralNetworkSystemClockService(&pClock));

    NN_RESULT_DO(pClock->SetCurrentTime(posixTime));
    NN_RESULT_SUCCESS;
}

// テスト用
Result SetEphemeralNetworkSystemClockContext(const SystemClockContext& clockContext) NN_NOEXCEPT
{
    nn::sf::SharedPointer<nn::timesrv::detail::service::ISystemClock> pClock;
    NN_RESULT_DO(GetEphemeralNetworkSystemClockService(&pClock));

    NN_RESULT_DO(pClock->SetSystemClockContext(clockContext));
    NN_RESULT_SUCCESS;
}

EphemeralNetworkSystemClock::time_point EphemeralNetworkSystemClock::now() NN_NOEXCEPT
{
    PosixTime posixTime = {0};

    Result result = GetCurrentTime(&posixTime);
    if(result.IsFailure())
    {
        // 補正していなければエラーが返ってくる
        // 明らかに過去になる0を返しておく
        posixTime.value = 0;
    }

    return time_point(duration(posixTime.value));
}

std::time_t EphemeralNetworkSystemClock::to_time_t(const EphemeralNetworkSystemClock::time_point& t) NN_NOEXCEPT
{
    // time_point::time_since_epoch() については、
    // NX向けビルド(clang)、Windowsビルド(VC++)にて time_point 生成時にセットする duration を
    // そのまま返す実装になっている.
    // time_since_epoch() で内部のduration に何らかの計算をした結果が返ってくる場合には、
    // 実装分岐する必要があり、その場合は from_time_t でもその逆の変換を行う必要が出てくる.
    return static_cast<std::time_t>( std::chrono::duration_cast<std::chrono::seconds>( t.time_since_epoch() ).count() );
}

EphemeralNetworkSystemClock::time_point EphemeralNetworkSystemClock::from_time_t(std::time_t t) NN_NOEXCEPT
{
    return time_point(duration(t));
}


}
}

