﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/time/time_ClockSnapshot.h>
#include <nn/time/time_AdjustableUserSystemClock.h>
#include <nn/time/time_AdjustableNetworkSystemClock.h>
#include <nn/time/detail/time_ClockSnapshotPrivateApi.h>
#include <nn/timesrv/detail/service/timesrv_IStaticService.sfdl.h>

#if ( defined(NN_BUILD_CONFIG_COMPILER_CLANG) || defined(NN_BUILD_CONFIG_COMPILER_VC) ) // gcc に std::is_trivially_copyable がないので回避
NN_STATIC_ASSERT(std::is_trivially_copyable<nn::time::ClockSnapshot>::value);
#endif
NN_STATIC_ASSERT(std::is_standard_layout<nn::time::ClockSnapshot>::value);
NN_STATIC_ASSERT(NN_ALIGNOF(nn::time::ClockSnapshot) == NN_ALIGNOF(nn::time::sf::ClockSnapshot));
NN_STATIC_ASSERT(sizeof(nn::time::ClockSnapshot) == sizeof(nn::time::sf::ClockSnapshot));
NN_STATIC_ASSERT(sizeof(nn::time::detail::ClockSnapshotInitialType) == sizeof(uint8_t));

namespace nn { namespace time {

extern nn::sf::SharedPointer<nn::timesrv::detail::service::IStaticService> g_pStaticService;

namespace
{
    // enum を排除した nn::time::sf::CalendarAdditionalInfo を enum 利用の nn::time::CalendarAdditionalInfo へ変換
    nn::time::CalendarAdditionalInfo GetNnTimeCalendarAdditionalInfo(const nn::time::sf::CalendarAdditionalInfo& src) NN_NOEXCEPT
    {
        nn::time::CalendarAdditionalInfo dst =
        {
            static_cast<nn::time::DayOfWeek>(src.dayOfWeek),
            src.yearDay,
            src.timeZone
        };
        return dst;
    }
}

ClockSnapshot::ClockSnapshot() NN_NOEXCEPT
{
    std::memset(&m_Storage, 0, sizeof(m_Storage));

    auto p = nn::time::detail::GetSfClockSnapshotPtr(this);

    nn::time::CalendarTime InitialCalendarTime;
    std::memset(&InitialCalendarTime, 0, sizeof(InitialCalendarTime));
    InitialCalendarTime.year = 1970;
    InitialCalendarTime.month = 1;
    InitialCalendarTime.day = 1;

    nn::time::LocationName InitialLocationName;
    std::memset(&InitialLocationName, 0, sizeof(InitialLocationName));
    InitialLocationName._value[0] = 'U';
    InitialLocationName._value[1] = 'T';
    InitialLocationName._value[2] = 'C';

    nn::time::TimeZone InitialTimeZone;
    std::memset(&InitialTimeZone, 0, sizeof(InitialTimeZone));
    InitialTimeZone.standardTimeName[0] = 'U';
    InitialTimeZone.standardTimeName[1] = 'T';
    InitialTimeZone.standardTimeName[2] = 'C';
    InitialTimeZone.isDaylightSavingTime = false;
    InitialTimeZone.utcOffsetSeconds = 0;

    nn::time::sf::CalendarAdditionalInfo InitialCalendarAdditionalInfo;
    std::memset(&InitialCalendarAdditionalInfo, 0 , sizeof(InitialCalendarAdditionalInfo));
    InitialCalendarAdditionalInfo.dayOfWeek = static_cast<int8_t>( nn::time::DayOfWeek_Thursday );
    InitialCalendarAdditionalInfo.yearDay = 0;
    InitialCalendarAdditionalInfo.timeZone = InitialTimeZone;

    p->userCalendarTime = InitialCalendarTime;
    p->netCalendarTime  = InitialCalendarTime;
    p->userCalendarAdditionalInfo = InitialCalendarAdditionalInfo;
    p->netCalendarAdditionalInfo  = InitialCalendarAdditionalInfo;
    p->locationName = InitialLocationName;

    p->initialType = static_cast<uint8_t>(nn::time::detail::ClockSnapshotInitialType::Constructor);
}


void ClockSnapshot::CreateWithStandardSystemClock(ClockSnapshot* pOut) NN_NOEXCEPT
{
    NN_ABORT_UNLESS(g_pStaticService != nullptr, "[TIME] The time library is not initialized.");
    NN_SDK_REQUIRES_NOT_NULL(pOut);

    NN_ABORT_UNLESS_RESULT_SUCCESS(g_pStaticService->GetClockSnapshot(
        nn::time::detail::GetSfClockSnapshotPtr(pOut),
        static_cast<uint8_t>(nn::time::detail::ClockSnapshotInitialType::WithStandardSystemClock)));
}

void ClockSnapshot::CreateWithAdjustableSystemClock(ClockSnapshot* pOut) NN_NOEXCEPT
{
    NN_ABORT_UNLESS(g_pStaticService != nullptr, "[TIME] The time library is not initialized.");
    NN_SDK_REQUIRES_NOT_NULL(pOut);

    nn::time::SystemClockContext userSystemClockContext, netSystemClockContext;
    nn::time::AdjustableUserSystemClock::GetSystemClockContext(&userSystemClockContext);
    nn::time::AdjustableNetworkSystemClock::GetSystemClockContext(&netSystemClockContext);

    NN_ABORT_UNLESS_RESULT_SUCCESS(g_pStaticService->GetClockSnapshotFromSystemClockContext(
        nn::time::detail::GetSfClockSnapshotPtr(pOut),
        userSystemClockContext,
        netSystemClockContext,
        static_cast<uint8_t>(nn::time::detail::ClockSnapshotInitialType::WithAdjustableSystemClock)));
}

nn::time::PosixTime ClockSnapshot::GetStandardUserSystemClockPosixTime() const NN_NOEXCEPT
{
    return nn::time::detail::GetSfClockSnapshotPtr(this)->userSystemClockPosixTime;
}

nn::time::SystemClockContext ClockSnapshot::GetStandardUserSystemClockContext() const NN_NOEXCEPT
{
    return nn::time::detail::GetSfClockSnapshotPtr(this)->userSystemClockContext;
}

nn::time::CalendarTime ClockSnapshot::GetStandardUserSystemClockCalendarTime() const NN_NOEXCEPT
{
    return nn::time::detail::GetSfClockSnapshotPtr(this)->userCalendarTime;
}

nn::time::CalendarAdditionalInfo ClockSnapshot::GetStandardUserSystemClockCalendarAdditionalInfo() const NN_NOEXCEPT
{
    auto info = nn::time::detail::GetSfClockSnapshotPtr(this)->userCalendarAdditionalInfo;
    return GetNnTimeCalendarAdditionalInfo(info);
}

nn::time::PosixTime ClockSnapshot::GetStandardNetworkSystemClockPosixTime() const NN_NOEXCEPT
{
    return nn::time::detail::GetSfClockSnapshotPtr(this)->netSystemClockPosixTime;
}

nn::time::SystemClockContext ClockSnapshot::GetStandardNetworkSystemClockContext() const NN_NOEXCEPT
{
    return nn::time::detail::GetSfClockSnapshotPtr(this)->netSystemClockContext;
}

nn::time::CalendarTime ClockSnapshot::GetStandardNetworkSystemClockCalendarTime() const NN_NOEXCEPT
{
    return nn::time::detail::GetSfClockSnapshotPtr(this)->netCalendarTime;
}

nn::time::CalendarAdditionalInfo ClockSnapshot::GetStandardNetworkSystemClockCalendarAdditionalInfo() const NN_NOEXCEPT
{
    auto info = nn::time::detail::GetSfClockSnapshotPtr(this)->netCalendarAdditionalInfo;
    return GetNnTimeCalendarAdditionalInfo(info);
}

nn::time::SteadyClockTimePoint ClockSnapshot::GetStandardSteadyClockTimePoint() const NN_NOEXCEPT
{
    return nn::time::detail::GetSfClockSnapshotPtr(this)->steadyClockTimePoint;
}

nn::time::LocationName ClockSnapshot::GetLocationName() const NN_NOEXCEPT
{
    return nn::time::detail::GetSfClockSnapshotPtr(this)->locationName;
}

bool operator == (const ClockSnapshot& lhs, const ClockSnapshot& rhs) NN_NOEXCEPT
{
    const auto pLhs = nn::time::detail::GetSfClockSnapshotPtr(&lhs);
    const auto pRhs = nn::time::detail::GetSfClockSnapshotPtr(&rhs);

    return true
        // SystemClockContext
        && pLhs->userSystemClockContext ==         pRhs->userSystemClockContext
        && pLhs->netSystemClockContext ==          pRhs->netSystemClockContext
        // PosixTime
        && pLhs->userSystemClockPosixTime ==       pRhs->userSystemClockPosixTime
        && pLhs->netSystemClockPosixTime ==        pRhs->netSystemClockPosixTime
        // CalendarTime
        && pLhs->userCalendarTime ==               pRhs->userCalendarTime
        && pLhs->netCalendarTime ==                pRhs->netCalendarTime
        // CalendarAdditionalInfo
        && pLhs->userCalendarAdditionalInfo.dayOfWeek ==   pRhs->userCalendarAdditionalInfo.dayOfWeek
        && pLhs->userCalendarAdditionalInfo.yearDay ==     pRhs->userCalendarAdditionalInfo.yearDay
        && pLhs->userCalendarAdditionalInfo.timeZone ==    pRhs->userCalendarAdditionalInfo.timeZone
        && pLhs->netCalendarAdditionalInfo.dayOfWeek ==    pRhs->netCalendarAdditionalInfo.dayOfWeek
        && pLhs->netCalendarAdditionalInfo.yearDay ==      pRhs->netCalendarAdditionalInfo.yearDay
        && pLhs->netCalendarAdditionalInfo.timeZone ==     pRhs->netCalendarAdditionalInfo.timeZone
        // SteadyClockTimePoint
        && pLhs->steadyClockTimePoint ==           pRhs->steadyClockTimePoint
        // Other
        && pLhs->locationName ==                   pRhs->locationName
        && pLhs->isAutomaticCorrectionEnabled ==   pRhs->isAutomaticCorrectionEnabled
        && pLhs->initialType ==                    pRhs->initialType
        && pLhs->version ==                        pRhs->version
        && pLhs->padding[0] ==                     pRhs->padding[0];
    ;
}

}}

