﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/time/time_CalendarTime.h>
#include <nn/time/time_Api.h>
#include <nn/time/time_TimeZoneApi.h>

namespace nn { namespace time {

namespace
{
    uint64_t ToUint64(const nn::time::CalendarTime& value) NN_NOEXCEPT
    {
        return 0ull
            | static_cast<uint64_t>(value.year)     << 40 // 1以上: int16_t
            | static_cast<uint64_t>(value.month)    << 32 // 1-12: int8_t
            | static_cast<uint64_t>(value.day)      << 24 // 1-31: int8_t
            | static_cast<uint64_t>(value.hour)     << 16 // 0-23: int8_t
            | static_cast<uint64_t>(value.minute)   << 8  // 1-59: int8_t
            | static_cast<uint64_t>(value.second)   << 0  // 1-59: int8_t
        ;
    }
}

bool CalendarTime::IsValid() const NN_NOEXCEPT
{
    return true
        && nn::time::IsValidDate(static_cast<int>(year), static_cast<int>(month), static_cast<int>(day))
        && (0 <= hour && hour < 24)
        && (0 <= minute && minute < 60)
        && (0 <= second && second < 60)
        ;
}

CalendarTime& CalendarTime::operator+=(const nn::TimeSpan& timeSpan) NN_NOEXCEPT
{
    const auto posix = ToPosixTimeFromUtc(*this) + timeSpan;
    *this = ToCalendarTimeInUtc(posix);
    return *this;
}

CalendarTime& CalendarTime::operator-=(const nn::TimeSpan& timeSpan) NN_NOEXCEPT
{
    const auto posix = ToPosixTimeFromUtc(*this) - timeSpan;
    *this = ToCalendarTimeInUtc(posix);
    return *this;
}

CalendarTime operator+(const CalendarTime& calendarTime, const nn::TimeSpan& timeSpan) NN_NOEXCEPT
{
    const auto posix = ToPosixTimeFromUtc(calendarTime) + timeSpan;
    return ToCalendarTimeInUtc(posix);
}

CalendarTime operator-(const CalendarTime& calendarTime, const nn::TimeSpan& timeSpan) NN_NOEXCEPT
{
    const auto posix = ToPosixTimeFromUtc(calendarTime) - timeSpan;
    return ToCalendarTimeInUtc(posix);
}

nn::TimeSpan operator-(const CalendarTime& lhs, const CalendarTime& rhs) NN_NOEXCEPT
{
    const auto posixLhs = ToPosixTimeFromUtc(lhs);
    const auto posixRhs = ToPosixTimeFromUtc(rhs);
    return posixLhs - posixRhs;
}

bool operator<=(const CalendarTime& lhs, const CalendarTime& rhs) NN_NOEXCEPT
{
    return !(rhs < lhs);
}

bool operator>=(const CalendarTime& lhs, const CalendarTime& rhs) NN_NOEXCEPT
{
    return !(lhs < rhs);
}

bool operator<(const CalendarTime& lhs, const CalendarTime& rhs) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(lhs.IsValid());
    NN_SDK_REQUIRES(rhs.IsValid());

    return ToUint64(lhs) < ToUint64(rhs);
}

bool operator>(const CalendarTime& lhs, const CalendarTime& rhs) NN_NOEXCEPT
{
    return rhs < lhs;
}

}}

