﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/time/time_ApiForRepair.h>
#include <nn/time/time_StandardNetworkSystemClock.h>

#include <nn/ntc/shim/ntc_Shim.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>

namespace nn { namespace time {

Result CalibrateSystemClockWithInternalOffset() NN_NOEXCEPT
{
    // 通常の自動自動補正を抑止
    nn::ntc::shim::SuspendAutonomicTimeCorrection();
    NN_UTIL_SCOPE_EXIT{ nn::ntc::shim::ResumeAutonomicTimeCorrection(); };

    // 元デバイスでネットワーク時計が補正済であれば、
    // データ移行後の本体の SteadyClockTimePoint を使ったデタラメなネットワーク時刻が取得される
    nn::time::PosixTime deviceNetworkTime;
    NN_RESULT_DO(nn::time::StandardNetworkSystemClock::GetCurrentTime(&deviceNetworkTime));

    // サーバー時刻の取得
    nn::ntc::shim::CorrectionNetworkClockAsyncTask task(
        nn::os::EventClearMode_AutoClear, nn::ntc::EnsureNetworkClockAvailabilityMode_GetServerTime);
    NN_RESULT_DO(task.StartTask());
    task.GetFinishNotificationEvent().Wait();
    NN_RESULT_DO(task.GetResult());

    // デタラメなネットワーク時計とサーバー時刻の差を、内部オフセットで埋めて補正する
    const auto diff = task.GetServerPosixTime() - deviceNetworkTime;
    auto internalOffset = nn::time::GetStandardSteadyClockInternalOffset();
    internalOffset += diff;
    nn::time::SetStandardSteadyClockInternalOffset(internalOffset);

    NN_RESULT_SUCCESS;
}

void SuspendAutonomicTimeCorrection() NN_NOEXCEPT
{
    nn::ntc::shim::SuspendAutonomicTimeCorrection();
}

void ResumeAutonomicTimeCorrection() NN_NOEXCEPT
{
    nn::ntc::shim::ResumeAutonomicTimeCorrection();
}

}} // nn::time

