﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/time/time_Api.h>
#include <nn/time/time_StandardSteadyClock.h>
#include <nn/time/time_AdjustableUserSystemClock.h>
#include <nn/time/time_AdjustableNetworkSystemClock.h>

#include <nn/time/detail/time_CommonDetail.h>
#include <nn/time/detail/time_SharedMemoryClient.h>
#include <nn/time/detail/util/time_UtilApi.h>
#include <nn/time/detail/service/time_ServiceProviderClient.h>

#include <nn/timesrv/detail/service/timesrv_ISystemClock.sfdl.h>
#include <nn/timesrv/detail/service/timesrv_ISteadyClock.sfdl.h>
#include <nn/timesrv/detail/service/timesrv_ITimeZoneService.sfdl.h>
#include <nn/timesrv/detail/service/timesrv_IStaticService.sfdl.h>

#include <nn/os/os_SdkMutex.h>
#include <nn/util/util_Optional.h>

namespace nn
{
namespace time
{

namespace
{
    uint32_t g_InitializeCount = 0;
    nn::os::SdkMutex g_MutexInitialize;

    nn::util::optional<nn::time::detail::service::ServiceProviderClient> g_pServiceProviderClient;
}

nn::sf::SharedPointer<nn::timesrv::detail::service::IStaticService> g_pStaticService = nullptr;
//nn::sf::SharedPointer<nn::timesrv::detail::service::ISystemClock> g_pStandardUserSystemClock = nullptr; // SIGLO-78154 の共有メモリ利用によって不要になった
nn::sf::SharedPointer<nn::timesrv::detail::service::ISystemClock> g_pStandardNetworkSystemClock = nullptr; // プライベート利用なので、APIが呼ばれたときに初めて初期化する
nn::sf::SharedPointer<nn::timesrv::detail::service::ISystemClock> g_pEphemeralNetworkSystemClock = nullptr; // プライベート利用なので、APIが呼ばれたときに初めて初期化する
nn::sf::SharedPointer<nn::timesrv::detail::service::ISteadyClock> g_pStandardSteadyClock = nullptr; // プライベート利用なので、APIが呼ばれたときに初めて初期化する
nn::sf::SharedPointer<nn::timesrv::detail::service::ITimeZoneService> g_pTimeZoneService = nullptr;

nn::util::optional<nn::time::detail::SharedMemoryClient> g_SharedMemoryClient;

namespace
{
    enum InitializeMode
    {
        InitializeMode_None,
        InitializeMode_Normal,  //!< 一般権限の初期化
        InitializeMode_Menu,    //!< メニュー向け権限の初期化
        InitializeMode_System,  //!< システム向け権限の初期化
        InitializeMode_Repair,  //!< 修理向け権限の初期化
    };

    InitializeMode g_FirstInitializeMode = InitializeMode_None;

    void EnsureAdjustableSystemClock() NN_NOEXCEPT
    {
        NN_FUNCTION_LOCAL_STATIC(bool, s_IsAdjustedOnce, = false);

        if(!s_IsAdjustedOnce)
        {
            AdjustableUserSystemClock::Adjust();
            AdjustableNetworkSystemClock::Adjust();
            s_IsAdjustedOnce = true;
        }
    }

    /**
     * @brief 時刻ライブラリ初期化処理内部実装
     * @param[in] mode
     * @details
     *  スレッドセーフです。
     *  Windows 実行環境だと持っている権限関係なく全モードで成功します。
     */
    Result InitializeImpl(InitializeMode mode) NN_NOEXCEPT
    {
        NN_UTIL_LOCK_GUARD(g_MutexInitialize);

        if( g_InitializeCount > 0 )
        {
            // time:u で初期化後に time:a で初期化した際などに成功を返すと、
            // time:a の IPC が使えないのに成功が返るのでダメ。
            // なので最初の初期化と同じモードでの初期化を事前条件とする。
            //
            // 最初と異なる権限で初期化しようとした場合に IPC を開きなおしても良いが、
            // そもそも異なる権限を同時に持つアプリケーションは存在しないはずなので対応しない.
            //
            // windows 開発環境では複数アプリが1バイナリになる可能性があり、
            // 意図せず異なるモードで初期化する可能性があるのでチェックしない。
            // そもそも windows 開発環境では権限チェックがないので特権APIを使うにはこれで問題なし。
#if !defined(NN_BUILD_CONFIG_OS_WIN)
            NN_ABORT_UNLESS(mode == g_FirstInitializeMode,
                "The time library is already initialized with another mode.");
#endif
            ++g_InitializeCount;
            NN_RESULT_SUCCESS;
        }

        if( !g_pServiceProviderClient )
        {
            g_pServiceProviderClient.emplace();
        }

        nn::sf::SharedPointer<nn::timesrv::detail::service::IStaticService> pStaticService = nullptr;
        if(mode == InitializeMode_Normal)
        {
            NN_RESULT_DO(g_pServiceProviderClient->GetUserStaticServiceSharedPointer(&pStaticService));
        }
        else if(mode == InitializeMode_Menu)
        {
            NN_RESULT_DO(g_pServiceProviderClient->GetAdminStaticServiceSharedPointer(&pStaticService));
        }
        else if(mode == InitializeMode_System)
        {
            NN_RESULT_DO(g_pServiceProviderClient->GetSystemStaticServiceSharedPointer(&pStaticService));
        }
        else if (mode == InitializeMode_Repair)
        {
            NN_RESULT_DO(g_pServiceProviderClient->GetRepairStaticServiceSharedPointer(&pStaticService));
        }
        else
        {
            NN_ABORT("[TIME]Unknown initialize mode.");
        }
        NN_SDK_ASSERT_NOT_NULL(pStaticService);

        // ここまで来た時点で g_pServiceProviderClient の サブドメインは初期化済であり、
        // 各サービスオブジェクトの取得に失敗した場合はサブドメインの破棄も行う必要がある.
        NN_UTIL_SCOPE_EXIT
        {
            if(!g_pTimeZoneService)
            {
                g_pServiceProviderClient->Finalize(); // サブドメイン Finalize
            }
        };

        // タイムゾーンAPI用サービスオブジェクトを確保
        // 他のサービスはPrivateAPIやシステム向けAPIでのみ利用するため、遅延初期化する
        nn::sf::SharedPointer<nn::timesrv::detail::service::ITimeZoneService> pTimeZoneService = nullptr;
        NN_RESULT_DO( pStaticService->GetTimeZoneService(&pTimeZoneService) );
        NN_SDK_ASSERT_NOT_NULL(pTimeZoneService);

        // ここまで来たら成功
        g_pStaticService = std::move( pStaticService );
        g_pTimeZoneService = std::move( pTimeZoneService );

        g_SharedMemoryClient.emplace(g_pStaticService);

        EnsureAdjustableSystemClock();

        ++g_InitializeCount;
        g_FirstInitializeMode = mode;

        NN_RESULT_SUCCESS;
    }
}

Result Initialize() NN_NOEXCEPT
{
    return InitializeImpl(InitializeMode_Normal);
}

Result InitializeForMenu() NN_NOEXCEPT
{
    return InitializeImpl(InitializeMode_Menu);
}

Result InitializeForSystem() NN_NOEXCEPT
{
    return InitializeImpl(InitializeMode_System);
}

Result InitializeForRepair() NN_NOEXCEPT
{
    return InitializeImpl(InitializeMode_Repair);
}

bool IsInitialized() NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(g_MutexInitialize);

    return g_InitializeCount > 0;
}

nn::Result Finalize() NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(g_MutexInitialize);

    if ( g_InitializeCount > 0 )
    {
        --g_InitializeCount;

        // Initialize のカウントが残っていなければセッションを解放
        if ( g_InitializeCount == 0 )
        {
            g_pStaticService = nullptr;
            g_pStandardNetworkSystemClock = nullptr;
            g_pStandardSteadyClock = nullptr;
            g_pTimeZoneService = nullptr;
            g_pEphemeralNetworkSystemClock = nullptr;
            g_SharedMemoryClient = nn::util::nullopt;

            // サービスオブジェクトを払いだしているサブドメインは
            // 上記のサービスオブジェクトの後に破棄
            g_pServiceProviderClient->Finalize(); // サブドメイン Finalize

            g_FirstInitializeMode = InitializeMode_None;
        }
    }

    NN_RESULT_SUCCESS;
}

Result GetEphemeralNetworkSystemClockService(
    nn::sf::SharedPointer<nn::timesrv::detail::service::ISystemClock>* pOut) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pOut);

    NN_UTIL_LOCK_GUARD(g_MutexInitialize);
    NN_RESULT_THROW_UNLESS(g_pStaticService != nullptr, ResultNotInitialized());

    if (!g_pEphemeralNetworkSystemClock)
    {
        NN_RESULT_DO(g_pStaticService->GetEphemeralNetworkSystemClock(&g_pEphemeralNetworkSystemClock));
    }

    *pOut = g_pEphemeralNetworkSystemClock;
    NN_RESULT_SUCCESS;
}

Result GetStandardNetworkSystemClockService(
    nn::sf::SharedPointer<nn::timesrv::detail::service::ISystemClock>* pOut) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pOut);

    NN_UTIL_LOCK_GUARD(g_MutexInitialize);
    NN_RESULT_THROW_UNLESS(g_pStaticService != nullptr, ResultNotInitialized());

    if (!g_pStandardNetworkSystemClock)
    {
        NN_RESULT_DO(g_pStaticService->GetStandardNetworkSystemClock(&g_pStandardNetworkSystemClock));
    }

    *pOut = g_pStandardNetworkSystemClock;
    NN_RESULT_SUCCESS;
}

Result GetStandardSteadyClockService(
    nn::sf::SharedPointer<nn::timesrv::detail::service::ISteadyClock>* pOut) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pOut);

    NN_UTIL_LOCK_GUARD(g_MutexInitialize);
    NN_RESULT_THROW_UNLESS(g_pStaticService != nullptr, ResultNotInitialized());

    if (!g_pStandardSteadyClock)
    {
        NN_RESULT_DO(g_pStaticService->GetStandardSteadyClock(&g_pStandardSteadyClock));
    }

    *pOut = g_pStandardSteadyClock;
    NN_RESULT_SUCCESS;
}

Result GetSpanBetween(int64_t* pOutSeconds, const SteadyClockTimePoint& from, const SteadyClockTimePoint& to) NN_NOEXCEPT
{
    return detail::util::GetSpanBetween(pOutSeconds, from, to);
}

Result GetSpanFromCorrection(int64_t* pOutSeconds, const SystemClockContext& systemClockContext) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutSeconds);
    NN_RESULT_THROW_UNLESS(pOutSeconds != nullptr, ResultInvalidPointer());

    nn::time::SteadyClockTimePoint timePoint;
    NN_RESULT_DO(nn::time::StandardSteadyClock::GetCurrentTimePoint(&timePoint));

    int64_t outSeconds;
    NN_RESULT_DO(GetSpanBetween(&outSeconds, systemClockContext.timeStamp, timePoint));

    *pOutSeconds = outSeconds;
    NN_RESULT_SUCCESS;
}

bool IsLeapYear(int year) NN_NOEXCEPT
{
    return detail::util::IsLeapYear(year);
}

bool IsValidDate(int year, int month, int day) NN_NOEXCEPT
{
    return detail::util::IsValidDate(year, month, day);
}

int GetDaysInMonth(int year, int month) NN_NOEXCEPT
{
    return detail::util::GetDaysInMonth(year, month);
}

int DateToDays(int inYear, int inMonth, int inDay) NN_NOEXCEPT
{
    return detail::util::DateToDays(inYear, inMonth, inDay);
}

void DaysToDate(int *pOutYear, int *pOutMonth, int *pOutDay, int inDays) NN_NOEXCEPT
{
    detail::util::DaysToDate(pOutYear, pOutMonth, pOutDay, inDays);
}

DayOfWeek GetDayOfWeek(int year, int month, int day) NN_NOEXCEPT
{
    return detail::util::GetDayOfWeek(year, month, day);
}

}
}

