﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/time/time_Api.h>
#include <nn/time/time_ApiForClockSnapshot.h>
#include <nn/time/detail/time_ClaimPeriodicBenefit.h>
#include <nn/time/detail/time_ClockSnapshotPrivateApi.h>
#include <nn/time/detail/time_CommonDetail.h>
#include <nn/time/detail/time_SystemClockPrivateApi.h>
#include <nn/time/detail/time_TimeZonePrivateApi.h>
#include <nn/timesrv/detail/service/timesrv_IStaticService.sfdl.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/util/util_Optional.h>

#if 0
#define NN_DETAIL_TIME_TRACE_CLAIM_BENEFIT(...) NN_DETAIL_TIME_TRACE("[ClaimPeriodicBenefitWithUserSystemClock] " __VA_ARGS__)
#else
#define NN_DETAIL_TIME_TRACE_CLAIM_BENEFIT(...) ((void)0)
#endif

namespace nn { namespace time { namespace detail {

namespace
{
    // ClockSnapshotの初期化タイプが ClockSnapshotInitialType::Constructor かどうか
    bool IsConstructorInitialState(const nn::time::ClockSnapshot& snapshot) NN_NOEXCEPT
    {
        const auto* p = nn::time::detail::GetSfClockSnapshotPtr(&snapshot);

        return
            static_cast<uint8_t>(nn::time::detail::ClockSnapshotInitialType::Constructor) == p->initialType;
    }

    // SteadyClockTimePoint 同士の連続性があるかどうか(比較可能か)
    bool IsContinuous(const nn::time::SteadyClockTimePoint& lhs, const nn::time::SteadyClockTimePoint& rhs) NN_NOEXCEPT
    {
        return lhs.sourceId == rhs.sourceId;
    }

    // 自動補正ONかつネットワーク時計が補正済かどうか
    bool IsAutoCorrectionEnabledAndNetworkClockCorrected(const ClockSnapshot& snapshot) NN_NOEXCEPT
    {
        const auto* pSfSnapshot = reinterpret_cast<const nn::time::sf::ClockSnapshot*>(&snapshot);
        return true
            && pSfSnapshot->isAutomaticCorrectionEnabled
            && pSfSnapshot->netSystemClockContext.timeStamp.sourceId == pSfSnapshot->steadyClockTimePoint.sourceId
        ;
    }

    // より未来を指す SteadyClockTimePoint を取得
    const nn::time::SteadyClockTimePoint& GetMoreFuture(
        const nn::time::SteadyClockTimePoint& lhs, const nn::time::SteadyClockTimePoint& rhs) NN_NOEXCEPT
    {
        NN_SDK_ASSERT(IsContinuous(lhs, rhs));
        return lhs.value > rhs.value ? lhs : rhs;
    }

    // ペナルティ情報の初期化
    void ResetPenaltyEndInfo(PeriodicBenefitClaimContext* pOut) NN_NOEXCEPT
    {
        pOut->_penaltyEndSteadyClockTimePoint.value = 0;
        pOut->_penaltyEndSteadyClockTimePoint.sourceId = nn::util::InvalidUuid;
    }

    // ペナルティ開始の時点を取得
    SteadyClockTimePoint GetPenaltyStart(const ClockSnapshot& last, const ClockSnapshot& current) NN_NOEXCEPT
    {
        // 以下のうちもっとも未来の(直近の)タイミングかつ、現在のSteadyClockTimePointと比較できる時点をペナルティ開始タイミングとする
        // - 時計の直接操作(自動補正ONかつネットワーク時計が補正済の場合を除く)
        // - ユーザー時計自動補正フラグ変更(自動補正フラグ変更時のみ)
        // - タイムゾーン変更(タイムゾーン変更時のみ)

        nn::util::optional<SteadyClockTimePoint> ret;

        // 自動補正ONかつネットワーク時計が補正済だと、 GetStandardUserSystemClockContext().timeStamp がネットワーク時計の補正タイミングなので不採用
        if(!IsAutoCorrectionEnabledAndNetworkClockCorrected(current))
        {
            NN_DETAIL_TIME_TRACE_CLAIM_BENEFIT("[GetPenaltyStart] user system clock timestamp %lld\n", current.GetStandardUserSystemClockContext().timeStamp);
            ret = current.GetStandardUserSystemClockContext().timeStamp;
        }

        if( nn::time::detail::IsClockSnapshotAutomaticCorrectionEnabled(last) !=
            nn::time::detail::IsClockSnapshotAutomaticCorrectionEnabled(current))
        {
            // 自動補正フラグ変更されている

            nn::time::SteadyClockTimePoint flagUpdatedTime;
            nn::time::detail::GetStandardUserSystemClockAutomaticCorrectionUpdatedTime(&flagUpdatedTime);

            NN_DETAIL_TIME_TRACE_CLAIM_BENEFIT("[GetPenaltyStart] automatic correction flag changed. %lld\n", flagUpdatedTime.value);

            if(IsContinuous(flagUpdatedTime, current.GetStandardSteadyClockTimePoint())) // 現在のSteadyClockTimePointと比較できるものだけ採用する
            {
                if(static_cast<bool>(ret))
                {
                    ret = GetMoreFuture(*ret, flagUpdatedTime); // より未来を採用
                }
                else
                {
                    ret = flagUpdatedTime;
                }
            }
        }

        if(last.GetLocationName() != current.GetLocationName())
        {
            // タイムゾーン変更されている

            nn::time::SteadyClockTimePoint locationUpdatedTime;
            nn::time::LocationName locationName;

            nn::time::detail::GetDeviceLocationNameAndUpdatedTime(&locationName, &locationUpdatedTime);
            NN_UNUSED(locationName);

            NN_DETAIL_TIME_TRACE_CLAIM_BENEFIT("[GetPenaltyStart] timezone changed. %lld\n", locationUpdatedTime.value);

            if(IsContinuous(locationUpdatedTime, current.GetStandardSteadyClockTimePoint())) // 現在のSteadyClockTimePointと比較できるものだけ採用する
            {
                if(static_cast<bool>(ret))
                {
                    ret = GetMoreFuture(*ret, locationUpdatedTime); // より未来を採用
                }
                else
                {
                    ret = locationUpdatedTime;
                }
            }
        }

        // 上記のいずれかで値が入っている想定
        NN_SDK_ASSERT(static_cast<bool>(ret));

        // フェイルセーフとして、値がない場合は GetStandardUserSystemClockContext().timeStamp を入れておく.
        if(!static_cast<bool>(ret))
        {
            ret = current.GetStandardUserSystemClockContext().timeStamp;
        }

        return *ret;
    }

    // 最後に利益を得た瞬間から開始されるペナルティが終わる SteadyClockTimePoint を取得
    // ただし、最後に利益を得た瞬間から
    SteadyClockTimePoint GetPenaltyEndFromLastBenefitReceived(
        const ClockSnapshot& lastReceivedSnapshot, const ClockSnapshot& currentSnapshot, nn::TimeSpan penaltySpan) NN_NOEXCEPT
    {
        nn::TimeSpan span;
        if(!nn::time::CalculateSpanBetween(&span, lastReceivedSnapshot, currentSnapshot) || span < nn::TimeSpan(0))
        {
            // 最後に利益を得た瞬間からの経過時間がわからない場合は、現在からペナルティ開始
            NN_DETAIL_TIME_TRACE_CLAIM_BENEFIT("[GetPenaltyEndFromLastBenefitReceived] Penalty from current.\n");
            return currentSnapshot.GetStandardSteadyClockTimePoint() + penaltySpan;
        }

        NN_DETAIL_TIME_TRACE_CLAIM_BENEFIT("[GetPenaltyEndFromLastBenefitReceived] Penalty from last benefit received.\n");

        // 最後に利益を得た瞬間からの経過時間分、ペナルティが終わっている
        // 現在より過去になることがあるが、そのときはペナルティが過ぎ去ってるとして扱われるのでOK
        nn::TimeSpan remainPenalty = penaltySpan - span;

        // 以降も確実に比較可能な currentSnapshot.GetStandardSteadyClockTimePoint() を使って計算している点に注意
        return currentSnapshot.GetStandardSteadyClockTimePoint() + remainPenalty;
    }

    // タイムゾーン変更による操作量を取得
    nn::TimeSpan CalculateTimeZoneChangeDifference(
        const nn::time::ClockSnapshot& from,
        const nn::time::ClockSnapshot& to) NN_NOEXCEPT
    {
        // LocationName 変更はしていないが夏時間などの影響による時差変化は、ユーザーが操作したとは言えない.
        // ので LocationNameが同じであれば、utcOffsetSeconds が違っていたとしても操作したことにはしない.
        if(from.GetLocationName() == to.GetLocationName())
        {
            return nn::TimeSpan(0);
        }

        auto diffSeconds =
            to.GetStandardUserSystemClockCalendarAdditionalInfo().timeZone.utcOffsetSeconds -
            from.GetStandardUserSystemClockCalendarAdditionalInfo().timeZone.utcOffsetSeconds;
        return nn::TimeSpan::FromSeconds(diffSeconds);
    }

    // 時計操作量を計算して true を返す
    // 計算できない場合(RTCリセット、もしくはデバイス引っ越しなど)、false を返す
    bool CalculateDifferenceByUser(nn::TimeSpan* pOutDifferenceByUser, const ClockSnapshot& previous, const ClockSnapshot& current) NN_NOEXCEPT
    {
        // 単調増加クロックの連続性がないため時計操作量の判定不可
        if(previous.GetStandardSteadyClockTimePoint().sourceId != current.GetStandardSteadyClockTimePoint().sourceId)
        {
            NN_DETAIL_TIME_TRACE_CLAIM_BENEFIT("[CalculateDifferenceByUser] SourceId missmatch!\n");
            return false;
        }

        nn::TimeSpanType difference
            = CalculateStandardUserSystemClockDifferenceByUser(previous, current) // 時計のコンテキストからの操作量
            + CalculateTimeZoneChangeDifference(previous, current) // タイムゾーン変更による時間変化
        ;

        *pOutDifferenceByUser = difference;
        NN_DETAIL_TIME_TRACE_CLAIM_BENEFIT("[CalculateDifferenceByUser] Difference by user:%lld sec\n", difference.GetSeconds());
        return true;
    }

    // 時計操作量からペナルティ期間を計算(ペナルティ開始がいつかは関与しない)
    nn::TimeSpan CalculatePenaltySpan(
        nn::TimeSpan differenceByUser, nn::TimeSpan maxPenaltyTimeSpan, nn::TimeSpan acceptableOperationTimeSpan) NN_NOEXCEPT
    {
        if(differenceByUser == nn::TimeSpan(0))
        {
            return nn::TimeSpan(0);
        }

        if(differenceByUser > nn::TimeSpan(0))
        {
            // 時計が未来へ進む方向へ操作された

            // acceptableOperationTimeSpan 以内の操作量は無視
            // acceptableOperationTimeSpan より大きければ MaxPenaltyTimeSpan
            if(differenceByUser <= acceptableOperationTimeSpan)
            {
                return nn::TimeSpan(0);
            }
            else
            {
                return maxPenaltyTimeSpan;
            }
        }
        else
        {
            // 時計が過去へ戻る方向へ操作された
            // 操作量がペナルティ期間(ただし最大で maxPenaltyTimeSpan )
            auto abs = nn::TimeSpan::FromSeconds(differenceByUser.GetSeconds() * -1ll);
            return std::min(abs, maxPenaltyTimeSpan);
        }
    }

    PeriodicBenefitClaimResult ReturnResult(
        PeriodicBenefitClaimContext* pHandover,
        const ClockSnapshot& currentSnapshot,
        const CheckCalendarTimeCallback* pCheckCalendarTimeCallback) NN_NOEXCEPT
    {
        if(HasPenaltyInfo(*pHandover))
        {
            int64_t seconds;
            auto result = nn::time::GetSpanBetween(&seconds, currentSnapshot.GetStandardSteadyClockTimePoint(), pHandover->_penaltyEndSteadyClockTimePoint);
            NN_DETAIL_TIME_ASSERT_IF_RESULT_FAILURE(result);
            if(result.IsSuccess() && seconds > 0)
            {
                // ペナルティ発生中

                NN_DETAIL_TIME_TRACE_CLAIM_BENEFIT(
                    "return PeriodicBenefitClaimResult_Penalty: Penalty end(SteadyClockTimePoint.value:%lld)\n",
                        pHandover->_penaltyEndSteadyClockTimePoint.value);

                // 以降の時計操作を正しく計測するため次回判定へ必ず引き継ぐ
                pHandover->_snapshot = currentSnapshot;

                return PeriodicBenefitClaimResult_Penalty;
            }

            ResetPenaltyEndInfo(pHandover);
        }

        bool isInitialCall = IsConstructorInitialState(pHandover->_snapshot);
        bool isMatchedTiming = pCheckCalendarTimeCallback->IsMatched(currentSnapshot.GetStandardUserSystemClockCalendarTime());

        if(!pHandover->_isMaxPenalatyOccurred && !isInitialCall)
        {
            // MaxPenaltyTimeSpan発生してないときにだけ、isSameTiming を確認する
            // (時計をN分戻してN分のペナルティ発生後、同じタイミングでは利益を得られなくする)
            bool isSameTiming = pCheckCalendarTimeCallback->IsSameTiming(
                    pHandover->_lastBenefitReceivedSnapshot.GetStandardUserSystemClockCalendarTime(),
                    currentSnapshot.GetStandardUserSystemClockCalendarTime());

            if(isSameTiming || !isMatchedTiming)
            {
                NN_DETAIL_TIME_TRACE_CLAIM_BENEFIT(
                    "return PeriodicBenefitClaimResult_NotReceivableTiming: isSameTiming(%d), isMatchedTiming(%d), isInitialCall(%d)\n",
                        isSameTiming, isMatchedTiming, isInitialCall);
                return PeriodicBenefitClaimResult_NotReceivableTiming;
            }
        }
        else if(!isMatchedTiming)
        {
            NN_DETAIL_TIME_TRACE_CLAIM_BENEFIT(
                "return PeriodicBenefitClaimResult_NotReceivableTiming: isSameTiming(ignore), isMatchedTiming(%d), isInitialCall(%d)\n",
                    isMatchedTiming, isInitialCall);
            return PeriodicBenefitClaimResult_NotReceivableTiming;
        }

        // 利益を受け取ることができる
        pHandover->_snapshot = currentSnapshot;                     // 操作検知ポイントの更新
        pHandover->_lastBenefitReceivedSnapshot = currentSnapshot;  // 利益受け取り時間の更新
        pHandover->_isMaxPenalatyOccurred = false;                  // 一度利益を受け取れば、次回の isSameTiming は必須

        NN_DETAIL_TIME_TRACE_CLAIM_BENEFIT("return PeriodicBenefitClaimResult_Success\n");
        return PeriodicBenefitClaimResult_Success;
    }
}

bool HasLastBenefitReceivedInfo(const PeriodicBenefitClaimContext& value) NN_NOEXCEPT
{
    return true
        && !IsConstructorInitialState(value._lastBenefitReceivedSnapshot)
    ;
}

bool HasPenaltyInfo(const PeriodicBenefitClaimContext& value) NN_NOEXCEPT
{
    return true
        // コンストラクタで初期化された状態だと、 ClaimPeriodicBenefitWithUserSystemClock() を叩いたことがない
        && !IsConstructorInitialState(value._snapshot)
        // InvalidUuidでさえなければ、現在と比較できないかもしれないけど情報は保持している
        && value._penaltyEndSteadyClockTimePoint.sourceId != nn::util::InvalidUuid
    ;
}

PeriodicBenefitClaimResult ClaimPeriodicBenefitWithUserSystemClock(
    PeriodicBenefitClaimContext* pHandover,
    const ClockSnapshot& currentSnapshot,
    const nn::TimeSpan& maxPenaltyTimeSpan,
    const nn::TimeSpan& acceptableOperationTimeSpan,
    const CheckCalendarTimeCallback* pCheckCalendarTimeCallback) NN_NOEXCEPT
{
    // TODO
    // 利用者へデバッグ用情報の提供
    // 操作量,ペナルティ量,ペナルティ開始の起点,ペナルティ終了の SteadyClockTimePoint

    if(IsConstructorInitialState(pHandover->_snapshot))
    {
        // コンストラクタで初期化されたままのケースは初回.時刻操作がないものとして扱う.

        NN_UTIL_SCOPE_EXIT
        {
            // 以降の時計操作を正しく計測するため次回判定へ必ず引き継ぐ
            pHandover->_snapshot = currentSnapshot;
        };

        // メンバ初期化
        {
            pHandover->_lastBenefitReceivedSnapshot = ClockSnapshot(); // 念のため
            ResetPenaltyEndInfo(pHandover);
            pHandover->_isMaxPenalatyOccurred = false;
            std::memset(&pHandover->_reserved, 0, sizeof(pHandover->_reserved));
        }

        return ReturnResult(pHandover, currentSnapshot, pCheckCalendarTimeCallback);
    }

    // 時計操作量の取得
    nn::TimeSpan differenceByUser;
    if(!CalculateDifferenceByUser(&differenceByUser, pHandover->_snapshot, currentSnapshot))
    {
        // 単調増加クロックの連続性がないので操作量を取得できない
        // 最後に利益を受けとったタイミングから MaxPenaltyTimeSpan 分のペナルティを開始
        // ただし、前後でネットワーク自動補正ON、ネットワーク時計補正済であれば操作はない

        NN_UTIL_SCOPE_EXIT
        {
            // 以降の時計操作を正しく計測するため次回判定へ必ず引き継ぐ
            pHandover->_snapshot = currentSnapshot;
        };

        if( IsAutoCorrectionEnabledAndNetworkClockCorrected(pHandover->_snapshot) &&
            IsAutoCorrectionEnabledAndNetworkClockCorrected(currentSnapshot))
        {
            NN_DETAIL_TIME_TRACE_CLAIM_BENEFIT("Auto correction enabled, and network clock corrected. then, no penalty.\n");

            // 単調増加クロックの連続性がない状態では _penaltyEndSteadyClockTimePoint からはペナルティ終了を判定できない。
            // のでもしペナルティがあっても強制的になかったことにする。
            // ペナルティが残っている状態で、自動補正ONかつ単調増加クロックの連続性がない、はレアすぎるので許容。
            ResetPenaltyEndInfo(pHandover);

            return ReturnResult(pHandover, currentSnapshot, pCheckCalendarTimeCallback);
        }

        if(HasLastBenefitReceivedInfo(*pHandover))
        {
            // 利益を受け取った瞬間から MaxPenaltyTimeSpan 分のペナルティ
            pHandover->_penaltyEndSteadyClockTimePoint =
                GetPenaltyEndFromLastBenefitReceived(pHandover->_lastBenefitReceivedSnapshot, currentSnapshot, maxPenaltyTimeSpan);

            // MaxPenaltyTimeSpan 発生時にだけ、次回利益受け取り時の isSameTiming チェックしなくてよい
            pHandover->_isMaxPenalatyOccurred = true;
        }
        else
        {
            NN_DETAIL_TIME_TRACE_CLAIM_BENEFIT("Has not last benefit received. then, not penalty.\n");
            // 利益を受けとったことがなければペナルティはなし
            ResetPenaltyEndInfo(pHandover);
        }

        return ReturnResult(pHandover, currentSnapshot, pCheckCalendarTimeCallback);
    }

    NN_UTIL_SCOPE_EXIT
    {
        // 今回分の操作があってペナルティが発生したら、次回判定時に新しい操作量を検知するため _snapshot を更新
        pHandover->_snapshot = currentSnapshot;
    };

    // 時計操作量からペナルティ期間を計算
    nn::TimeSpan penaltySpan = CalculatePenaltySpan(differenceByUser, maxPenaltyTimeSpan, acceptableOperationTimeSpan);

    if(penaltySpan == nn::TimeSpan(0))
    {
        // 今回の判定ではペナルティなし
        return ReturnResult(pHandover, currentSnapshot, pCheckCalendarTimeCallback);
    }

    // 以降、今回の判定でペナルティありだったケース。すでにペナルティ中であればより長いペナルティを採用。

    NN_SDK_ASSERT(penaltySpan > nn::TimeSpan(0));

    if(penaltySpan == maxPenaltyTimeSpan)
    {
        // MaxPenaltyTimeSpan 発生時にだけ、次回利益受け取り時の isSameTiming チェックしなくてよい
        pHandover->_isMaxPenalatyOccurred = true;
    }

    const auto penaltyStartSteadyClockTimePoint = GetPenaltyStart(pHandover->_snapshot, currentSnapshot);
    const auto penaltyEndSteadyClockTimePoint = penaltyStartSteadyClockTimePoint + penaltySpan;

    // より未来のペナルティ終了を採用
    if(IsContinuous(pHandover->_penaltyEndSteadyClockTimePoint, penaltyEndSteadyClockTimePoint))
    {
        pHandover->_penaltyEndSteadyClockTimePoint = GetMoreFuture(pHandover->_penaltyEndSteadyClockTimePoint, penaltyEndSteadyClockTimePoint);
    }
    else
    {
        pHandover->_penaltyEndSteadyClockTimePoint = penaltyEndSteadyClockTimePoint;
    }

    return ReturnResult(pHandover, currentSnapshot, pCheckCalendarTimeCallback);
} // NOLINT(impl/function_size)

}}} // nn::time::detail
